/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/sl/api/WDSystemLandscape.java#5 $
 */
package com.sap.tc.webdynpro.services.sal.sl.api;

import com.sap.lcr.api.sapmodel.SAP_J2EEEngineCluster;
import com.sap.tc.webdynpro.services.sal.core.AbstractManager;

/**
 * This class provides static access to <code>IJCOClientConnection</code> instances, 
 * and to get information about the SLD.   
 *
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDSystemLandscape {

  /**
   * Returns an istance of <code>IJCOClientConnection</code>. 
   *
   * @param name the name of the JCO client connection.
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
   * @return an instance of <code>IJCOClientConnection</code>
   * @throws JCOClientException if there is no such JCO client connection defined.
   */
  public static IWDJCOClientConnection getJCOClientConnection(String name, String j2eeClusterName) throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getJCOClientConnection(name, j2eeClusterName);
  }

  /**
   * Returns an instance of <code>IJCOClientConnection</code> which is assigned to the local J2EE cluster.
   * 
   * @param name the name of the JCO client connection.
   * @return an instance of <code>IJCOClientConnection</code>
   * @throws JCOClientException if there is no such JCO client connection defined.
   */
  public static IWDJCOClientConnection getJCOClientConnection(String name) throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getJCOClientConnection(name);
  }
 
  /**
   * Removes the specified <code>JCOClientConnection</code>.
   * 
   * @param name the name of the JCO client connection.
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
   * @throws WDSystemLandscapeException if the specified JCO client connection could not be removed
   */
  public static void removeJCOClientConnection(String name, String j2eeClusterName) throws WDSystemLandscapeException {
    AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().removeJCOClientConnection(name, j2eeClusterName);
  }

  /**
   * Creates a new instance of <code>IWDJCOClientConnection</code> defined by the message server name,
   * the system name and the logon group.
   * The used repository depends on the implementation.
   *
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param maxNumberOfConnections the maximal number of connections provided by the pool.
   * @param maxWaitTime the maximal wait time (i.e. the maximal time, waiting for a new <code>JCO.Client</code>)
   * @param connectionTimeOut defines the time after that a open and unused connection managed by the pool is closed.
   * @param msgServerName the name of the message server
   * @param systemName the name of the R/3 system
   * @param logonGroup the used logon group
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is use
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
   * @return the new JCOClient connection
   * @throws WDSystemLandscapeException if the creation of the new JCO client connection failed.
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    int maxNumberOfConnections,
    int maxWaitTime,
    int connectionTimeOut,
    String msgServerName,
    String systemName,
    String logonGroup,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String sncMode,
    String sncSecurityLevel,
    String sncPartnerName,
    String sncName,
    String sncLibraryPath,
    String language,
    String j2eeClusterName)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      maxNumberOfConnections,
      maxWaitTime,
      connectionTimeOut,
      msgServerName,
      systemName,
      logonGroup,
      client,
      userDefinition,
      user,
      password,
      sncMode,
      sncSecurityLevel,
      sncPartnerName,
      sncName,
      sncLibraryPath,
      language,
      j2eeClusterName);
  }

  /**
   * Creates a new instance of <code>IJCOClientConnection</code> defined by the message server name,
   * the system name and the logon group.
   * The used repository depends on the implementation.
   * 
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size
   * @param msgServerName the name of the message server
   * @param systemName the name of the R/3 system
   * @param logonGroup the used logon group
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   *                          'useSNC': the user is defined using SNC
   * @param user the name of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
  
   * @return the new JCOClient connection
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    String msgServerName,
    String systemName,
    String logonGroup,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String sncMode,
    String sncSecurityLevel,
    String sncPartnerName,
    String sncName,
    String sncLibraryPath,
    String language,
    String j2eeClusterName)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      msgServerName,
      systemName,
      logonGroup,
      client,
      userDefinition,
      user,
      password,
      sncMode,
      sncSecurityLevel,
      sncPartnerName,
      sncName,
      sncLibraryPath,
      language,
      j2eeClusterName);
  }

  /**
   * Creates a new instance of <code>IJCOClientConnection</code> defined by the message server name,
   * the system name and the logon group assigned to the local J2EE engine.
   * The used repository depends on the implementation.
   * 
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size
   * @param msgServerName the name of the message server
   * @param systemName the name of the R/3 system
   * @param logonGroup the used logon group
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is use
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param language the language of the user
   * @return the new JCOClient connection
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    String msgServerName,
    String systemName,
    String logonGroup,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String language)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      msgServerName,
      systemName,
      logonGroup,
      client,
      userDefinition,
      user,
      password,
      "0",
      "0",
      null,
      null,
      null,
      language,
      null);
  }

  /**
   * Creates a new instance of <code>IWDJCOClientConnection</code> defined by application server name
   * and system number.
   * The used repository depends on the implementation.
   *
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param maxNumberOfConnections the maximal number of connections provided by the pool.
   * @param maxWaitTime the maximal wait time (i.e. the maximal time, waiting for a new <code>JCO.Client</code>)
   * @param connectionTimeOut defines the time after that a open and unused connection managed by the pool is closed.
   * @param appServerName the name of the application server
   * @param systemNumber the system number name of the R/3 system
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
   * @return the new JCOClient connection
   * @throws WDSystemLandscapeException if the creation of the new JCO client connection failed.
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    int maxNumberOfConnections,
    int maxWaitTime,
    int connectionTimeOut,
    String appServerName,
    String systemNumber,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String sncMode,
    String sncSecurityLevel,
    String sncPartnerName,
    String sncName,
    String sncLibraryPath,
    String language,
    String j2eeClusterName)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      maxNumberOfConnections,
      maxWaitTime,
      connectionTimeOut,
      appServerName,
      systemNumber,
      client,
      userDefinition,
      user,
      password,
      sncMode,
      sncSecurityLevel,
      sncPartnerName,
      sncName,
      sncLibraryPath,
      language,
      j2eeClusterName);
  }

  /**
   * Creates a new instance of <code>IJCOClientConnection</code> defined by application server name 
   * and system number.
   * The used repository depends on the implementation.
   * 
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size
   * @param appServerName the name of the application server
   * @param systemNumber the system number name of the R/3 system
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   *                          'useSNC': the user is defined using SNC
   * @param user the name of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigend
   * @return the new JCOClient connection
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    String appServerName,
    String systemNumber,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String sncMode,
    String sncSecurityLevel,
    String sncPartnerName,
    String sncName,
    String sncLibraryPath,
    String language,
    String j2eeClusterName)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      appServerName,
      systemNumber,
      client,
      userDefinition,
      user,
      password,
      sncMode,
      sncSecurityLevel,
      sncPartnerName,
      sncName,
      sncLibraryPath,
      language,
      j2eeClusterName);
  }

  /**
   * Creates a new instance of <code>IWDJCOClientConnection</code> defined by application server name 
   * and system number assigned to the local J2EE cluster.
   * The used repository depends on the implementation.
   * 
   * @param name the name of the JCO client connection
   * @param maxPoolSize the maximal pool size
   * @param appServerName the name of the application server
   * @param systemNumber the system number name of the R/3 system
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER
   * @param language the language of the user
   * @return the new JCOClient connection
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public static IWDJCOClientConnection createJCOClientConnection(
    String name,
    int maxPoolSize,
    String appServerName,
    String systemNumber,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String language)
    throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(
      name,
      maxPoolSize,
      appServerName,
      systemNumber,
      client,
      userDefinition,
      user,
      password,
      "0",
      "0",
      null,
      null,
      null,
      language,
      null);
  }

  /**
   * Creates a new instance of <code>IWDJCOClientConnection</code> as a copy of the specified 
   * <code> IWDJCOOClientConnection> instance.
   * 
   * @param name the name of the new JCO client connection
   * @param source the <code>IWDJCOClientConnection</code> which should be copied
   * @return the new (copied) JCOClient connection
   * @throws WDSystemLandscapeException if the creation of the new JCO client connection failed.
   */
  public IWDJCOClientConnection createJCOClientConnection(String name, IWDJCOClientConnection source) throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().createJCOClientConnection(name, source);
  }

  /**
   * Returns the list of the defined JCO client connections associated to the specified J2EE engine cluster.
   * 
   * @param clusterName the name of the J2EE engine cluster to which the JCO client connections should be associated.
   * @return an array containing all defined JCO client connections or NULL if there are no.
   * @throws WDSystemLandscapeException if the creation of the new JCO client connection failed.  
   */
  public static String[] getJCOClientConnectionNames(String clusterName) throws WDSystemLandscapeException {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getJCOClientConnectionNames(clusterName);
  }

  /**
   * Returns the status of requested JCO client connection.
   * 
   * @param name the name of the JCO client connection
   * @return the status of the requested JCO client connection
   *    Possible values are:
   *      STATUS_OK : Everything is fine
   *      STATUS_NOT_DEFINED: The JCO client connection is not defined.
   *      STATUS_INVALID_DEFINITION: The JCO client connection is defined but there are some parameters missed.
   *      STATUS_UNKNOWN: THe status of the JCO client connection could not be checked. 
   */
  public static WDJCOClientConnectionStatus checkStatus(String name) {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().checkStatus(name, true);
  }

  /**
   * Returns the status of requested JCO client connection.
   * 
   * @param name the name of the JCO client connection
   * @param resetCache TRUE if the check should reset the cached connection information. FALSE otherwise.
   * @return the status of the requested JCO client connection
   *    Possible values are:
   *      STATUS_OK : Everything is fine
   *      STATUS_NOT_DEFINED: The JCO client connection is not defined.
   *      STATUS_INVALID_DEFINITION: The JCO client connection is defined but there are some parameters missed.
   *      STATUS_UNKNOWN: The status of the JCO client connection could not be checked. 
   */
  public static WDJCOClientConnectionStatus checkStatus(String name, boolean resetCache) {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().checkStatus(name, resetCache);
  }

  /**
   * Return the instance of <code>SAP_J2EEEngineCluster</code>, which represents the current J2EE cluster.
   * 
   * @return the SLD node for the current SAP J2EE cluster
   * @deprecated To keep the Web Dynpro Services API independant from the 
   *   SLD API, support for this method has been discarded. When running in 
   *   the SAP J2EE Engine, you can use the API of the engine's SLD service 
   *   to retrieve the SAP_J2EEEngineCluster. 
   *   This method may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static SAP_J2EEEngineCluster getCurrentJ2EECluster() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getCurrentJ2EECluster();
  }
  
  /**
   * Return the name of the current J2EE cluster instance.
   * 
   * @return the name of the current J2EE cluster. 
   */
  public static String getCurrentJ2EEClusterName() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getCurrentJ2EEClusterName();
  }

  /** 
   * Sets the SLD conncetion information.
   * 
   * @deprecated Beginning with NetWeaver 04, connection information for the System Landscape Directory
   *   (SLD) is maintained by the J2EE service <code>sld</code>. Support for setting this information
   *   from within Web Dynpro therefore has been discarded, use the Visual Admin tool instead.
   *   This method may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static void setSLDConnection(String host, String port, String user, String password) {
    AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().setSLDConnection(host, port, user, password);
  }

  /**
   * Return the host name of the used system landscape directory (SLD).
   * 
   * @return the host name of the used system landscape directory (SLD).
   */
  public static String getSLDHost() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getSLDHost();
  }

  /**
   * Return the port name  of the used system landscape directory (SLD).
   * 
   * @return the port name of the used system landscape directory (SLD).
   */
  public static int getSLDPort() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getSLDPort();
  }

  /**
   * Return the user name of the used system landscape directory (SLD).
   * 
   * @return the user name of the used system landscape directory (SLD).
   */
  public static String getSLDUser() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getSLDUser();
  }

  /**
   * Return the URL to start the assigned SLD.
   * 
   * @return the URL to start the assigned SLD.
   */
  public static String getSLDStartURL() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getSLDStartURL();
  }

  /**
   * Return FALSE if the configured SLD connection could not be pinged. TRUE otherwise.
   * 
   * @return FALSE if the configured SLD connection could not be pinged. TRUE otherwise. 
   */
  public static boolean pingSLDConnection() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().pingSLDConnection();
  }

  /**
   * Invaldiate the internal cache.
   */
  public static void invalidateCache() {
    AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().invalidateCache();
  }

  /**
   * Returns the version of the current J2EE Engine cluster. 
   * @return version of the current J2EE Engine cluster
   */    
  public static String getJ2EEVersionInfo() {
    return AbstractManager.getServerPlatformImplementation().getSystemLandscapeFactory().getJ2EEVersionInfo();
  }
}
