/*
 * SAP Copyright (c) 2002
 * All rights reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/WDDeployableObjectPartType.java#2 $
 */
 
package com.sap.tc.webdynpro.services.sal.deployment.api;

import java.util.Enumeration;
import java.util.Vector;
import com.sap.tc.webdynpro.services.exceptions.WDRuntimeException;
import com.sap.tc.webdynpro.services.sal.core.ServerConstants;

/**
 * Constants identifying deployable object part types. 
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public final class WDDeployableObjectPartType {

  private static final Vector deployableObjectPartTypes = new Vector();

  /** Number of deployable object part types. */  
  private static int NUMBER_OF_PARTS = 0;  

  /** Constant identifying Web Dynpro application. */
  public static final WDDeployableObjectPartType APPLICATION = 
    new WDDeployableObjectPartType("Application", "Applications", NUMBER_OF_PARTS++, "xml"); 

  /** Constant identifying Web Dynpro application. */
  public static final WDDeployableObjectPartType COMPONENT = 
    new WDDeployableObjectPartType("Component", "Components", NUMBER_OF_PARTS++, "xml"); 

  /** 
   * Constant identifying Web Dynpro application.      
   * @deprecated Use constant CONFIGURATION for configuration instead. 
   *   This constant may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static final WDDeployableObjectPartType COMPONENT_CONFIGURATION = 
    new WDDeployableObjectPartType("ComponentConfiguration", "ComponentConfigurations", NUMBER_OF_PARTS++, "rrconfiguration");  

  /** Constant identifying Web Dynpro application. */
  public static final WDDeployableObjectPartType COMPONENT_INTERFACE = 
    new WDDeployableObjectPartType("ComponentInterface", "ComponentInterfaces", NUMBER_OF_PARTS++, "xml");
 
  /** Constant identifying a Web resource. */
  public static final WDDeployableObjectPartType WEB_RESOURCE = 
    new WDDeployableObjectPartType("WebResource", ServerConstants.SYSTEM_RELATIVE_GLOBAL_PATH, NUMBER_OF_PARTS++, "");
  
  /** Constant identifying a model part. */
  public static final WDDeployableObjectPartType MODEL = 
    new WDDeployableObjectPartType("Model", "", NUMBER_OF_PARTS++, "");  
   
  /** Constant identifying variants. */
  public static final WDDeployableObjectPartType VARIANT = 
    new WDDeployableObjectPartType("Variant", "Variants", NUMBER_OF_PARTS++, "xml");  
  
  /** Constant identifying layers. */
  public static final WDDeployableObjectPartType CUSTOMIZING_LAYER = 
    new WDDeployableObjectPartType("CustomizingLayer", "Layers", NUMBER_OF_PARTS++, "xml");  
  
  /** Constant identifying a configuration part. */
  public static final WDDeployableObjectPartType CONFIGURATION = 
    new WDDeployableObjectPartType("Configuration", "Configs", NUMBER_OF_PARTS++, "xml");  
 
  static {
    deployableObjectPartTypes.add(APPLICATION); 
    deployableObjectPartTypes.add(COMPONENT);
    deployableObjectPartTypes.add(COMPONENT_CONFIGURATION);
    deployableObjectPartTypes.add(COMPONENT_INTERFACE);
    deployableObjectPartTypes.add(WEB_RESOURCE);
    deployableObjectPartTypes.add(MODEL);
    deployableObjectPartTypes.add(VARIANT);
    deployableObjectPartTypes.add(CUSTOMIZING_LAYER);
    deployableObjectPartTypes.add(CONFIGURATION);
  }

  private String typeName;
  private String folderName;
  private String fileExtension; 
  private int id;

  /**
   * Constructor for WDDeployableObjectPartType.
   */
  private WDDeployableObjectPartType(String typeName, String folderName, int id, String fileExtension) {
    this.typeName = typeName;
    this.folderName = folderName;
    this.id = id;
    this.fileExtension = fileExtension;     
  }

  /**
   * Returns all defined deployable object part types. 
   * @return all defined deployable object part types
   */
  public static Enumeration getAllWebResourceTypes() {
    return deployableObjectPartTypes.elements();
  }

  /**
   * Returns the <code>WDDEployableObjectPartType</code> specified by the type string.
   * 
   * @param typeString specifies the requested <code>WDDeployableObjectPartType</code>
   * @return the requested <code>WDDeployableObjectPartType</code>
   * @deprecated For better consistency across the Web Dynpro API this method
   *   has been renamed to {@link #valueOf(String)}. The method with the old 
   *   name has been kept for backward compatibility reasons and may be withdrawn
   *   with the first new NetWeaver release in 2006.
   */
  public static WDDeployableObjectPartType getType(String typeString) {
    return valueOf(typeString); 
  }

  /**
   * Returns the name of the deployable object part type. 
   * @return the name of the deployable object part type
   */
  public String toString() {
    return this.typeName;
  }

  /**
   * Returns the name of the folder associated to this type. 
   * @return the name of the folder associated to this type
   */
  public String getFolderNameForType() {
    return folderName;
  }

  /**
   * Returns a unique type identifier. 
   * @return a unique type identifier
   */
  public int getID() {
    return id;
  }

  /**
   * Returns the file extension of configuration files associated 
   * to this deployable object part type. 
   * @return the file extension of configuration files associated 
   * to this deployable object part type
   */
  public String getFileExtension() {
    return fileExtension;
  }
  
  /**
   * Return is the URL to the icon representing the current deployable object part type. 
   * 
   * @return the URL to the icon representing the current deployable object part type
   */ 
  public String getIconURL() {
    throw new UnsupportedOperationException("method not supported");  
  }  
  
  /**
   * Return is the <code>WDDeployableObjectPartType</code> instance for the specified <code>typeName</code>. 
   * If <code>typeName</code> is an undefined name, a <code>WDRuntimeException</code> is thrown. 
   * 
   * @param typeName name of the part type 
   * @return the <code>WDDeployableObjectPartType</code> instance for the specified <code>typeName</code>
   */
  public static WDDeployableObjectPartType valueOf(String typeName) {
    if (APPLICATION.typeName.equals(typeName)) {
      return APPLICATION;
    } else if (COMPONENT.typeName.equals(typeName)) {
      return COMPONENT;
    } else if (COMPONENT_CONFIGURATION.typeName.equals(typeName)) {
      return COMPONENT_CONFIGURATION;
    } else if (COMPONENT_INTERFACE.typeName.equals(typeName)) {
      return COMPONENT_INTERFACE;
    } else if (WEB_RESOURCE.typeName.equals(typeName)) {
      return WEB_RESOURCE;
    } else if (MODEL.typeName.equals(typeName)) {
      return MODEL;
    } else if (VARIANT.typeName.equals(typeName)) {
      return VARIANT;
    } else if (CUSTOMIZING_LAYER.typeName.equals(typeName)) {
      return CUSTOMIZING_LAYER;
    } else if (CONFIGURATION.typeName.equals(typeName)) {
      return CONFIGURATION;    
    } else {
      throw new WDRuntimeException("Unknown deployable object type name ''{0}''", new Object[]{typeName});
    }
  }
  
  /**
   * Return is the <code>WDDeployableObjectPartType</code> instance for the specified <code>folderName</code>. 
   * If <code>folderName</code> is an undefined folder name of a deployable object part, a 
   * <code>WDRuntimeException</code> is thrown. 
   * 
   * @param folderName name of the folder of a part type 
   * @return the <code>WDDeployableObjectPartType</code> instance for the specified <code>folderName</code>
   */
  public static WDDeployableObjectPartType valueOfFolderName(String folderName) {
    WDDeployableObjectPartType partType = null;
    if (COMPONENT.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.COMPONENT;
    } else if (APPLICATION.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.APPLICATION;
    } else if (COMPONENT_INTERFACE.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.COMPONENT_INTERFACE;
    } else if (VARIANT.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.VARIANT;
    } else if (CUSTOMIZING_LAYER.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.CUSTOMIZING_LAYER;
    } else if (CONFIGURATION.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.CONFIGURATION;
    } else if (WEB_RESOURCE.folderName.equals(folderName)) {    
      partType = WDDeployableObjectPartType.WEB_RESOURCE;    
    } else if (MODEL.folderName.equals(folderName)) {
      partType = WDDeployableObjectPartType.MODEL;      
    } else {
      throw new WDRuntimeException(
        "Unknown folder name ''{0}'' for deployable object part.",
        new Object[] { folderName });
    }
    return partType;
  }
  
  /**
   * Return is the number of part types. 
   * @return the number of part types
   */
  public static int getNumberOfPartTypes() {
    return NUMBER_OF_PARTS;   
  }  
}
