/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/WDDeployableObjectPart.java#1 $
 */

package com.sap.tc.webdynpro.services.sal.deployment.api;

import java.util.Collection;

import com.sap.tc.logging.Location;
import com.sap.tc.webdynpro.basesrvc.util.StringUtil;
import com.sap.tc.webdynpro.services.sal.deployment.core.DeployableObjectInternal;

/**
 * <code>WDDeployableObjectPart</code> represents the content of a Web Dynpro 
 * deployment unit, like applications, components, configurations or component 
 * interfaces. There is always exactly one deployable object related to a deployable 
 * object part.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public class WDDeployableObjectPart {

  static {
    Location.getLocation("ID.com.sap.tc.webdynpro.services.sal.deployment.api").infoT("$Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/WDDeployableObjectPart.java#1 $");
  }

  /** name of the corresponding deployable object */
  private final String deployableObjectName;
  
  /** name of the deployable object part */
  private final String name;
  
  /** the type of the deployable object part */
  private final WDDeployableObjectPartType type;
  
  /** short name of the deployable object */
  private final String shortName;
  
  private final String key; 

   
  /**
   * Constructs a <code>WDDeployableObjectPart</code> representing a deployable object part 
   * with the specified name, short name and type, that is contained within the deployable 
   * object <code>deployableObjectName</code>. 
   * 
   * @param deployableObjectName the name of the deployable object
   * @param name the name of the part to be created
   * @param shortName the short name of the deployable object part 
   * @param type the part type 
   * @throws NullPointerException if deployable object name, name or type is null or the empty string
   */
  public WDDeployableObjectPart(String deployableObjectName, String name, String shortName, WDDeployableObjectPartType type) {
    if (StringUtil.isEmpty(deployableObjectName)) {
      throw new NullPointerException("Parameter ''deployableObjectName'' is null."); 
    }
    if (StringUtil.isEmpty(name)) {
      throw new NullPointerException("Parameter ''name'' is null."); 
    } 
    if (type == null) {
      throw new NullPointerException("Parameter ''type'' is null.");    
    }    
    
    this.deployableObjectName = deployableObjectName;
    this.name = name;
    this.shortName = shortName;
    this.type = type;
    this.key = deployableObjectName + '/' + name + '/' + type;
  }

  /**
   * Constructs a WDDeployableObejctPart that represents a part of type 
   * <code>WDDeployableObjectPart.WEB_RESOURCE</code>. The constructor 
   * <code>new WDDeployableObectPart(deployableObjectName, "", WDDeployableObjectPartType.WEB_RESOURCE)</code>
   * creates the same object. 
   * 
   * @param deployableObjectName the name of the deployable object
   * @throws NullPointerException if <code>deployableObjectName</code> is null or the empty string
   */
  public WDDeployableObjectPart (String deployableObjectName) {
    if (StringUtil.isEmpty(deployableObjectName)) {
      throw new NullPointerException("Parameter ''deployableObjectName'' is null."); 
    }    
    this.deployableObjectName = deployableObjectName; 
    this.name = ""; 
    this.shortName = ""; 
    this.type = WDDeployableObjectPartType.WEB_RESOURCE;  
    this.key = deployableObjectName + '/' + name + '/' + type;
  }

  /**
   * Resolves the specified alias. An alias is a logical name for a specified   
   * application within a certain deployable object and can be defined in the 
   * configuration. 
   *
   * @param alias the alias that should be resolved
   * @return the deployable object that is described by the alias  
   * @throws WDAliasResolvingException if the alias cannot be resolved
   */
  public WDDeployableObjectPart resolveAlias(String alias) throws WDAliasResolvingException {
    return DeployableObjectInternal.getDeployableObjectInternal(deployableObjectName).resolveAlias(this, alias);
  }

  /**
   * Returns the deployableObjectName.
   * 
   * @return the name of the deployable object
   */
  public String getDeployableObjectName() {
    return deployableObjectName;
  }

  /**
   * Return is the deployable object to which the current deployable object part belongs.
   * 
   * @return the deployable object to which the current deployable object part belongs
   */ 
  public IWDDeployableObject getDeployableObject() {
    return DeployableObjectInternal.getDeployableObjectInternal(deployableObjectName);
  }

  /**
   * Returns the partName.
   * 
   * @return the name of the deployable object part
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the partType.
   * 
   * @return the type of the deployable object part
   */
  public WDDeployableObjectPartType getType() {
    return type;
  }
  
  /**
   * Returns the shortName.
   * 
   * @return the short name of the deployable object part
   */
  public String getShortName() {      
    return shortName;
  }

  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    return key;
  }
  
  /**
   * Return is the collection of defined aliases. 
   * Currently the method is not implemented and returns <code>null</code>. 
   *  
   * @return the collection of defined aliases; currently, the method is not 
   * implemented and <code>null</code> is returned 
   */
  public Collection getAliases(){
    return null;    
  }

}
