/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/WDDeployableObject.java#3 $
 */

package com.sap.tc.webdynpro.services.sal.deployment.api;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;

/**
 * <code>WDDeployableObject</code> provides static methods to access 
 * an <code>IWDDeployableObject</code> instance. Such an object represents 
 * a Web Dynpro deployment unit. It does not represent deployment units 
 * of other types (e.g. like Web application). 
 * <p>
 * <b>Example:</b> An <code>IWDDeployableObject</code> object can be accessed as follows:  
 * <pre>
 * // let objectName be the name of the deployable object 
 * IWDDeployableObject deployableObject = WDDeployableObject.getDeployableObject(objectName); 
 * </pre>
 *
 * @see com.sap.tc.webdynpro.services.sal.deployment.api.IWDDeployableObject
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDDeployableObject {

  /**
   * Returns the deployable object for the object specified by the given <code>objectName</code>. 
   *
   * @param objectName the name of the deployable object
   * @return the deployable object for the object specified by the given <code>objectName</code>
   * @throws WDRuntimeException if there is no deployable object deployed on the server for the 
   * given name that represents a Web Dynpro application or if the specified <code>objectName</code> 
   * does not represent a deployed Web Dynpro application, but for instance a Web application.    
   * @throws NullPointerException if objectName is null or the empty String
   */
  public static IWDDeployableObject getDeployableObject(String objectName) {
    return AbstractManager.getServerPlatformImplementation().getDeployableObjectFactory().getDeployableObject(objectName);
  } 
  
  /**
   * Returns a deployable object part description for the specified object part.
   * 
   * @param objectName the name of the deployable object
   * @param partName the name of the deployable object part
   * @param partType the type of the deployable object part
   * @return the specified deployable object part 
   * @throws WDDeplyomentException if the deployable object part is not deployed correctly or if 
   * the part couldn't be found 
   * @throws NullPointerException if <code>partName</code> is null or empty string or 
   * if <code>partType</code> is null   
   */  
  public static WDDeployableObjectPart getDeployableObjectPart(
    String objectName,
    String partName, 
    WDDeployableObjectPartType partType)
    throws WDDeploymentException {
    return AbstractManager.getServerPlatformImplementation().getDeployableObjectFactory().getDeployableObjectPart(
      objectName,
      partName,
      partType);
  }
}
