/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/IWDDeployableObject.java#1 $
 */
package com.sap.tc.webdynpro.services.sal.deployment.api;

import com.sap.tc.webdynpro.services.sal.api.IWDService;

/**
 * <code>IWDDeployableObject</code> specifies the interface of a deployment unit 
 * containing Web Dynpro content. 
 * <br>
 * An equivalent term for deployable object is <it>development component</it> (DC).   
 *  
 * @see com.sap.tc.webdynpro.services.sal.deployment.api.WDDeployableObject
 *  
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDDeployableObject extends IWDService {

  /**
   * Returns the name of deployable object. 
   *
   * @return the name of deployable object
   */
  public String getName();

  /**
   * Returns the <code>IWDDeployableObjectPart</code> instances located in the 
   * deployable object.
   * 
   * @return the array of deployable object parts which are located in this deployable object; 
   * if no part exists, an array of length 0 is returned
   */
  public WDDeployableObjectPart[] getParts();

  /**
   * Returns the <code>IWDDeployableObjectPart</code> instances located in the 
   * deployable object with the specified part type.
   * 
   * @param partType the type of the requested deployable object parts 
   * @return the array of deployable object parts; if there exists no deployable object part 
   * of the specified <code>partType</code>, an array of length 0 is returned 
   */
  public WDDeployableObjectPart[] getParts(WDDeployableObjectPartType partType);

  /**
   * Resolves the specified alias and return the referenced deployable object part. 
   * It is tried to resolve the specified alias: Therefore, the list of deployable object parts 
   * of the current deployable object is scanned and the first matching alias within one 
   * of these parts is used for the resolution of the alias. 
   * 
   * @param alias the alias that should be resolved
   * @return the deployable object part that is described by the alias
   * @throws WDAliasResolvingException if the specified alias cannot be resolved
   */
  public WDDeployableObjectPart resolveAlias(String alias) throws WDAliasResolvingException;

  /**
   * Resolves the specified alias on the level of the deployable object part contained within 
   * the current deployable object and return the referenced deployable object part.    
   *
   * @param alias the alias that should be resolved
   * @param part the deployable object part that defines the alias which should be resolved
   * @return the deployable object part that is described by the alias
   * @throws WDAliasResolvingException if the specified alias to an deployable object 
   * within a certain deployable object cannot be resolved
   */
  public WDDeployableObjectPart resolveAlias(WDDeployableObjectPart part, String alias)
    throws WDAliasResolvingException;

  /**
   * Return is the state of the deployable object. 
   * @return the state of the deployable object
   */
  public WDDeployableObjectState getState();

  /**
   * Return the list of referenced deployable objects.
   * 
   * @return an array containing the names of referenced deployable objects.
   * @throws WDSharingAccessException if the sharing references of this deployable object 
   * cannot be determined 
   */
  public String[] getSharingReferences() throws WDSharingAccessException;

  /**
   * Return the list of referenced deployable objects.
   * 
   * @return an array containing the referenced deployable objects.
   * @throws WDSharingAccessException if the referenced deployable objects of this deployable object 
   * cannot be determined    
   */
  public IWDDeployableObject[] getReferencedDeployableObjects() throws WDSharingAccessException;

  /**
   * Return the list of referenced services.
   * 
   * @return an array containing the names of referenced services.
   * @throws WDSharingAccessException if the service references of this deployable object 
   * cannot be determined    
   */
  public String[] getServiceReferences() throws WDSharingAccessException;
  
  /**
   * Return the list of referenced libraries.
   * 
   * @return an array containing the names of referenced libraries.
   * @throws WDSharingAccessException if the library references of this deployable object 
   * cannot be determined     
   */
  public String[] getLibrariesReferences() throws WDSharingAccessException;
  
  /**
   * Return the list of referenced interfaces.
   * 
   * @return an array containing the names of referenced interfaces.
   * @throws WDSharingAccessException if the interface references of this deployable object 
   * cannot be determined    
   */
  public String[] getInterfacesReferences() throws WDSharingAccessException;

  /**
   * Return the list of deployable objects referencing the actual deployable object.
   * 
   * @return an array containing the names of the deployable objects referencing the actual deployable object.
   * @throws WDSharingAccessException if the deployable objects referencing this deployable object 
   * cannot be determined
   */
  public String[] isReferencedBy() throws WDSharingAccessException;

  /**
   * Return the list of deployable objects referencing the actual deployable object.
   * 
   * @return an array containing the deployable objects referencing the actual deployable object.
   * @throws WDSharingAccessException if the deployable objects referencing this deployable object 
   * cannot be determined   
   */
  public IWDDeployableObject[] isReferencedByObjects() throws WDSharingAccessException;

  /**
   * Return the list of used JCO clients defined in the actual deployable object.
   * 
   * @return an array containing the names of the used referenced JCO clients.
   * @throws WDSharingAccessException if the used JCO clients of this deployable object 
   * cannot be determined   
   */
  public String[] getUsedJCOClients() throws WDSharingAccessException;
}
