package com.sap.tc.loggingStandard;

/**
 * Title:        LoggingStandard
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      SAP Markets, Inc
 * @author
 * @version $Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/Logger.java#1 $
 */

import java.lang.*;
import java.util.*;

import com.sap.tc.logging.*;

/**
 * <p>
 *   This class is similar to <code>java.util.logging.Logger</code>, that
 *   contains a subset of method calls that are mainly for writing log
 *   messages.
 * </p>
 * <p>
 *   All the methods that supports the output of a log messages are available,
 *   including the check for resource bundle for translation, and the
 *   additional check if message is loggable or not. <br>
 *   Other configuration, such as assigning handler, filter is not mapped. And
 *   concept of global handlers is not supported in the API, but is made
 *   possible through the configuration file.
 * </p>
 * <p>
 *   Note that the output API is based on the JDK1.4 beta 2 version.
 * </p>
 */
public class Logger {

  static String version = "$Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/Logger.java#1 $";

  private final static String ENTRY_TEXT = "ENTRY";
  private final static String ENTRY_TEXT_PARAMS = ENTRY_TEXT + " with {0}";
  private final static String EXIT_TEXT = "RETURN";
  private final static String EXIT_TEXT_PARAMS = EXIT_TEXT + " with {0}";
  private final static String THROW_TEXT = "THROWING {0}";

  protected Logger(){
  }

  protected Logger(Location loc){
    name = loc.getName();    //root ok: empty string
    _loc = loc;
    _wLoc = new WLocation(_loc);

//new
    //SPECIAL case: ROOT: singleton, from LogManager constructor
    if (loc == Location.getRoot()){
      if (loc.getMaximumSeverity() == loc.getMinimumSeverity()){  //already manipulated by SAP API
        this.forcedLevel = Level.toLevel(loc.getEffectiveSeverity());
      }
      else{
        this.forcedLevel = Level.INFO;
        _loc.setEffectiveSeverity(Severity.INFO);
      }
      this.localEffSev = _loc.getEffectiveSeverity();
      return;
    }

    //OTHERS
//    Logger _effLogger;
//    Location _effLoc = loc;
    int _realSev;

    _realSev = this.getValidSev(loc);

    //vital step: explicitly set Effective severity value
    _loc.setEffectiveSeverity(_realSev);
    this.localEffSev = _realSev;
    this.forcedLevel = Level.toLevel(_realSev);
  }

  /**
   * Find or create a logger for a named subsystem.  If a logger has
   * already been created with the given name it is returned.  Otherwise
   * a new logger is created.
   * <p>
   * If a new logger is created its log level will be configured
   * based on the LogManager info.
   * It will be registered in the LogManager global namespace.  This will
   * mean it will be affected by subsequent LogManager.setLevel calls.
   *
   * @param	name		A name for the logger.  This should
   *				be a dot-separated name and should normally
   *				be based on the package name or class name
   *				of the subsystem, such as java.net
   *				or javax.swing. No empty Logger name allowed, as
   *                            root is always there.
   * @return a suitable Logger
   */
  public static synchronized Logger getLogger(String name) {
    if (name.equals("")) {
      throw new IllegalArgumentException("need a logger name");
    }
//    String classname = classnameLookup();   //static .....bad
//    if (classname.equals("")){
//      classname = name;    //the best we can do
//    }
//    return LogManager.getLogManager().makeLogger(name, classname);
    return LogManager.getLogManager().makeLogger(name);
  }

  /**
   * Find or create a logger for a named subsystem.  If a logger has
   * already been created with the given name it is returned.  Otherwise
   * a new logger is created.
   * <p>
   * If a new logger is created its log level will be configured
   * based on the LogManager. It will be registered in
   * the LogManager global namespace.  This will mean it will be
   * affected by subsequent LogManager.setLevel calls.
   * <p>
   * If the named Logger already exists and does not yet have a
   * localization resource bundle then the given resource bundle
   * name is used.  If the named Logger already exists and has
   * a different resource bundle name then an IllegalArgumentException
   * is thrown.
   * <p>
   * @param	name	A name for the logger.  This should
   *				be a dot-separated name and should normally
   *				be based on the package name or class name
   *				of the subsystem, such as java.net
   *				or javax.swing.No empty string allowed, as
   *                          root is always there.
   * @param 	resourceBundleName  name of ResourceBundle to be used for localizing
   *				messages for this logger.
   * @return a suitable Logger
   * @throws MissingResourceException if the named ResourceBundle cannot be found.
   * @throws IllegalArgumentName if the Logger already exists and uses
   *		   a different resource bundle name.
   */
  public static synchronized Logger getLogger(String name, String resourceBundleName) {
    if (name.equals("")) {
      throw new IllegalArgumentException("need a logger name");
    }
    ResourceBundle.getBundle(resourceBundleName);

//    String classname = classnameLookup();
//    if (classname.equals("")){
//      classname = name;    //the best we can do
//    }
    Logger newLogger= LogManager.getLogManager().makeLogger(name, resourceBundleName);
//    Logger newLogger =
//        LogManager.getLogManager().makeLogger(name, classname, resourceBundleName);

    if (!newLogger.getLoc().getResourceBundleName().equalsIgnoreCase(resourceBundleName)){
      throw new IllegalArgumentException(newLogger.getLoc().getResourceBundleName() +
	         			" != " + resourceBundleName);
    }
    return newLogger;
  }


    /**
     * Set the log level specifying which message levels will be
     * logged by this logger (not affecting children).  Message levels
     * lower than this value will be discarded.  The level value Level.OFF
     * can be used to turn off logging.
     *
     * @param newLevel   the new value for the log level
     * @exception  SecurityException  if a security manager exists and if
     *             the caller does not have LoggingPermission("control").
     */
  public synchronized void setLevel(Level newLevel) throws SecurityException {
    if (newLevel == null) {
      throw new NullPointerException();
    }
    int sev = Level.mapLevel(newLevel);
    if (_loc.getEffectiveSeverity() != this.localEffSev){ //somehow messed by external SAP API
      this.effSevByLoc = _loc.getEffectiveSeverity();
    }
    _loc.setEffectiveSeverity(sev);
    this.localEffSev = sev;
  }

  /**
   * Get the log level specifying which messages will be
   * logged by this logger.
   * @return	the level of messages being logged.
   */
  public Level getLevel() {
    return Level.toLevel(_loc.getEffectiveSeverity());
  }

  /**
   * Get the name for this logger.
   * @return logger name.
   */
  public String getName() {
    return name;
  }


  /**
   * Retrieve the localization resource bundle name for this
   * logger.
   * @return localization bundle name (may be null)
   */
  public String getResourceBundleName() {
    return _loc.getResourceBundleName();
  }

  /**
   * Retrieve the localization resource bundle for this
   * logger.
   * @return localization bundle (may be null)
   */
  public ResourceBundle getResourceBundle() {
    String rbName;
    if ((rbName = _loc.getResourceBundleName()) == null)
      return null;
    return ResourceBundle.getBundle(rbName);
  }

  /**
   * Check if a message of the given level would actually be logged
   * by this logger, in terms of level comparison.
   * @param	level	a message logging level
   * @return	true if the given message level is currently being logged.
   */
  public boolean isLoggable(Level level) {
    return _loc.beLogged(Level.mapLevel(level));
  }

//**************************
//*****Logging methods******
//**************************
//(1) simple severity logging method
//(2) basic log methods (with argument(s))
//(3) intermediate log methods (with specific class & method names,
//                              not-known resourceBundle)
//(4) intermediate log methods (with specific class & method names & ResourceBundle)
//(5) flow logic: entering, exiting, throwing

////////////////////////////////////
//(1) simple severity logging method
////////////////////////////////////
  /**
   * Log a CONFIG message.
   * <p>
   * If the logger is currently enabled for the CONFIG message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void config(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.CONFIG))){
//    if (Level.mapLevel(Level.CONFIG) < _loc.getEffectiveSeverity()) {
      return;
    }
    log(Level.CONFIG, msg);
  }

  /**
   * Log a FINE message.
   * <p>
   * If the logger is currently enabled for the FINE message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void fine(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINE))){
      return;
    }
    log(Level.FINE, msg);
  }

  /**
   * Log a FINER message.
   * <p>
   * If the logger is currently enabled for the FINER message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void finer(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    log(Level.FINER, msg);
  }

  /**
   * Log a FINEST message.
   * <p>
   * If the logger is currently enabled for the FINEST message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void finest(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINEST))){
      return;
    }
    log(Level.FINEST, msg);
  }

  /**
   * Log a INFO message.
   * <p>
   * If the logger is currently enabled for the INFO message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void info(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.INFO))){
      return;
    }
    log(Level.INFO, msg);
  }


  /**
   * Log a SEVERE message.
   * <p>
   * If the logger is currently enabled for the SEVERE message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void severe(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.SEVERE))){
      return;
    }
    log(Level.SEVERE, msg);
  }

  /**
   * Log a WARNING message.
   * <p>
   * If the logger is currently enabled for the WARNING message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void warning(String msg) {
    if(!_loc.beLogged(Level.mapLevel(Level.WARNING))){
      return;
    }
    log(Level.WARNING, msg);
  }

//////////////////////////////////////////
//(2) basic log methods (with argument(s))
// In SUN, it always have 'inferCaller'
// to resolve the actual class & method names
//////////////////////////////////////////
  /**
   * Log a message, with no arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	One of the message level identifiers, e.g. SEVERE
   * @param   msg	The string message (or a key in the message catalog)
   */
  public void log(Level level, String msg) {
//    _loc.logT(Level.mapLevel(level), msg);

//Discrepancy between SUN & SAP:
//Assumption:- Name of location is valid and intelligent ====>package name

    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    logp(level, _loc.getName(), methodnameLookup(), msg);
    String[] autoNames = {"", ""};
    autoSourceLookup(autoNames);
    logp(level, autoNames[0], autoNames[1], msg);
  }

  /**
   * Log a message, with no arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	One of the message level identifiers, e.g. SEVERE
   * @param   msg	The string message (or a key in the message catalog)
   * @param   param1    parameter to the message
   */
  public void log(Level level, String msg, Object param1) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    logp(level, _loc.getName(), methodnameLookup(), msg, param1);
    String[] autoNames = {"", ""};
    autoSourceLookup(autoNames);
    logp(level, autoNames[0], autoNames[1], msg, param1);
  }

  /**
   * Log a message, with an array of object arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	One of the message level identifiers, e.g. SEVERE
   * @param   msg	The string message (or a key in the message catalog)
   * @param   params    array of parameters to the message
   */
  public void log(Level level, String msg, Object[] params) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    logp(level, _loc.getName(), methodnameLookup(), msg, params);
    String[] autoNames = {"", ""};
    autoSourceLookup(autoNames);
    logp(level, autoNames[0], autoNames[1], msg, params);
  }

  /**
   * Log a message, with associated Throwable information.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * Note (SAP wrapper):
   * that the thrown argument is stored in the LogRecord parameters property.
   * <p>
   * @param	level	One of the message level identifiers, e.g. SEVERE
   * @param   msg	The string message (or a key in the message catalog)
   * @param   params    array of parameters to the message
   */
  public void log(Level level, String msg, Throwable thrown) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    logp(level, _loc.getName(), methodnameLookup(), msg, thrown);
    String[] autoNames = {"", ""};
    autoSourceLookup(autoNames);
    logp(level, autoNames[0], autoNames[1], msg, thrown);
  }

///////////////////////////////////////////////////////////////////
//(3) intermediate log methods (with specific class & method names)
//                              not-known resourceBundle)
///////////////////////////////////////////////////////////////////
  /**
   * Log a message, specifying source class and method, with no arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param   msg	 The string message (or a key in the message catalog)
   */
  public void logp(Level level,
                   String sourceClass, String sourceMethod,
                   String msg) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    _loc.logT(Level.mapLevel(level), sourceMethod, msg);
    if ((_loc.getResourceBundleName() != null) && (_loc.getResourceBundleName() != "")){
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, _loc.getResourceBundleName(), null);
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg);
    }
  }

  /**
   * Log a message, specifying source class and method, with a single object
   * parameter to the log message.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param   msg	 The string message (or a key in the message catalog)
   * @param  param1  Parameter to the log message
   */
  public void logp(Level level,
                   String sourceClass, String sourceMethod,
                   String msg,
                   Object param1) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    _loc.logT(Level.mapLevel(level), sourceMethod, msg, new Object[]{param1});
    if ((_loc.getResourceBundleName() != null) && (_loc.getResourceBundleName() != "")){
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, _loc.getResourceBundleName(), null,
                  new Object[]{param1});
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg,
                  new Object[]{param1});
    }
  }

  /**
   * Log a message, specifying source class and method, with an array of object
   * arguments
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param   msg	 The string message (or a key in the message catalog)
   * @param  params  Array of parameters to the message
   */
  public void logp(Level level,
                   String sourceClass, String sourceMethod,
                   String msg,
                   Object[] params) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    _loc.logT(Level.mapLevel(level), sourceMethod, msg, params);
    if ((_loc.getResourceBundleName() != null) && (_loc.getResourceBundleName() != "")){
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, _loc.getResourceBundleName(), null, params);
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg, params);
    }
  }

  /**
   * Log a message, specifying source class and method, with associated
   * Throwable information.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * Note (SAP wrapper):
   * that the thrown argument is stored in the LogRecord parameters property.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param   msg	 The string message (or a key in the message catalog)
   * @param  thrown      Throwable associated with log message.
   */
  public void logp(Level level,
                   String sourceClass, String sourceMethod,
                   String msg,
                   Throwable thrown) {
    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
//    _loc.logT(Level.mapLevel(level), sourceMethod, msg, new Object[]{thrown});
    if ((_loc.getResourceBundleName() != null) && (_loc.getResourceBundleName() != "")){
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, _loc.getResourceBundleName(), null,
                  new Object[]{thrown});
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg,
                  new Object[]{thrown});
    }
  }

////////////////////////////////////////////////////////////////////////////////////
//(4) intermediate log methods (with specific class & method names & ResourceBundle)
////////////////////////////////////////////////////////////////////////////////////
  /**
   * Log a message, specifying source class, method, and resource bundle name
   * with no arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * The msg string is localized using the named resource bundle. If the
   * resource bundle name is null, then the msg string is not localized.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param bundleName   name of resource bundle to localize msg
   * @param   msg	 The string message (or a key in the message catalog)
   * @throws  MissingResourceException if no suitable ResourceBundle can be
   *                                      found.
   */
  public void logrb(Level level,
                   String sourceClass, String sourceMethod,
                   String bundleName,
                   String msg) {

    if (bundleName != null)
      ResourceBundle.getBundle(bundleName);   //throw if missing

    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
    if (bundleName != null){
    //  _loc.setResourceBundleName(bundleName);
    //_loc.log(Level.mapLevel(level), intCat, sourceMethod, msg, msg);
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, bundleName, null);
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg);
    }
  }

  /**
   * Log a message, specifying source class, method, and resource bundle name
   * with a single object parameter to the log message.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * The msg string is localized using the named resource bundle. If the
   * resource bundle name is null, then the msg string is not localized.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param bundleName   name of resource bundle to localize msg
   * @param   msg	 The string message (or a key in the message catalog)
   * @param param1       Parameter to the log message
   * @exception  MissingResourceException if no suitable ResourceBundle can be
   *                                      found.
   */
  public void logrb(Level level,
                   String sourceClass, String sourceMethod,
                   String bundleName,
                   String msg, Object param1) {

    if (bundleName != null)
      ResourceBundle.getBundle(bundleName);

    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
    if (bundleName != null){
    // _loc.setResourceBundleName(bundleName);
    //_loc.log(Level.mapLevel(level), intCat, sourceMethod, msg,
    //         new Object[]{param1}, msg);
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, bundleName, null, new Object[]{param1});
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg, new Object[]{param1});
    }

  }

  /**
   * Log a message, specifying source class, method, and resource bundle name
   * with an array of object arguments.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * The msg string is localized using the named resource bundle. If the
   * resource bundle name is null, then the msg string is not localized.
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param bundleName   name of resource bundle to localize msg
   * @param   msg	 The string message (or a key in the message catalog)
   * @param params       Array of parameters to the message
   * @exception  MissingResourceException if no suitable ResourceBundle can be
   *                                      found.
   */
  public void logrb(Level level,
                   String sourceClass, String sourceMethod,
                   String bundleName,
                   String msg, Object[] params) {
    if (bundleName != null)
      ResourceBundle.getBundle(bundleName);   //throw if missing

    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }
    if (bundleName != null){
    //  _loc.setResourceBundleName(bundleName);
    //_loc.log(Level.mapLevel(level), intCat, sourceMethod, msg,
    //         params, msg);
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, bundleName, null, params);
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg, params);
    }
  }

  /**
   * Log a message, specifying source class, method, and resource bundle name
   * with associated Throwable information.
   * <p>
   * If the logger is currently enabled for the given message
   * level then the given message is forwarded to all the
   * registered output Handler objects.
   * <p>
   * The msg string is localized using the named resource bundle. If the
   * resource bundle name is null, then the msg string is not localized.
   * <p>
   * Note (SAP wrapper):
   * that the thrown argument is stored in the LogRecord parameters property
   * <p>
   * @param	level	 One of the message level identifiers, e.g. SEVERE
   * @param sourceClass  name of class that issued the logging request
   * @param sourceMethod name of method that issued the logging request
   * @param bundleName   name of resource bundle to localize msg
   * @param   msg	 The string message (or a key in the message catalog)
   * @param thrown       Throwable associated with log message.
   * @exception  MissingResourceException if no suitable ResourceBundle can be
   *                                      found.
   */
  public void logrb(Level level,
                   String sourceClass, String sourceMethod,
                   String bundleName,
                   String msg, Throwable thrown) {

    if (bundleName != null)
      ResourceBundle.getBundle(bundleName);   //throw if missing

    if(!_loc.beLogged(Level.mapLevel(level))){
      return;
    }

    if (bundleName != null){
     // _loc.setResourceBundleName(bundleName);
//    _loc.log(Level.mapLevel(level), intCat, sourceMethod, msg,
//             new Object[]{thrown}, msg);
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  msg, bundleName, null, new Object[]{thrown});
    }
    else{
      _wLoc.logST(Level.mapLevel(level), intCat, sourceClass, sourceMethod,
                  null, null, msg, new Object[]{thrown});
    }
  }

/////////////////////////////////////////////
//(5) flow logic: entering, exiting, throwing
/////////////////////////////////////////////

  /**
   * Log a method entry.
   * <p>
   * This is a convenience method that can be used to log entry
   * to a method.  A LogRecord with message "ENTRY", log level
   * FINER, and the given sourceMethod and sourceClass is logged.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod   name of method that is being entered
   */
  public void entering(String sourceClass, String sourceMethod) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    logp(Level.FINER, sourceClass, sourceMethod, ENTRY_TEXT);
  }

  /**
   * Log a method entry, with one parameter
   * <p>
   * This is a convenience method that can be used to log entry
   * to a method.  A LogRecord with message "ENTRY with {0}", log level
   * FINER, and the given sourceMethod and sourceClass and parameter is logged.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod   name of method that is being entered
   * @param   param1  parameter to the method being entered
   */
  public void entering(String sourceClass, String sourceMethod, Object param1) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    logp(Level.FINER, sourceClass, sourceMethod, ENTRY_TEXT_PARAMS, param1);
  }

  /**
   * Log a method entry, with an array of parameters.
   * <p>
   * This is a convenience method that can be used to log entry
   * to a method.  A LogRecord with message "ENTRY", (followed by a
   * format {N} indicator for each entry in the parameter array),
   * log level FINER, and the given sourceMethod and sourceClass,
   * and parameters is logged.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod   name of method that is being entered
   * @param   params  array of parameters to the method being entered
   */
  public void entering(String sourceClass, String sourceMethod, Object[] params) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    String entryMsg = ENTRY_TEXT;
    if (params != null) {
      for(int i=0;i<params.length;i++){
        entryMsg = entryMsg + " {" + i + "}";
      }
    }
    logp(Level.FINER, sourceClass, sourceMethod, entryMsg, params);
  }


  /**
   * Log a method return.
   * <p>
   * This is a convenience method that can be used to log returning
   * from a method.  A LogRecord with message "RETURN", log level
   * FINER, and the given sourceMethod and sourceClass is logged.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod   name of the method
   */
  public void exiting(String sourceClass, String sourceMethod) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    logp(Level.FINER, sourceClass, sourceMethod, EXIT_TEXT);
  }

  /**
   * Log a method return, with result object.
   * <p>
   * This is a convenience method that can be used to log returning
   * from a method.  A LogRecord with message "RETURN with {0}", log level
   * FINER, and the given sourceMethod and sourceClass, and result object is
   * logged.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod   name of the method
   * @param   result         Object that is being returned
   */
  public void exiting(String sourceClass, String sourceMethod, Object result) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    logp(Level.FINER, sourceClass, sourceMethod, EXIT_TEXT_PARAMS,
                                                 new Object[]{result});
  }


  /**
   * Log throwing an exception.
   * <p>
   * This is a convenience method to log that a method is
   * terminating by throwing an exception.  The logging is done
   * using the FINER level.
   * <p>
   * If the logger is currently enabled for the given message
   * level then it is forwarded to all registered output destinations.  The
   * LogRecord's message is set to "THROWING {0}".
   * <p>
   * Note that the thrown argument is stored in the LogRecord parameters
   * property.
   * <p>
   * @param   sourceClass    name of class that issued the logging request
   * @param   sourceMethod  name of the method.
   * @param   thrown  The Throwable that is being thrown.
   */
  public void throwing(String sourceClass, String sourceMethod, Throwable thrown) {
    if(!_loc.beLogged(Level.mapLevel(Level.FINER))){
      return;
    }
    logp(Level.FINER, sourceClass, sourceMethod, THROW_TEXT, new Object[]{thrown});
  }

//****The following methods, mainly dealing with configuration, are not mapped//
//?addHandler
//?getAnonymousLogger()
//?getAnonymousLogger(String)
//?getFilter
//?getHandlers
//?getUseGlobalHandlers
//?setUseGlobalHandlers
//?setFilter

  //Resolve the information of the Source caller: Classname, methodname
  //Return: autoNames[0] : source class name
  //        autoNames[1] : source method name
  private void autoSourceLookup(String[] autoNames){
   //TODO: try to use stackFrame ....
    final String method = "autoSourceLookup(String[])";
    try {
      classLoc.entering(method);
      java.io.StringWriter sw = new java.io.StringWriter();
      java.io.PrintWriter pw = new java.io.PrintWriter(sw);
      (new Exception("")).printStackTrace(pw);
      pw.close();
      String trace = sw.toString();
      // Find first call on Logger.
      int ix = trace.lastIndexOf("com.sap.tc.loggingStandard.Logger");
      ix = trace.indexOf("(", ix)+3;
      // Now skip to the next stack frame.
      int end = trace.indexOf("(", ix);
      int dot = trace.lastIndexOf(".", end);
      int start = trace.lastIndexOf(" ", dot)+1;
      autoNames[0] = trace.substring(start, dot);
      autoNames[1] = trace.substring(dot+1, end);
    } catch (Exception ex) {
	    // ok. Simply try the best already. Not serious
       LogManager.getIntErrorCat().infoT(classLoc,
                                          method,
                                          "Not able to resolve source classname");
    }
    classLoc.exiting(method, autoNames);
  }

  //Routine to get the valid severity from ancestor, be it manipulated by
  //SAP API (on location directly) or by wrapper API (through setLevel(s))
  //Param: _effLoc: location to be evaluated
  //Return: the real valid severity from the ancestor
  static int getValidSev(Location loc){
    Logger _effLogger;
    Location _effLoc = loc;
    int _realSev;

    while ((_effLoc.getMaximumSeverity() == Severity.NONE) &&
           (_effLoc.getMinimumSeverity() == Severity.ALL)){
      _effLoc = _effLoc.getParent();     //ok...can all the way to the ROOT
    }
    _effLogger = LogManager.getLogManager().getLogger(_effLoc.getName());
    if (_effLogger == null){
      _realSev = _effLoc.getEffectiveSeverity();   //observe whatever SAP value, if any
    }
    else{    //logger exists already, use 'forcedLevel'
      int newEffSev = _effLoc.getEffectiveSeverity();
      if(_effLogger.getLocalEffSev() != newEffSev){  //external change from SAP API, need sync
        _effLogger.setForcedLevel(Level.toLevel(newEffSev));
        _effLogger.setLocalEffSev(newEffSev);
        _effLogger.setEffSevByLoc(-1);    //for sequence of setLevel, setEffSev, setLevel, setEffSev
      }
      else{
        if(_effLogger.getEffSevByLoc() >= 0){
          _effLogger.setForcedLevel(Level.toLevel(_effLogger.getEffSevByLoc()));
          _effLogger.setEffSevByLoc(-1);
//          _effLogger.setLocalEffSev(_effLogger.getEffSevByLoc());  //NO: not nec
        }
        //else  ----> no change from SAP API, do nothing, use currently available 'forcedLevel'
      }

      _realSev = Level.mapLevel(_effLogger.getForcedLevel());
    }
    return _realSev;
  }     //getValidSev(loc)


  protected Location getLoc(){
    return _loc;
  }

  Category getIntCat(){
    return intCat;
  }

//new
  protected void setForcedLevel(Level inLevel){
    forcedLevel = inLevel;
  }
//new
  protected Level getForcedLevel(){
    return forcedLevel;
  }

//new
  protected void setLocalEffSev(int sev){
    localEffSev = sev;
  }
//new
  protected int getLocalEffSev(){
    return localEffSev;
  }
//new
  protected void setEffSevByLoc(int sev){
    effSevByLoc = sev;
  }
//new
  protected int getEffSevByLoc(){
    return effSevByLoc;
  }


  private Location _loc;
  private WLocation _wLoc;
  private String name;
//new
  private Level forcedLevel;  //keep track of 'global' log level
  private int localEffSev;    //keep track of internal location eff. severity
  private int effSevByLoc = -1;  //change eff. Severity by SAP API

  private static Category intCat = Category.getCategory("/System/LoggingStandard/Logger/internal");
                          //default is good: NONE
  private static Location classLoc    = Location.getLocation(Logger.class);
}