package com.sap.tc.loggingStandard;

/**
 * Title:        LoggingStandard
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      SAP Markets, Inc
 * @author
 * @version $Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/LogManager.java#1 $
 */

import java.lang.*;
import java.util.*;

import com.sap.tc.logging.*;

/**
 * <p>
 *   This class is similar to <code>java.util.logging.LogManager</code>.
 * </p>
 * <p>
 *   It contains mainly the factory methods that retrieves the unique
 *   instance of itself and creates Logger instance, plus methods
 *   that will affect the message output: severity level setting. Other
 *   configuration methods are not included.
 * </p>
 */

public class LogManager{

  static String version = "$Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/LogManager.java#1 $";

  protected LogManager(){
    _loggers = new HashMap();
    Logger _logger = new Logger(Location.getRoot());
    _loggers.put("", _logger);
  }

 /**
   * Return the unique, global LogManager.
   */
  public static LogManager getLogManager() {
    if (_manager == null)
      _manager = new LogManager();
    return _manager;
  }

 /**
  * Method to find a named logger.
  * Note that since untrusted code may create loggers with arbitrary names
  * this method should not be relied on to find Loggers for security
  * sensitive logging.
  * <p>
  * @param name name of the logger
  * @return  matching logger or null if none is found
  */
  public Logger getLogger(String name){
    if (_loggers.containsKey(name))
      return (Logger) _loggers.get(name);
    return null;
  }


//  protected Logger makeLogger(String name, String classname){
  protected Logger makeLogger(String name){
    Logger _logger = this.getLogger(name);
    if(_logger != null)
      return _logger;
    //_logger = new Logger(name, Location.getLocation(classname));
    _logger = new Logger(Location.getLocation(name));
    _loggers.put(name, _logger);
    return _logger;
  }


//  protected Logger makeLogger(String name, String classname, String rbName){
  protected Logger makeLogger(String name, String rbName){
    Logger _logger = getLogger(name);
    if((_logger != null) &&
        ((_logger.getResourceBundleName() != null) && (_logger.getResourceBundleName() != "")))
      return _logger;
    //Location loc = Location.getLocation(classname);
    Location loc = Location.getLocation(name);
    loc.setResourceBundleName(rbName);
    //_logger = new Logger(name, loc);
    _logger = new Logger(loc);

    _loggers.put(name, _logger);
    return _logger;
  }


  /**
   * Get an enumeration of known logger names.
   * <p>
   * Note:  Loggers may be added dynamically as new classes are loaded.
   * This method only reports on the loggers that are currently registered.
   * <p>
   * @return  enumeration of logger name strings
   */
  public synchronized Enumeration getLoggerNames() {
    LoggerEnum resultEnum = new LoggerEnum();
    Iterator loggers = _loggers.keySet().iterator();
    while (loggers.hasNext()){
      resultEnum.addLoggerName((String)loggers.next());
    }
    return resultEnum;
  }

  static class LoggerEnum implements Enumeration{
    private Vector loggerNames;
    private int index = 0;

    LoggerEnum(){
      loggerNames = new Vector();
    }

    private void addLoggerName(String name){
      loggerNames.addElement(name);
    }

    public boolean hasMoreElements(){
      return (index < loggerNames.size());
    }

    public Object nextElement() {
      if (index >= loggerNames.size()) {
	throw new NoSuchElementException();
      }
      return (Object)loggerNames.elementAt(index++);
    }
  }

  /**
   * Get the log level for a given logger name.
   * See the setLevel method for more details.
   *
   * @param   name    The logger name
   * @return the log level
   */
  public synchronized Level getLevel(String name) {
  //  if logger does not exist, it has to 'predict' & evaluate
    Logger _logger = (Logger)_loggers.get(name);
    if (_logger == null){
      ////should it search up for the closest forcedLevel, as in the Logger constructor
      int _sev;
      _sev = Logger.getValidSev(Location.getLocation(name));
      return Level.toLevel(_sev);
//      return Level.toLevel(Location.getRoot().getEffectiveSeverity());
    }
    return Level.toLevel(_logger.getLoc().getEffectiveSeverity());
  }

    /**
     * Set a log level for a given set of loggers.  Subsequently the
     * target loggers will only log messages whose types are greater
     * than or equal to the given level.  The level value Level.OFF
     * can be used to turn off logging.
     * <p>
     * A setLevel() from Manager will FORCE the update of severity on
     * the specified logger and its children.
     * <p>
     * The empty string "" can be used to identify the root.
     *
     * @param   name    The logger name to update.
     * @param   level   the new log level, e.g. Level.SEVERE or Level.FINER
     * @exception  SecurityException  if a security manager exists and if
     *             the caller does not have LoggingPermission("control").
     */
    public synchronized void setLevel(String name, Level level) throws SecurityException {
      //simply, access the logger and also traverse all children and call
      //setEffectiveSeverity() explicitly.
      if (level == null || name == null) {
        throw new NullPointerException();
      }
//SUN: Doesn't matter if logger exists or not, the node will be created anyway, but
//Logger created&assigned whenever later.
//Here, still no logger, simply create LogController.....doesn't hurt

      Logger _logger = getLogger(name);
//      if (_logger == null)
//        _logger = Logger.getLogger(name);
//new
      if (_logger != null){
        _logger.setForcedLevel(level);
        _logger.setLocalEffSev(Level.mapLevel(level));
        _logger.setEffSevByLoc(-1);
      }

//      _logger.getLoc().setEffectiveSeverity(Level.mapLevel(level));
      Location a_loc = Location.getLocation(name);
      a_loc.setEffectiveSeverity(Level.mapLevel(level));
//      Iterator iterChild = _logger.getLoc().getChildren().iterator();    //Collection
      Iterator iterChild = a_loc.getChildren().iterator();    //Collection

      Logger _childLogger;
      while (iterChild.hasNext()){
        Location tmpLoc = (Location)iterChild.next();
//new
        _childLogger = this.getLogger(tmpLoc.getName());
        if (_childLogger != null){
          _childLogger.setForcedLevel(level);
          _childLogger.setLocalEffSev(Level.mapLevel(level));
          _childLogger.setEffSevByLoc(-1);
        }

        tmpLoc.setEffectiveSeverity(Level.mapLevel(level));
        Iterator moreChild = tmpLoc.getChildren().iterator();   //depth search
        while (moreChild.hasNext()){
          setLevel(((Location)moreChild.next()).getName(), level);
        }
      }
    }
//?addLogger  (mainly internal factory method....can be skipped here)
//?checkAccess
//?Publish


  static class TokenType extends com.sap.tc.loggingStandard.TokenType {

    public static final int COMMA         = com.sap.tc.loggingStandard.TokenType.MAX + 1,
                            DOT           = COMMA + 1,
                            SLASH         = DOT + 1,
                            LEFT_PARAN    = SLASH + 1,
                            RIGHT_PARAN   = LEFT_PARAN + 1,
                            LEFT_BRACKET  = RIGHT_PARAN + 1,
                            RIGHT_BRACKET = LEFT_BRACKET + 1;
    public static final int MAX           = RIGHT_BRACKET;

  }

  static class Tokenizer extends com.sap.tc.loggingStandard.Tokenizer {

    public Tokenizer(String content) {
      super(content);
    }

    public Token nextTokenInt() {
      Token token = super.nextTokenInt();
      int   type  = token.getType();

      if (type != TokenType.UNKNOWN) {
        return token;
      } else {
        char val = token.getValue().charAt(0);

        switch (val) {
          case ',': {
            return new Token(TokenType.COMMA,
                             ",");
          }
          case '.': {
            return new Token(TokenType.DOT,
                             ".");
          }
          case '/': {
            return new Token(TokenType.SLASH,
                             "/");
          }
          case '(': {
            return new Token(TokenType.LEFT_PARAN,
                             "(");
          }
          case ')': {
            return new Token(TokenType.RIGHT_PARAN,
                             ")");
          }
          case '[': {
            return new Token(TokenType.LEFT_BRACKET,
                             "[");
          }
          case ']': {
            return new Token(TokenType.RIGHT_BRACKET,
                             "]");
          }
          default: {
            return token;
          }
        }
      }
    }

  }



  static Category getIntErrorCat(){
    if (_errLogger == null){
      _errLogger = Category.getCategory("/System/Logging");
      _errLogger.setMaximumSeverity(Severity.ERROR);
      _errLogger.addLog(new ConsoleLog());
    }
    return _errLogger;
  }


  private static LogManager _manager;
  private HashMap _loggers;
  private static Category _errLogger = null;

}