/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/configuration/IConfiguration.java#2 $
 */

package com.sapportals.wcm.crt.configuration;

/**
 * This interface represents a node of hierarchical configuration data. A
 * configuration data node has a name, attributes (name/value pairs) and child
 * nodes. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/configuration/IConfiguration.java#3
 *      $
 */
public interface IConfiguration {
  /**
   * Returns the name of the node (not <code>null</code> ).
   *
   * @return the name of the node (not <code>null</code> ).
   */
  String getName();

  /**
   * Returns a string describing the location of the configuration data. This
   * string is client specific, examples: line number (for file-based
   * persistence), identifier, URI, database table, ...
   *
   * @return location
   */
  String getLocation();

  /**
   * Returns the namespace this configuration node belongs to.
   *
   * @return the namespace this configuration node belongs to.
   * @exception ConfigurationException Exception raised in failure situation
   */
  String getNamespace()
    throws ConfigurationException;

  /**
   * Returns the child node with the specified name. If no such child node
   * exists, an empty configuration will be returned.
   *
   * @param child The name of the child node.
   * @return the child node with the specified name.
   */
  IConfiguration getChild(String child);

  /**
   * Returns the specified child node.<p>
   *
   *
   *
   * @param child The name of the child node.
   * @param createNew If <code>true</code> , an empty <code>IConfiguration
   *      </code> node will be created and returned if the specified child does
   *      not exist. If <code>false</code> , <code>null</code> will be returned
   *      if the specified child doesn't exist.
   * @return <{IConfiguration}>
   */
  IConfiguration getChild(String child, boolean createNew);

  /**
   * Returns an array of all child nodes. If the node has no children the array
   * is empty.
   *
   * @return an array of all child nodes.
   */
  IConfiguration[] getChildren();

  /**
   * Returns an array of all child nodes with the specified name. If the node
   * has no children with the specified name the array is empty.
   *
   * @param name TBD: Description of the incoming method parameter
   * @return an array of all child nodes.
   */
  IConfiguration[] getChildren(String name);

  /**
   * Returns an array of all attribute names. The order of attributes is the
   * array is undefinded.
   *
   * @return an array of all attribute names.
   */
  String[] getAttributeNames();

  /**
   * Returns the string value of the specified attribute.
   *
   * @param name The name of the attribute.
   * @return the string value of the specified attribute.
   * @exception ConfigurationException If the attribute does not exist.
   */
  String getAttribute(String name)
    throws ConfigurationException;

  /**
   * Returns the integer value of the specified attribute.
   *
   * @param paramName TBD: Description of the incoming method parameter
   * @return the integer value of the specified attribute.
   * @exception ConfigurationException If the attribute does not exist or if the
   *      value is not an integer.
   */
  int getAttributeAsInteger(String paramName)
    throws ConfigurationException;

  /**
   * Returns the long value of the specified attribute.
   *
   * @param name The name of the attribute.
   * @return the long value of the specified attribute.
   * @exception ConfigurationException If the attribute does not exist or if the
   *      value is not a long.
   */
  long getAttributeAsLong(String name)
    throws ConfigurationException;

  /**
   * Returns the float value of the specified attribute.
   *
   * @param paramName TBD: Description of the incoming method parameter
   * @return the float value of the specified attribute.
   * @exception ConfigurationException If the attribute does not exist or if the
   *      value is not a float.
   */
  float getAttributeAsFloat(String paramName)
    throws ConfigurationException;

  /**
   * Returns the boolean value of the specified attribute.
   *
   * @param paramName TBD: Description of the incoming method parameter
   * @return the boolean value of the specified attribute.
   * @exception ConfigurationException If the attribute does not exist or if the
   *      value is not a boolean.
   */
  boolean getAttributeAsBoolean(String paramName)
    throws ConfigurationException;

  /**
   * Returns the string value of this node.
   *
   * @return the string value of this node.
   * @exception ConfigurationException
   */
  String getValue()
    throws ConfigurationException;

  /**
   * Returns the integer value of this node.
   *
   * @return the integer value of this node.
   * @exception ConfigurationException If the value is not an integer.
   */
  int getValueAsInteger()
    throws ConfigurationException;

  /**
   * Returns the float value of this node.
   *
   * @return the float value of this node.
   * @exception ConfigurationException If the value is not an float.
   */
  float getValueAsFloat()
    throws ConfigurationException;

  /**
   * Returns the boolean value of this node.
   *
   * @return the boolean value of this node.
   * @exception ConfigurationException If the value is not an boolean.
   */
  boolean getValueAsBoolean()
    throws ConfigurationException;

  /**
   * Returns the long value of this node.
   *
   * @return the long value of this node.
   * @exception ConfigurationException If the value is not an long.
   */
  long getValueAsLong()
    throws ConfigurationException;

  /**
   * Returns the string value of this node.
   *
   * @param defaultValue The default for the returned value if the node has no
   *      value.
   * @return the integer value of this node.
   */
  String getValue(String defaultValue);

  /**
   * Returns the integer value of this node.
   *
   * @param defaultValue The default for the returned value if the node has no
   *      value.
   * @return the integer value of this node.
   */
  int getValueAsInteger(int defaultValue);

  /**
   * Returns the long value of this node.
   *
   * @param defaultValue The default for the returned value if the node has no
   *      value.
   * @return the long value of this node.
   */
  long getValueAsLong(long defaultValue);

  /**
   * Returns the float value of this node.
   *
   * @param defaultValue The default for the returned value if the node has no
   *      value.
   * @return the float value of this node.
   */
  float getValueAsFloat(float defaultValue);

  /**
   * Returns the boolean value of this node.
   *
   * @param defaultValue The default for the returned value if the node has no
   *      value.
   * @return the boolean value of this node.
   */
  boolean getValueAsBoolean(boolean defaultValue);

  /**
   * Returns the string value of the specified attribute.
   *
   * @param defaultValue The default for the returned value if the attribute
   *      does not exist.
   * @param name The name of the attribute.
   * @return the string value of the specified attribute.
   */
  String getAttribute(String name, String defaultValue);

  /**
   * Returns the integer value of the specified attribute.
   *
   * @param defaultValue The default for the returned value if the attribute
   *      does not exist.
   * @param name The name of the attribute.
   * @return the integer value of the specified attribute.
   */
  int getAttributeAsInteger(String name, int defaultValue);

  /**
   * Returns the long value of the specified attribute.
   *
   * @param defaultValue The default for the returned value if the attribute
   *      does not exist.
   * @param name The name of the attribute.
   * @return the long value of the specified attribute.
   */
  long getAttributeAsLong(String name, long defaultValue);

  /**
   * Returns the float value of the specified attribute.
   *
   * @param defaultValue The default for the returned value if the attribute
   *      does not exist.
   * @param name The name of the attribute.
   * @return the float value of the specified attribute.
   */
  float getAttributeAsFloat(String name, float defaultValue);

  /**
   * Returns the boolean value of the specified attribute.
   *
   * @param defaultValue The default for the returned value if the attribute
   *      does not exist.
   * @param name The name of the attribute.
   * @return the boolean value of the specified attribute.
   */
  boolean getAttributeAsBoolean(String name, boolean defaultValue);
}
