/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/configuration/DefaultConfiguration.java#5 $
 */

package com.sapportals.wcm.crt.configuration;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This is the default implementation of a configuration node. <p>
 *
 * Copyright (c) SAP AG 2001-2002 <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/configuration/DefaultConfiguration.java#3
 *      $
 */
public class DefaultConfiguration extends AbstractConfiguration {

  protected final static IConfiguration[] NO_CHILDREN = new IConfiguration[0];
  protected final static String[] NO_ATTRIBUTES = new String[0];

  private final String name;
  private final String location;
  private final String namespace;
  private final String prefix;
  private Map attributes;
  private List children;
  private String value;
  private boolean readOnly;

  /**
   * Creates a new instance.
   *
   * @param name The name of the configuration node.
   */
  public DefaultConfiguration(final String name) {
    this(name, "", "", "");
  }

  public DefaultConfiguration(final String name, final String location, Map attributes) {
    this(name, location, new HashMap(attributes));
  }
  
  /**
   * @deprecated as of NW04. Use DefaultConfiguration(final String name, final String location, Map attributes) 
   * instead.
   */
  public DefaultConfiguration(final String name, final String location, HashMap attributes) {
    this(name, location);
    this.attributes = attributes;
  }

  /**
   * Creates a new instance.
   *
   * @param name The name of the configuration node.
   * @param location A client specific location string, examples: line number
   *      (for file-based persistence), identifier, URI, database table, ...
   */
  public DefaultConfiguration(final String name, final String location) {
    this(name, location, "", "");
  }

  /**
   * Creates a new instance.
   *
   * @param name The name of the configuration node.
   * @param location A client specific location string, examples: line number
   *      (for file-based persistence), identifier, URI, database table, ...
   * @param namespace A namespace string
   * @param prefix
   */
  public DefaultConfiguration(final String name, final String location, final String namespace, final String prefix) {
    if (name == null) {
      throw new IllegalArgumentException("name is null");
    }
    this.name = name;
    this.location = location;
    this.namespace = namespace;
    this.prefix = prefix;
  }

  public String getName() {
    return this.name;
  }

  public String getNamespace()
    throws ConfigurationException {
    if (null != this.namespace) {
      return this.namespace;
    }
    else {
      throw this.createConfigurationException("No namespace");
    }
  }

  public String getLocation() {
    return this.location;
  }

  public String getValue()
    throws ConfigurationException {
    if (null != this.value) {
      return this.value;
    }
    else {
      throw this.createConfigurationException("Missing node value");
    }
  }

  public String[] getAttributeNames() {
    if (null == this.attributes) {
      return NO_ATTRIBUTES;
    }
    else {
      return (String[])this.attributes.keySet().toArray(NO_ATTRIBUTES);
    }
  }

  public IConfiguration[] getChildren() {
    if (null == this.children) {
      return NO_CHILDREN;
    }
    else {
      return (IConfiguration[])this.children.toArray(NO_CHILDREN);
    }
  }

  public String getAttribute(final String name)
    throws ConfigurationException {
    if (null != this.attributes) {
      Object result = this.attributes.get(name);
      if (null != result) {
        return result.toString();
      }
    }
    throw this.createConfigurationException("Missing attribute: " + name);
  }

  public IConfiguration getChild(final String name, final boolean createNew) {
    if (null != this.children) {
      final int size = this.children.size();
      for (int i = 0; i < size; i++) {
        final IConfiguration configuration = (IConfiguration)this.children.get(i);
        if (name.equals(configuration.getName())) {
          return configuration;
        }
      }
    }
    if (createNew) {
      return new DefaultConfiguration(name, "-");
    }
    else {
      return null;
    }
  }

  public IConfiguration[] getChildren(final String name) {
    if (null == this.children) {
      return new IConfiguration[0];
    }
    else {
      final ArrayList children = new ArrayList();
      final int size = this.children.size();
      for (int i = 0; i < size; i++) {
        final IConfiguration configuration = (IConfiguration)this.children.get(i);
        if (name.equals(configuration.getName())) {
          children.add(configuration);
        }
      }
      return (IConfiguration[])children.toArray(new IConfiguration[children.size()]);
    }
  }

  public void setValue(final String value) {
    this.checkWriteable();
    this.value = value;
  }

  public void setAttribute(final String name, final String value) {
    this.checkWriteable();
    if (null == this.attributes) {
      this.attributes = new HashMap(2);
    }
    this.attributes.put(name, value);
  }

  /**
   * Adds a child configuration node to this node.
   *
   * @param childConfiguration child to be added
   */
  public void addChild(final IConfiguration childConfiguration) {
    this.checkWriteable();
    if (null == this.children) {
      this.children = new ArrayList();
    }
    this.children.add(childConfiguration);
  }

  /**
   * Removes the child configuration node from this node.
   *
   * @param childConfiguration The child node to remove.
   */
  public void removeChild(final IConfiguration childConfiguration) {
    this.checkWriteable();
    if (null == this.children) {
      return;
    }
    this.children.remove(childConfiguration);
  }

  /**
   * Returns the number of child nodes.
   *
   * @return the number of child nodes.
   */
  public int getChildCount() {
    if (null == this.children) {
      return 0;
    }
    return this.children.size();
  }

  /**
   * Sets the read-only flag for this node. The add/remove methods will throw an
   * IllegalStateException.
   */
  public void makeReadOnly() {
    this.readOnly = true;
  }

  private final static StringBuffer indent(StringBuffer buffer, int in) {
    for (int i = 0; i < in; i++) {
      buffer.append("    ");
    }
    return buffer;
  }

  protected StringBuffer dump(StringBuffer buffer, int in) {
    indent(buffer, in++);
    buffer.append("<configuration ");
    buffer.append("name='").append(this.name).append("' ");
    buffer.append("value='").append(this.value).append("' ");
    buffer.append("location='").append(this.location).append("' ");
    buffer.append("namespace='").append(this.namespace).append("' ");
    buffer.append("prefix='").append(this.prefix).append("' ");
    buffer.append("readonly='").append(this.readOnly).append("'>\n");
    String[] attribute = getAttributeNames();
    for (int i = 0; i < attribute.length; i++) {
      String name = attribute[i];
      indent(buffer, in);
      buffer.append("<attribute name='").append(name).append("' value='");
      buffer.append(this.attributes.get(name)).append("' />\n");
    }
    IConfiguration[] configuration = getChildren();
    for (int i = 0; i < configuration.length; i++) {
      IConfiguration conf = configuration[i];
      if (conf instanceof DefaultConfiguration) {
        ((DefaultConfiguration)conf).dump(buffer, in);
      }
      else {
        buffer.append(conf.toString());
      }
      buffer.append('\n');
    }
    indent(buffer, --in);
    return buffer.append("</configuration>");
  }

  public String toString() {
    return this.dump(new StringBuffer(), 0).toString();
  }

  protected String getPrefix()
    throws ConfigurationException {
    if (null != this.prefix) {
      return this.prefix;
    }
    else {
      throw this.createConfigurationException("No prefix");
    }
  }

  protected final void checkWriteable()
    throws IllegalStateException {
    if (this.readOnly) {
      throw new IllegalStateException("Configuration node is read only and can not be modified: " + this.getName());
    }
  }

  private ConfigurationException createConfigurationException(String message) {
    return new ConfigurationException(message + ", configuration node = " + this.getName() +
      ", location = " + this.getLocation());
  }
}
