/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/configuration/AbstractConfiguration.java#3 $
 */

package com.sapportals.wcm.crt.configuration;

/**
 * This is an abstract <code>IConfiguration</code> implementation that deals
 * with methods that can be abstracted away from underlying implementations. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/configuration/AbstractConfiguration.java#3
 *      $
 */
public abstract class AbstractConfiguration implements IConfiguration {
  /**
   * Returns the prefix of the namespace. This is only used as a serialization
   * hint, therefore is not part of the client API. It should be included in all
   * Configuration implementations though.
   *
   * @return A non-null String (defaults to "")
   * @throws ConfigurationException if no prefix was defined (prefix is <code>
   *      null</code> .
   */
  protected abstract String getPrefix()
    throws ConfigurationException;

  /**
   * Returns the value of the configuration element as an <code>int</code> .
   * Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and binary
   * numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public int getValueAsInteger()
    throws ConfigurationException {
    final String value = getValue();
    try {
      if (value.startsWith("0x")) {
        return Integer.parseInt(value.substring(2), 16);
      }
      else if (value.startsWith("0o")) {
        return Integer.parseInt(value.substring(2), 8);
      }
      else if (value.startsWith("0b")) {
        return Integer.parseInt(value.substring(2), 2);
      }
      else {
        return Integer.parseInt(value);
      }
    }
    catch (final Exception nfe) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value +
        "\" as an integer in the configuration element \"" + getName() + "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the configuration element as an <code>int</code> .
   * Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and binary
   * numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public int getValueAsInteger(final int defaultValue) {
    try {
      return getValueAsInteger();
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the configuration element as a <code>long</code> .
   * Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and binary
   * numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public long getValueAsLong()
    throws ConfigurationException {
    final String value = getValue();
    try {
      if (value.startsWith("0x")) {
        return Long.parseLong(value.substring(2), 16);
      }
      else if (value.startsWith("0o")) {
        return Long.parseLong(value.substring(2), 8);
      }
      else if (value.startsWith("0b")) {
        return Long.parseLong(value.substring(2), 2);
      }
      else {
        return Integer.parseInt(value);
      }
    }
    catch (final Exception nfe) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as a long in the configuration element \"" +
        getName() + "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the configuration element as a <code>long</code> .
   * Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and binary
   * numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public long getValueAsLong(final long defaultValue) {
    try {
      return getValueAsLong();
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the configuration element as a <code>float</code> .
   *
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public float getValueAsFloat()
    throws ConfigurationException {
    final String value = getValue();
    try {
      return Float.parseFloat(value);
    }
    catch (final Exception nfe) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as a float in the configuration element \"" +
        getName() + "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the configuration element as a <code>float</code> .
   *
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public float getValueAsFloat(final float defaultValue) {
    try {
      return getValueAsFloat();
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return (defaultValue);
    }
  }

  /**
   * Returns the value of the configuration element as a <code>boolean</code> .
   *
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public boolean getValueAsBoolean()
    throws ConfigurationException {
    final String value = getValue();
    if (value.equals("true")) {
      return true;
    }
    else if (value.equals("false")) {
      return false;
    }
    else {
      throw new ConfigurationException("Cannot parse the value \"" + value +
        "\" as a boolean in the configuration element \"" + getName() + "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the configuration element as a <code>boolean</code> .
   *
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public boolean getValueAsBoolean(final boolean defaultValue) {
    try {
      return getValueAsBoolean();
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the configuration element as a <code>String</code> .
   *
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public String getValue(final String defaultValue) {
    try {
      return getValue();
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the attribute specified by its name as an <code>int
   * </code>. Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and
   * binary numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param name the name of the attribute
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public int getAttributeAsInteger(final String name)
    throws ConfigurationException {
    final String value = getAttribute(name);
    try {
      if (value.startsWith("0x")) {
        return Integer.parseInt(value.substring(2), 16);
      }
      else if (value.startsWith("0o")) {
        return Integer.parseInt(value.substring(2), 8);
      }
      else if (value.startsWith("0b")) {
        return Integer.parseInt(value.substring(2), 2);
      }
      else {
        return Integer.parseInt(value);
      }
    }
    catch (final Exception nfe) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as an integer in the attribute \"" + name +
        "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the attribute specified by its name as an <code>int
   * </code>. Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and
   * binary numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param name the name of the attribute
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public int getAttributeAsInteger(final String name, final int defaultValue) {
    try {
      return getAttributeAsInteger(name);
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>long
   * </code>. Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and
   * binary numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param name the name of the attribute
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public long getAttributeAsLong(final String name)
    throws ConfigurationException {
    final String value = getAttribute(name);
    try {
      if (value.startsWith("0x")) {
        return Long.parseLong(value.substring(2), 16);
      }
      else if (value.startsWith("0o")) {
        return Long.parseLong(value.substring(2), 8);
      }
      else if (value.startsWith("0b")) {
        return Long.parseLong(value.substring(2), 2);
      }
      else {
        return Long.parseLong(value);
      }
    }
    catch (final Exception nfe) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as a long in the attribute \"" + name +
        "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>long
   * </code>. Hexadecimal numbers begin with 0x, Octal numbers begin with 0o and
   * binary numbers begin with 0b, all other values are assumed to be decimal.
   *
   * @param name the name of the attribute
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public long getAttributeAsLong(final String name, final long defaultValue) {
    try {
      return getAttributeAsLong(name);
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>float
   * </code>.
   *
   * @param name the name of the attribute
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public float getAttributeAsFloat(final String name)
    throws ConfigurationException {
    final String value = getAttribute(name);
    try {
      return Float.parseFloat(value);
    }
    catch (final Exception e) {
      //$JL-EXC$ 
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as a float in the attribute \"" + name +
        "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>float
   * </code>.
   *
   * @param name the name of the attribute
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public float getAttributeAsFloat(final String name, final float defaultValue) {
    try {
      return getAttributeAsFloat(name);
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>boolean
   * </code>.
   *
   * @param name the name of the attribute
   * @return the value
   * @exception ConfigurationException if an error occurs
   */
  public boolean getAttributeAsBoolean(final String name)
    throws ConfigurationException {
    final String value = getAttribute(name);
    if (value.equalsIgnoreCase("true")) {
      return true;
    }
    else if (value.equalsIgnoreCase("false")) {
      return false;
    }
    else {
      throw new ConfigurationException("Cannot parse the value \"" + value + "\" as a boolean in the attribute \"" + name +
        "\" at " + getLocation());
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>boolean
   * </code>.
   *
   * @param name the name of the attribute
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public boolean getAttributeAsBoolean(final String name, final boolean defaultValue) {
    try {
      return getAttributeAsBoolean(name);
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Returns the value of the attribute specified by its name as a <code>String
   * </code>.
   *
   * @param name the name of the attribute
   * @param defaultValue the default value to return if value malformed or empty
   * @return the value
   */
  public String getAttribute(final String name, final String defaultValue) {
    try {
      return getAttribute(name);
    }
    catch (final ConfigurationException ce) {
      //$JL-EXC$ 
      return defaultValue;
    }
  }

  /**
   * Return the first <code>Configuration</code> object child of this associated
   * with the given name.
   *
   * @param name the name of the child
   * @return the child Configuration
   */
  public IConfiguration getChild(final String name) {
    return getChild(name, false);
  }

}
