/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/component/ComponentStateHandler.java#3 $
 */

package com.sapportals.wcm.crt.component;

import com.sapportals.wcm.crt.configuration.ConfigurationException;

import java.util.Date;

/**
 * This class must be used for state and life-cycle handling in component
 * implementations. All of the relevant pre and post methods MUST be called. <p>
 *
 * Example: <pre>
 * public class TestComponent implements IComponent, IConfigurable, IStartable, ILifecycleInfo {
 *
 *   private final ComponentStateHandler h;
 *
 *   public TestComponent() {
 *     this.h = new ComponentStateHandler(TestComponent.class);
 *   }
 *
 *   public ComponentState getState() {
 *     return this.h.getState();
 *   }
 *
 *   // ... other ILifecycleInfo methods ...
 *
 *   public configure(IConfiguration config) throws ConfigurationException {
 *     this.h.preConfigure();
 *     try {
 *       // ...
 *       throw new ConfigurationException("missing parameter");
 *     }
 *     catch (ConfigurationException ex) {
 *       throw this.h.postConfigure(ex);
 *     }
 *     catch (Exception ex) {
 *       throw new ConfigurationException(this.h.postConfigure(ex));
 *     }
 *     this.h.postConfigure();
 *   }
 *
 *   public start() throws StartupException {
 *     this.h.preStart();
 *     try {
 *       // ...
 *       throw new StartupException("failed to start");
 *     }
 *     catch (StartupException ex) {
 *       throw this.h.postStart(ex);
 *     }
 *     catch (Exception ex) {
 *       throw new StartupException(this.h.postStartUp(ex));
 *     }
 *     this.h.postStart();
 *   }
 *
 *   public someServiceMethod() throws ServiceException {
 *    try {
 *       if (!this.h.preService()) {
 *        throw new ServiceException("Service not available");
 *       }
 *       // ...
 *     }
 *     finally {
 *       this.h.postService();
 *     }
 *   }
 * }
 * </pre>
 *
 * <p>
 *      Copyright (c) SAP AG 2001-2002
 * @author Jens Kaiser
 * @author Markus Breitenfelder
 * @see ComponentState
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/component/ComponentStateHandler.java#3
 *      $
 */
public class ComponentStateHandler implements ILifecycleInfo {

  protected final Class cc;
  protected final boolean isContextualizable;
  protected final boolean isConfigurable;
  protected final boolean isStartable;
  protected final boolean isSuspendable;
  protected final boolean isReconfigurable;
  protected final boolean isThreadSafe;

  protected boolean contextualized;
  protected boolean configured;
  protected boolean started;
  protected boolean stopped;
  protected boolean suspended;

  protected ComponentState state;

  protected ConfigurationException configException;
  protected StartupException startupException;

  protected Date creationDate;
  protected Date reconfigDate;
  protected Date nextAutoRestartDate;

  protected long startTime;
  protected long configurationTime;

  protected transient int numWorkers;
  protected transient Thread suspender;


  /**
   * Constructs a new state handler instance.
   *
   * @param component the object of the component
   */
  public ComponentStateHandler(Object component) {
    this.creationDate = new Date();

    this.cc = component.getClass();
    this.isContextualizable = component instanceof IContextualizable;
    this.isConfigurable = component instanceof IConfigurable;
    this.isStartable = component instanceof IStartable;
    this.isSuspendable = component instanceof ISuspendable;
    this.isReconfigurable = component instanceof IReconfigurable;
    this.isThreadSafe = component instanceof IThreadSafe;
    if (!this.isContextualizable && !this.isStartable) {
      this.state = ComponentState.RUNNING;
    }
    else {
      this.state = ComponentState.NOT_INIT;
    }
  }


  public Date getCreationDate() {
    return this.creationDate;
  }


  public ComponentState getState() {
    return this.state;
  }


  public void preContextualize() {
    if (!this.isContextualizable) {
      throw new IllegalStateException(this.cc.getName() + " is not contextualizable");
    }
    if (this.contextualized) {
      throw new IllegalStateException("already contextualized");
    }
  }

  public ContextException postContextualize(ContextException x)
    throws ContextException {
    this.state = ComponentState.INIT_ERROR;
    this.contextualized = true;
    throw x;
  }


  public void postContextualize() {
    this.contextualized = true;
    if (!this.isStartable) {
      this.state = ComponentState.RUNNING;
    }
  }


  public void preConfigure() {
    if (!this.isConfigurable) {
      throw new IllegalStateException(this.cc.getName() + " is not configurable");
    }
    if (this.isContextualizable && !this.contextualized) {
      throw new IllegalStateException("not contextualized");
    }
    if (this.configured) {
      throw new IllegalStateException("already configured");
    }
    if (this.stopped) {
      throw new IllegalStateException("already shut down");
    }
    this.configurationTime = System.currentTimeMillis();
  }


  public ConfigurationException postConfigure(ConfigurationException x) {
    this.configException = x;
    this.state = ComponentState.INIT_ERROR;
    this.configured = true;
    this.configurationTime = System.currentTimeMillis() - this.configurationTime;
    return x;
  }


  public void postConfigure() {
    this.configured = true;
    this.configException = null;
    if (!this.isStartable) {
      this.state = ComponentState.RUNNING;
    }
    this.configurationTime = System.currentTimeMillis() - this.configurationTime;
  }


  public void preStart() {
    if (!this.isStartable) {
      throw new IllegalStateException(this.cc.getName() + " is not startable");
    }
    if (this.isContextualizable && !this.contextualized) {
      throw new IllegalStateException("not contextualized");
    }
    if (this.isConfigurable && !this.configured) {
      throw new IllegalStateException("not configured");
    }
    if (this.configException != null) {
      throw new IllegalStateException("configure failed");
    }
    if (this.started) {
      throw new IllegalStateException("already started");
    }
    if (this.stopped) {
      throw new IllegalStateException("already shut down");
    }
    this.startTime = System.currentTimeMillis();
  }


  public StartupException postStart(StartupException x) {
    this.startupException = x;
    this.state = ComponentState.INIT_ERROR;
    if (x instanceof AutoRestartException) {
      this.nextAutoRestartDate = new Date(System.currentTimeMillis() + ((AutoRestartException)x).getDelay());
      this.started = false;
      // Can retry startup !
    }
    else {
      this.nextAutoRestartDate = null;
      this.started = true;
    }
    return x;
  }


  public StartupException postStart(ConfigurationException x) {
    this.configException = x;
    this.state = ComponentState.INIT_ERROR;
    this.started = true;
    this.nextAutoRestartDate = null;
    this.startTime = System.currentTimeMillis() - this.startTime;
    return new StartupException("Configuration exception: " + x.getMessage(), x);
  }


  public void postStart() {
    this.started = true;
    this.state = ComponentState.RUNNING;
    this.startTime = System.currentTimeMillis() - this.startTime;
    this.nextAutoRestartDate = null;
    this.startupException = null;
  }


  public void preStop() {
    if (!this.isStartable) {
      throw new IllegalStateException(this.cc.getName() + " is not startable");
    }
    if (!this.started) {
      throw new IllegalStateException("not started");
    }
    if (this.stopped) {
      throw new IllegalStateException("already shut down");
    }
    startMonitor();
  }


  public void postStop() {
    this.stopped = true;
    this.state = ComponentState.STOPPED;
    stopMonitor();
  }


  public void preSuspend() {
    if (!this.isSuspendable) {
      throw new IllegalStateException(this.cc.getName() + " is not suspendable");
    }
    if (!this.started) {
      throw new IllegalStateException("not started");
    }
    if (this.stopped) {
      throw new IllegalStateException("already shut down");
    }
  }


  public void postSuspend() {
    this.startMonitor();

    this.suspended = true;
    this.state = ComponentState.STOPPED;
  }


  public void preResume()
    throws IllegalStateException {
    if (!this.isSuspendable) {
      throw new IllegalStateException(this.cc.getName() + " is not suspendable");
    }
    if (!this.suspended) {
      throw new IllegalStateException("not suspended");
    }
  }


  public void postResume() {
    this.suspended = false;
    this.state = ComponentState.RUNNING;

    this.stopMonitor();
  }


  public void preReconfigure() {
    if (!this.isReconfigurable) {
      throw new IllegalStateException(this.cc.getName() + " is not reconfigurable");
    }
    if (!this.started) {
      throw new IllegalStateException("not started");
    }
    if (this.stopped) {
      throw new IllegalStateException("stopped");
    }
    if (this.isThreadSafe && this.isSuspendable) {
      if (!this.suspended) {
        throw new IllegalStateException("not suspended");
      }
    }
  }


  public ConfigurationException postReconfigure(ConfigurationException x) {
    this.configException = x;
    return x;
  }


  public void postReconfigure() {
    this.configException = null;
    this.reconfigDate = new Date();
  }


  /**
   * Returns true if a component specific service method can be called. The
   * component should throw a dedicated exception if false is returned. <p>
   *
   * This method should be used very carefully to avoid unintended compromise of
   * the service.It should always be balanced with a <code>postService</code> .
   *
   * @return TBD: Description of the outgoing return value
   * @see #postService()
   */
  public boolean preService() {
    this.aquireWorker();
    return ComponentState.RUNNING.equals(this.state);
  }


  /**
   * This method should be balanced with a <code>preService</code> , and should
   * occur in a finally statement so that the balance is guaranteed. The
   * following is an example. <pre>
   * stateHandler.preService();
   * try {
   *   // do something
   * }
   * finally {
   *   stateHandler.postService();
   * }
   * </pre>
   *
   * @see #preService()
   */
  public void postService() {
    this.releaseWorker();
  }


  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public ConfigurationException getLastConfigurationException() {
    return this.configException;
  }


  public StartupException getStartupException() {
    return this.startupException;
  }


  public Date getLastReconfigurationDate() {
    return this.reconfigDate;
  }


  public Date getNextAutoRestartDate() {
    return this.nextAutoRestartDate;
  }


  // --------------------------------------------------------------------------
  // profiling
  // ---------------------------------------------------------------------------


  public long getStartTime() {
    return this.startTime;
  }


  public long getConfigurationTime() {
    return this.configurationTime;
  }


  // --------------------------------------------------------------------------
  // monitor workers
  // ---------------------------------------------------------------------------

  protected synchronized void startMonitor() {
    try {
      while ((this.numWorkers > 0) || (this.suspender != null)) {
        if (this.suspender == Thread.currentThread()) {
          return;
        }
        wait();
      }
      this.suspender = Thread.currentThread();
    }
    catch (InterruptedException e) {
      //$JL-EXC$
      throw new IllegalThreadStateException("Interrupted attempt to aquire suspension lock");
    }
  }


  protected synchronized void stopMonitor() {
    this.suspender = null;
    notifyAll();
  }


  protected synchronized void aquireWorker() {
    try {
      while (this.suspender != null) {
        if (this.suspender == Thread.currentThread()) {
          // suspender has full access.... may try to acquire
          // lock in notification
          return;
        }
        wait();
      }
      this.numWorkers++;
    }
    catch (InterruptedException e) {
      //$JL-EXC$ 
      throw new IllegalThreadStateException("Interrupted attempt to aquire suspension lock");
    }
  }


  protected synchronized void releaseWorker() {
    if (this.suspender == Thread.currentThread()) {
      // suspender has full access.... may try to acquire
      // lock in notification
      return;
    }
    if (this.numWorkers <= 0) {
      throw new IllegalStateException("Post service failure");
    }
    this.numWorkers--;
    notify();
  }

}
