/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/CrtSystem.java#4 $
 */

package com.sapportals.wcm.crt;

import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sapportals.wcm.crt.component.ComponentException;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.crt.component.IComponentListener;
import com.sapportals.wcm.crt.component.IComponentManager;
import com.sapportals.wcm.crt.configuration.IConfigurationProvider;

/**
 * Provides access to the component runtime system. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/CrtSystem.java#3
 *      $
 */
public abstract class CrtSystem {

  private final static String CRTSYSTEM_IMPL = "com.sapportals.wcm.crt.CrtSystemImpl";
	private static final Location LOC;

  private static CrtSystem instance;
	
  static {
		LOC = Location.getLocation(CrtSystem.class);
    try {
      CrtSystem.instance = (CrtSystem)Class.forName(CRTSYSTEM_IMPL).newInstance();
    }
    catch (Exception ex) {
      //$JL-EXC$
      System.err.println("Exception loading class " + CRTSYSTEM_IMPL + ":" + ex.getMessage());
      
      // useless, but it satisfies JLin
      LOC.debugT("Exception loading class " + CRTSYSTEM_IMPL + ":" + ex.getMessage());
    }
  }


  /**
   * Returns a reference to the component manager for the specified client
   * identifier, or <code>null</code> if the client does not exist.
   *
   * @param clientID The identifier of a CRT client.
   * @return a reference to the component manager or <code>null</code> .
   */
  public abstract IComponentManager getComponentManager(String clientID);

  /**
   * Creates a component manager for a new CRT client.
   *
   * @param clientID An identifier for the client.
   * @param configProvider A configuration provider instance
   * @param listener A component listener implementation, can be <code>null
   *      </code>.
   * @param loader The class loader for the application.
   * @param category The logger category to use for all components, can be
   *      <code>null</code> . The CRT will also use this logger to log all
   *      exceptions that occure during configration and initialization of
   *      components.
   * @exception CascadingException If some arguments are invalid or the creation
   *      of the manager fails.
   */
  public abstract void createComponentManager(String clientID, IConfigurationProvider configProvider,
    IComponentListener listener, ClassLoader loader, Category category)
    throws CascadingException;

  
  /**
   * Returns the component instance for the specified URI or <code>null</code>
   * if the component does not exist.
   *
   * @param uri The component URI to look for. <p>
   *
   *      The caller must subseqently call <code>releaseComponent()</code> when
   *      it is finished using the component. This is optional for thread-safe
   *      components.
   * @return the component instance or <code>null</code> .
   * @exception ComponentException If the component could not be instanciated.
   */
  public abstract IComponent lookupComponentByUri(ComponentUri uri)
    throws ComponentException;

  /**
   * Releases the component. This method should be called when the caller is
   * finished using the component. It must be called for all components that do
   * not implement the {@link com.sapportals.wcm.crt.component.IThreadSafe} interface - it is optional for
   * thread-safe components.
   *
   * @param component A reference to the component to release.
   */
  public abstract void releaseComponent(IComponent component);

  /**
   * This method can optionally be called after the creation of a new component
   * manager. If the initial configuration contains components that implement
   * the {@link com.sapportals.wcm.crt.component.IAutoStartable} interface these components will be started. If
   * no such components exist this method will do nothing. <p>
   *
   * This method must be called only once. It must be called before any lookup
   * methods are called.
   *
   * @param clientID The client ID
   */
  public abstract void startUpComponentManager(String clientID);

  /**
   * This method can optionally be called to shutdown all components of a CRT
   * client. The method must be called only once. It is not possible to restart
   * a component manager after shutdown.
   *
   * @param clientID The client ID
   */
  public abstract void shutDownComponentManager(String clientID);

  /**
   * Returns an instance of this class.
   *
   * @return instance
   */
  public static CrtSystem getInstance() {
    return CrtSystem.instance;
  }
  
  /**
   * Returns true if the manager startup is in progress, false otherwise.
   * @param clientID The client ID.
   * @return true if the manager startup is in progress, false otherwise.
   */
  public abstract boolean managerStartUpIsRunning(String clientID);
}
