/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/CrtClassLoaderRegistry.java#5 $
 */

package com.sapportals.wcm.crt;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;

/**
 * Provides a registry for class loaders. 
 * <p>
 *
 * {@link #getClassLoader() getClassLoader} provides a class loader that
 * aggregates all registered class loaders to find and load classes and
 * resources. 
 * <p>
 *
 * Registered class loaders are chained creating one virtual class loader. From
 * the perspective of the JDK, it's all one classloader. Internally, the class
 * loader chain searches like a classpath. <p>
 *
 * The following code shows an example how to use this registry: 
 * <pre>
 *    // add your personal class loader (this should be done only once)
 *    CrtClassLoaderRegistry.addClassLoader( MyPrettyCoolClass.class.getClassLoader() );
 *    ...
 *    // load another class with this class loader
 *    MyOtherClass myOtherInstance = null;
 *    try {
 *      Class myOtherClass = CrtClassLoaderRegistry.forName( "com.company.prg.MyOtherClass" );
 *      myOtherInstance = (MyOtherClass)myOtherClass.newInstance();
 *    }
 *    catch( .... x ) {
 *      ....
 *    }
 * 
 * </pre> 
 * <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/CrtClassLoaderRegistry.java#4
 *      $
 */
public final class CrtClassLoaderRegistry extends ClassLoader {

  private final static Location TRACE = Location.getLocation(com.sapportals.wcm.crt.CrtClassLoaderRegistry.class);
  private final static Enumeration EMPTY_ENUM = (new Vector()).elements();

	private final static Category LOG = Category.getCategory(Category.APPLICATIONS,"KMC/CRT");

  private static volatile CrtClassLoaderRegistry topLoader = new CrtClassLoaderRegistry();

  private static volatile int counter = 0;

  private final static HashMap name2loader = new HashMap();
  private final static HashMap appId2Loader = new HashMap();
  private final static ArrayList noHintLoaders = new ArrayList();
  // loaders without hints
  private final static HashMap hintMap = new HashMap();
  // maps from package to an array list of loaders

  private final static ArrayList classLoaderListener = new ArrayList();

  
  private CrtClassLoaderRegistry() {
    super(CrtClassLoaderRegistry.class.getClassLoader());
  }

  private Iterator getLinkedLoaders(String name) {
    // iterator over all hint loaders concatenated with all 
    // loaders without hints
    ArrayList result = new ArrayList();
    ClassLoader l = (ClassLoader)name2loader.get(name);
    if (l != null) {
      if (TRACE.beDebug()) {
        TRACE.debugT("getLoaders", "found loader directly");
      }
      result.add(l);
    }

    // check, if it is a resource path
    int resIndex = name.lastIndexOf('/');
    int index = 0;

    if (resIndex > 0) {
      // it is a resource path
      index = resIndex;
      name = name.replace('/', '.'); // create key
    } else {
      // it is a class name
      index = name.lastIndexOf('.');
    }
    if (index > 0) {
      // valid package
      String packageName = name.substring(0, index);
      ArrayList hintLoaders = (ArrayList)hintMap.get(packageName);
      if (hintLoaders != null) {
        result.addAll(hintLoaders);
      }
    }

    result.addAll(noHintLoaders);
    return result.iterator();
  }

  protected synchronized Class findClass(final String name) throws ClassNotFoundException {
    long time = 0L;
    int misses = 0;

    if (this != topLoader) {
      if (TRACE.beDebug()) {
        TRACE.debugT("findClass", "invalid CRT loader");
      }
      // this loader is no longer valid
      throw new ClassNotFoundException(name + " -- invalid CRT loader");
    }

    try {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis();
      }

      Iterator it = getLinkedLoaders(name);
      while (it.hasNext()) {
        try {
          ClassLoader loader = (ClassLoader)it.next();
          Class clazz = Class.forName(name, false, loader);
          name2loader.put(name, loader);
          return clazz;
        } catch (ClassNotFoundException x) {
          //$JL-EXC$ 
          // try next loader
          misses++;
        } catch(LinkageError x) {
        	if( TRACE.beDebug() ) {
          	TRACE.debugT("findClass",x.getMessage() + ": " + extractCallstack(x));        	 
					}
        	throw x;
        }
      }

      // none of the loader found it
      throw new ClassNotFoundException(name);
    } finally {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis() - time;
        TRACE.debugT("findClass", "findClass(" + name + ") [" + time + " ms] [" + misses + " misses]");
      }
    }
  }

	private static String extractCallstack(Throwable throwable) {
		StringWriter stringWriter = new StringWriter();
		throwable.printStackTrace(new PrintWriter(stringWriter));
		return stringWriter.toString();
	}
	
  protected synchronized Enumeration findResources(final String name) throws IOException {
    long time = 0L;

    if (this != topLoader) {
      if (TRACE.beDebug()) {
        TRACE.debugT("findResources", "invalid loader");
      }
      // this loader is no longer valid
      throw new IOException(name + " -- invalid CRT loader");
    }

    int misses = 0;
    try {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis();
      }

      Iterator it = getLinkedLoaders(name);
      while (it.hasNext()) {
        Enumeration enum = ((ClassLoader)it.next()).getResources(name);
        if (enum != null) {
          if (enum.hasMoreElements()) {
            // not empty
            return enum;
          } else {
            misses++;
          }
        }
      }

      // none of the loaders found anything
      return EMPTY_ENUM;
    } finally {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis() - time;
        TRACE.debugT("findResources(155)", "findResources(" + name + ") [" + time + "ms] [" + misses + " misses]");
      }
    }
  }

  protected synchronized URL findResource(final String name) {
    long time = 0L;

    if (this != topLoader) {
      if (TRACE.beDebug()) {
        TRACE.debugT("findResource", "invalid loader");
      }
      // this loader is no longer valid
      return null;
    }

    int misses = 0;
    try {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis();
      }

      Iterator it = getLinkedLoaders(name);
      while (it.hasNext()) {
        URL url = ((ClassLoader)it.next()).getResource(name);
        if (url != null) {
          return url;
        } else {
          misses++;
        }
      }

      return null; // not found
    } finally {
      if (TRACE.beDebug()) {
        time = System.currentTimeMillis() - time;
        TRACE.debugT("findResource", "findResource(" + name + ") [" + time + "ms] [" + misses + " misses]");
      }
    }
  }

  /**
   * Register a class loader to the component runtime.
   *
   * @param newLoader The new class loader instance to be added.
   */
  public static void addClassLoader(final ClassLoader newLoader) {
    CrtClassLoaderRegistry.addClassLoader(null, newLoader, null);
  }

  /**
   * Register a class loader with an optional id to the component runtime. If a
   * class loader with the same id was already registered, this method replaces
   * the former loader with the <code>newLoader</code> . <p>
   *
   * If <code>newLoader</code> is <code>null</code> the class loader with <code>
   * id</code> is removed. <p>
   *
   * If <code>id</code> is <code>null</code> the class loader is added anonymous
   * and cannot be removed.
   *
   * @param id The optional id of this class loader
   * @param newLoader The new class loader instance to be added.
   */
  public static void addClassLoader(final String id, final ClassLoader newLoader) {
    CrtClassLoaderRegistry.addClassLoader(id, newLoader, null);
  }

  /**
   * Register a class loader with an optional id to the component runtime. If a
   * class loader with the same id was already registered, this method replaces
   * the former loader with the <code>newLoader</code> . <p>
   *
   * This method is meant as a hook for improved class loading speed. Each class
   * loader <code>newLoader</code> registered comes bundled with a set of Java
   * package names that are in this loaders scope.
   *
   * @param id The id of this class loader
   * @param newLoader The new class loader instance to be added
   * @param hints A set of package names in the loader's scope
   */
  public static void addClassLoader(String id, final ClassLoader newLoader, final Set hints) {
    if (id == null || id.length() == 0) {    	
			try {
					throw new Exception();
			} catch (Exception ex) { 
					LOG.warningT(TRACE, "register new classloader with id = null");
			}
			id = "__ generated__" + counter++;
		} else {
			LOG.infoT(TRACE, "register new classloader with id " + id);
		}

    CrtClassLoaderRegistry.registerClassLoader(id, newLoader, hints);
  }

  /**
   * Returns the component runtime default class loader. This class loader uses
   * all registered class loaders to find and load class files and resources.
   *
   * @return The chained class loader
   */
  public static synchronized ClassLoader getClassLoader() {
    return CrtClassLoaderRegistry.topLoader;
  }

  /**
   * Returns the <code>Class</code> object associated with the class or
   * interface with the given string name, using this registry.
   *
   * @param className
   * @return Class object
   * @throws ClassNotFoundException
   */
  public static Class forName(String className) throws ClassNotFoundException {
    return Class.forName(className, true, getClassLoader());
  }

  /**
   * Registers a class loader
   *
   * @param id An optional id of the class loader
   * @param loader A class loader
   * @param hints TBD: Description of the incoming method parameter
   */
  private static synchronized void registerClassLoader(String id, final ClassLoader loader, final Set hints) {
    if (loader == null || loader == topLoader) {
      return; // someone
    }

    boolean replacement = false;

		if (id == null || id.length() == 0) {
			try {
				throw new Exception();
			} catch (Exception ex) { 
				LOG.warningT(TRACE, "registered new classloader with id = null");
			}
		}

    // check for old loader with the same ID
    // if there is one, remove it and remove it from the hint map.
    // anonymous loaders cannot be removed
    ClassLoader oldLoader = (ClassLoader)appId2Loader.put(id, loader);

    // replaced old loader with same id. Doesn't replace anonymous loaders
    if (oldLoader != null) {
      if (oldLoader == loader) {
        // loader is registered a second time. => don't do anything
				LOG.warningT(TRACE, "loader with id " + id + " was already registered - no replacement");
        return;
      }
      replacement = true;

      // remove old Loader from hint set
      Set toRemove = new HashSet();
      Iterator it = hintMap.keySet().iterator();
      while (it.hasNext()) {
        String hintMapKey = (String)it.next();
        ArrayList loaderList = (ArrayList)hintMap.get(hintMapKey);
        int i = 0;
        while (i < loaderList.size()) {
          if (oldLoader == loaderList.get(i)) {
            loaderList.remove(i); // remove old loader
          } else {
            i++; // next element
          }
        }

        if (loaderList.isEmpty()) {
          //  the list is empty now. Remeber for removal from hint map
          toRemove.add(hintMapKey);
        }
      }

      it = toRemove.iterator();
      while (it.hasNext()) {
        hintMap.remove(it.next());
      }

      // check, array of loaders without hints, if it contained the loader
      it = noHintLoaders.iterator();
      while (it.hasNext()) {
        // remove any occurance of the old loader 
        if (it.next() == oldLoader) {
          it.remove();
        }
      }

      // check name 2 loader hash map
      it = name2loader.entrySet().iterator();
      toRemove = new HashSet();
      while (it.hasNext()) {
        Map.Entry entry = (Map.Entry)it.next();
        if (entry.getValue() == oldLoader) {
          toRemove.add(entry.getKey());
        }
      }

      it = toRemove.iterator();
      while (it.hasNext()) {
        name2loader.remove(it.next());
      }

		  LOG.warningT(TRACE, "loader was already registered - replacement : oldLoader = " + oldLoader.toString() + " -> newLoader = " + loader.toString());
      notifiyClassLoaderReplaced(id, loader, oldLoader);
    } else {
      notifyClassLoaderAdded(id, loader);
    }

    // process hints
    if (null != hints && !hints.isEmpty()) {
      //System.out.println("Adding hints for context " + id);

      // register loader for every package it contains
      Iterator i = hints.iterator();
      while (i.hasNext()) {
        String packageName = i.next().toString().replace('/', '.');

        ArrayList loaderList = (ArrayList)hintMap.get(packageName);
        if (loaderList == null) {
          loaderList = new ArrayList();
          hintMap.put(packageName, loaderList);
        }
        loaderList.add(loader);

      }
    } else {
      noHintLoaders.add(loader);
    }

    if (replacement) {
      // we need a new top loader
      topLoader = new CrtClassLoaderRegistry(); // create a new toploader
      if (TRACE.beDebug()) {
        TRACE.debugT("registerClassLoader", "new top loader");
      }
    }
  }

  public static void addClassLoaderRegistryListener(IClassLoaderRegistryListener l) {
		synchronized (classLoaderListener) {
      classLoaderListener.add(l);
		}
  }

  public static void removeClassLoaderRegistryListener(IClassLoaderRegistryListener l) {
		synchronized (classLoaderListener) {
      classLoaderListener.remove(l);
		}
  }

  private static void notifiyClassLoaderReplaced(String id, ClassLoader newLoader, ClassLoader oldLoader) {
    synchronized (classLoaderListener) {
    	ArrayList listeners = new ArrayList(classLoaderListener);
      for (Iterator i = listeners.iterator(); i.hasNext();) {
        ((IClassLoaderRegistryListener)i.next()).classLoaderReplaced(id, newLoader, oldLoader);
      }
    }
  }

  private static void notifyClassLoaderAdded(String id, ClassLoader newLoader) {
    synchronized (classLoaderListener) {
			ArrayList listeners = new ArrayList(classLoaderListener);
      for (Iterator i = listeners.iterator(); i.hasNext();) {
        ((IClassLoaderRegistryListener)i.next()).classLoaderAdded(id, newLoader);
      }
    }
  }
}