/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.crt/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/crt/ComponentUri.java#2 $
 */

package com.sapportals.wcm.crt;

import java.util.*;

/**
 * A component URI (<a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> )
 * is an identifier for a component in the CRT. <p>
 *
 * Scheme: "crt"<br>
 * Authority: The CRT client identifier<br>
 * Path: Slash separated component keys (container hierarchie)<br>
 * Query: Not used<br>
 * <p>
 *
 * This identifier is not relativ to a container like single component keys are
 * but is a unique identifier for all components in the CRT system. The {@link
 * com.sapportals.wcm.repository.runtime.CmSystem#lookupComponentByUri} method can be used to lookup a component
 * instance for a given component URI. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Jens Kaiser
 * @version $Id: //kmgmt/bc.crt/dev/src/_framework/java/api/com/sapportals/wcm/crt/ComponentUri.java#3
 *      $
 */
public final class ComponentUri {

  public final static String PATH_SEPARATOR = "/";
  public final static String SCHEME = "crt";

  private final static String COLON = ":";
  private final static String DOUBLE_SLASH = "//";
  private final static String SCHEME_PREFIX = SCHEME + COLON + DOUBLE_SLASH;

  /**
   * The key this uri represents
   */
  private final String key;

  /**
   * the parent uri
   */
  private ComponentUri parent;

  /**
   * String representation
   */
  private String str;

  /**
   * child uris
   */
  private Map childs;

  /**
   * collection of root uris
   */
  private static Map clients = new HashMap();

  /**
   * Creates a root uri
   *
   * @param clientID TBD: Description of the incoming method parameter
   */
  private ComponentUri(String clientID) {
    if (null == clientID) {
      throw new IllegalArgumentException("clientID is null");
    }
    if (clientID.length() == 0) {
      throw new IllegalArgumentException("clientID is empty");
    }
    if (clientID.indexOf(ComponentUri.PATH_SEPARATOR) != -1) {
      throw new IllegalArgumentException("clientID contains invalid character: " + ComponentUri.PATH_SEPARATOR);
    }

    this.childs = new HashMap();
    this.key = clientID;
  }


  /**
   * Construct a new ComponentUri from an existing one (copy)
   *
   * @param other TBD: Description of the incoming method parameter
   * @param appendKey TBD: Description of the incoming method parameter
   */
  private ComponentUri(ComponentUri other, String appendKey) {
    if (null == appendKey) {
      throw new IllegalArgumentException("appendKey is null");
    }
    if (appendKey.indexOf(ComponentUri.PATH_SEPARATOR) != -1) {
      throw new IllegalArgumentException("Invalid character: " + ComponentUri.PATH_SEPARATOR);
    }

    this.key = appendKey;
    this.parent = other;
    this.childs = new HashMap();
  }


  /**
   * Appends the specified key at the end of the path.
   *
   * @param key The component key to append.
   * @return The new URI with the key added to the path.
   * @exception IllegalArgumentException If the key is invalid (e.g. contains
   *      invalid characters).
   */
  public ComponentUri appendKey(String key)
    throws IllegalArgumentException {
    if (this.childs.containsKey(key)) {
      return (ComponentUri)this.childs.get(key);
    }
    else {
      ComponentUri newUri = new ComponentUri(this, key);
      this.childs.put(key, newUri);
      return newUri;
    }
  }


  /**
   * Returns the parent URI.
   *
   * @return the parent URI.
   */
  public ComponentUri getParent() {
    return this.parent;
  }


  /**
   * Returns true if this is a root URI.
   *
   * @return true if this is a root URI.
   */
  public boolean isRootUri() {
    return null == this.parent;
  }


  /**
   * Returns this URI's key
   *
   * @return this URI's key.
   */
  public String getKey() {
    return this.key;
  }


  /**
   * Returns a child URI with the specified key or <code>null</code> if it does
   * not exist.
   *
   * @param key The key of the child URI.
   * @return a child URI with the specified key or <code>null</code> if it does
   *      not exist.
   */
  public ComponentUri getChildUri(String key) {
    return (ComponentUri)this.childs.get(key);
  }


  /**
   * Removes a child URI with the specified key.
   *
   * @param key The key of the child uri
   * @return The removed URI or <code>null</code> if it does not exist.
   */
  public ComponentUri removeChildUri(String key) {
    return (ComponentUri)this.childs.remove(key);
  }


  /**
   * Returns all child keys of the child URIs of this URI.
   *
   * @return all child keys of the child URIs of this URI.
   */
  public String[] getChildKeys() {
    Set keys = this.childs.keySet();
    return (String[])keys.toArray(new String[keys.size()]);
  }


  public String toString() {
    if (null == this.str) {
      // save the string representation (uri is read-only)
      this.str = ComponentUri.makeString(this, new StringBuffer(256)).toString();
    }
    return this.str;
  }


  private static StringBuffer makeString(ComponentUri uri, StringBuffer buffer) {
    if (null == uri.getParent()) {
      return buffer.append(SCHEME_PREFIX).append(uri.getKey());
    }
    else {
      return ComponentUri.makeString(uri.getParent(), buffer).append(PATH_SEPARATOR).append(uri.getKey());
    }
  }


  public boolean equals(Object other) {
    if (null == other) {
      return false;
    }
    if (this == other) {
      return true;
    }

    if (this.hashCode() != other.hashCode()) {
      return false;
    }

    if (!(other instanceof ComponentUri)) {
      return false;
    }

    ComponentUri otherUri = (ComponentUri)other;
    return this.toString().equals(otherUri.toString());
  }


  public int hashCode() {
    // strings have cached hash codes!
    return this.toString().hashCode();
  }


  /**
   * Returns a new root component URI.
   *
   * @param clientID The CRT client identifier.
   * @return a new root component URI.
   */
  public static ComponentUri getRootUri(String clientID) {
    ComponentUri newUri = (ComponentUri)ComponentUri.clients.get(clientID);
    if (newUri == null) {
      newUri = new ComponentUri(clientID);
      ComponentUri.clients.put(clientID, newUri);
    }
    return newUri;
  }
}
