/**
 * @author Jan Wetzel
 *
 * Allows access to local Smart Sync data. Wraps all Smart Sync APIs
 * and should make data access easier
 */

package mdkInventory1.dataAccess;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Vector;

import mdkInventory1.Constants;
import mdkInventory1.GetProperties;

import com.sap.ip.me.api.persist.core.PersistenceException;
import com.sap.ip.me.api.services.MeIterator;
import com.sap.ip.me.api.smartsync.BasisFieldType;
import com.sap.ip.me.api.smartsync.CharacterField;
import com.sap.ip.me.api.smartsync.DecimalField;
import com.sap.ip.me.api.smartsync.Field;
import com.sap.ip.me.api.smartsync.FieldDescriptor;
import com.sap.ip.me.api.smartsync.FieldDescriptorIterator;
import com.sap.ip.me.api.smartsync.NumericField;
import com.sap.ip.me.api.smartsync.Row;
import com.sap.ip.me.api.smartsync.RowDescriptor;
import com.sap.ip.me.api.smartsync.SmartSyncException;
import com.sap.ip.me.api.smartsync.SmartSyncRuntime;
import com.sap.ip.me.api.smartsync.SmartSyncTransactionManager;
import com.sap.ip.me.api.smartsync.SyncBo;
import com.sap.ip.me.api.smartsync.SyncBoDataFacade;
import com.sap.ip.me.api.smartsync.SyncBoDescriptor;
import com.sap.ip.me.api.smartsync.SyncBoDescriptorFacade;

public class SmartSyncDBAccess implements Constants {
	private String[] tableHeaderNamesToDisplay;
	private String[] tableHeaderNames;
	private int headerFieldCount;
	private static SyncBoDataFacade dataFacade;
	private static SyncBoDescriptorFacade descriptorFacade;

	public SmartSyncDBAccess() {
		dataFacade = SmartSyncRuntime.getInstance().getSyncBoDataFacade();
		descriptorFacade = SmartSyncRuntime.getInstance().getSyncBoDescriptorFacade();
	}

	// Methods
	public MeIterator getSyncBoInstances(String syncBoName) {
		try {
			SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
			MeIterator iteratorSyncBos = dataFacade.getSyncBos(sbd).iterator();
			return iteratorSyncBos;
		} catch (PersistenceException pex) {
			System.out.println(pex.getMessage());
			return null;
		}
	}
	public SyncBo getSyncBoInstance(String syncBoName, String syncKey) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		try {
			SyncBo sb = dataFacade.getSyncBo(sbd, syncKey);
			return sb;
		} catch (PersistenceException pex) {
			System.out.println(pex.getMessage());
			return null;
		}
	}
	public String[] getHeaderFieldNames(String syncBoName) {
		SyncBoDescriptor sbd = descriptorFacade.getSyncBoDescriptor(syncBoName);
		RowDescriptor trd = sbd.getTopRowDescriptor();
		String[] arrayFieldNames = new String[trd.getFieldCount()];
		tableHeaderNamesToDisplay = new String[trd.getFieldCount()];
		int i = 0;
		if (trd.getFieldCount() > 0) {
			FieldDescriptorIterator fdi = trd.getAllFieldDescriptors();
			while (fdi.hasNext()) {
				// load array with name foer headers defined in the properties file
				arrayFieldNames[i] = fdi.next().getFieldName();
				tableHeaderNamesToDisplay[i] = GetProperties.getString(arrayFieldNames[i]);
				i++;
			}
			return arrayFieldNames;
		} else {
			System.out.println("SmartSyncDBAccess.getHeaderFieldNames - Array of Header Field Names is empty");
			return null;
		}
	}

	public String getHeaderFieldValue(SyncBo sb, String headerFieldName) {
		SyncBoDescriptor sbd = sb.getSyncBoDescriptor();
		RowDescriptor trd = sbd.getTopRowDescriptor();
		FieldDescriptor fd = trd.getFieldDescriptor(headerFieldName);
		if (fd != null) {
			BasisFieldType bft = fd.getFieldType();
			Row header = sb.getTopRow();
			try {
				if (bft == BasisFieldType.N) {
					NumericField nf = header.getNumericField(fd);
					return nf.getValueWithLeadingZeros();
				} else {
					Field f = header.getField(fd);
					if (f == null || (f.getValue() == null)) {
						return "";
					} else {
						return f.getValue().toString();
					}
				}
			} catch (SmartSyncException ssex) {
				System.out.println(ssex.getMessage());
				return null;
			}
		} else {
			return null;
		}
	}

	public boolean setHeaderFieldValue(SyncBo sb, String headerFieldName, Object value) {
		SyncBoDescriptor sbd = sb.getSyncBoDescriptor();
		RowDescriptor trd = sbd.getTopRowDescriptor();
		FieldDescriptor fd = trd.getFieldDescriptor(headerFieldName);
		if (fd != null) {
			BasisFieldType bft = fd.getFieldType();
			Row header = sb.getTopRow();
			try {
				if (bft == BasisFieldType.N) {
					NumericField nf = header.getNumericField(fd);
					if (nf != null) {
						BigInteger ii = new BigInteger(value.toString());
						nf.setValue(ii);
						return true;
					} else {
						return false;
					}
				}
				if (bft == BasisFieldType.C) {
					CharacterField cf = header.getCharacterField(fd);
					if (cf != null) {
						cf.setValue(value.toString());
						return true;
					} else {
						return false;
					}

				}
				if (bft == BasisFieldType.P) {
					DecimalField df = header.getDecimalField(fd);
					if (df != null) {
						BigDecimal bd = new BigDecimal(value.toString());
						df.setValue(bd);
						return true;
					} else {
						return false;
					}

				}
			} catch (SmartSyncException ex) {
				System.out.println(ex.getMessage());
			} catch (PersistenceException e) {
				System.out.println(e.getMessage());
			}
		}
		return false;
	}

	public String getItemFieldValue(Row item, String itemFieldName) {
		RowDescriptor rd = item.getRowDescriptor();
		FieldDescriptor fd = rd.getFieldDescriptor(itemFieldName);
		BasisFieldType bft = fd.getFieldType();

		try {
			if (bft == BasisFieldType.N) {
				NumericField nf = item.getNumericField(fd);
				return nf.getValueWithLeadingZeros();
			} else {
				Field f = item.getField(fd);
				if (f == null) {
					return "";
				} else {
					return f.getValue().toString();
				}
			}
		} catch (SmartSyncException ex) {
			System.out.println(ex.getMessage());
			return null;
		}
	}

	public void setItemFieldValue(Row item, String itemFieldName, Object value) {
		RowDescriptor rd = item.getRowDescriptor();
		FieldDescriptor fd = rd.getFieldDescriptor(itemFieldName);
		BasisFieldType bft = fd.getFieldType();

		try {
			if (bft == BasisFieldType.N) {
				NumericField nf = item.getNumericField(fd);
				nf.setValue(value);
				return;
			} else {
				Field f = item.getField(fd);
				if (f != null) {
					f.setValue(value);
				}
			}
		} catch (SmartSyncException ex) {
			System.out.println(ex.getMessage());
		} catch (PersistenceException e) {
			System.out.println(e.getMessage());
		}
	}

	public MeIterator readEntitiesFromDB(int index, int sortIndex, boolean sort_ascending, String obj) {
		Vector lineVector = new Vector();
		tableHeaderNames = getHeaderFieldNames(SYNCBO_NAME_INVENTORY);
		setColumns(tableHeaderNames.length);

		return getSyncBoInstances(SYNCBO_NAME_INVENTORY);
	}

	public Vector getEntities(MeIterator syncBos, int fromIndex, int count) {
		Vector retVec = new Vector();
		int i = 0;
		syncBos.reset();
		// Fill table header with Field Names of Top Row
		while (syncBos.hasNext()) {
			SyncBo sb = (SyncBo) syncBos.next();
			if (i >= fromIndex) {
				Vector rowData = new Vector();
				for (int col = 0; col < getColumns(); col++) {
					rowData.addElement(getHeaderFieldValue(sb, tableHeaderNames[col]));
				}
				retVec.addElement(rowData);
			}
			i++;
			//return when count has been reached
			if (i >= (fromIndex + count))
				return retVec;
		}
		return retVec;
	}

	public void modifyRowInDB(String syBName, String syncKey, String newValue, int col) {
		String syncBoName = syBName;
// Set default Syncboname for this example in case value is null.
		if (syncBoName == null){
			syncBoName = SYNCBO_NAME_INVENTORY;
		}
		SyncBo syncBo = getSyncBoInstance(syncBoName, syncKey);
		SmartSyncTransactionManager transactionManager;
		try {
			transactionManager = dataFacade.getSmartSyncTransactionManager();
			transactionManager.beginTransaction();
			setHeaderFieldValue(syncBo, tableHeaderNames[col], newValue);
			//Commit the transaction
			transactionManager.commit();
		} catch (SmartSyncException e) {
			System.out.println(e);
		} catch (PersistenceException e) {
			System.out.println(e);
		}

	}

	public int getColumns() {
		return headerFieldCount;
	}

	public void setColumns(int col) {
		headerFieldCount = col;
	}
	public String[] getTableHeaderNames() {
		return tableHeaderNamesToDisplay;
	}

	public void setTableHeaderNames(String[] strings) {
		tableHeaderNamesToDisplay = strings;
	}

	public String getTableHeaderName(int col) {
		return tableHeaderNamesToDisplay[col];
	}
}
