package com.sap.caf.km.datasvc;

import com.sap.caf.km.datasvc.data.BO;
import com.sap.caf.km.datasvc.data.BODescription;
import com.sap.caf.km.datasvc.data.BOInstance;
import com.sap.caf.km.datasvc.data.Relation;
import com.sap.caf.rt.exception.CAFPermissionException;
import com.sap.caf.rt.exception.DataAccessException;

/**
 * @author d040882
 *
 * This is the main interface for the business object browser. It contains methods to open and
 * close the connection to the model reporitory and methods to browse the business objects, its
 * instances and aggregations/associations between business objects. The "broewse methods" return
 * dataholder classes which can easily been serialized to expose them as a webservice.
 */
public interface ICAFDataService {
	
	/**
	 * returns all business objects which are registered in the 
	 * model repository.
	 * @return Array of guids for the business objects
	 */
	public BODescription[] getAllBOs()
				throws DataAccessException;
	/**
	 * if the given business object has associations to other
	 * business objects, these are returned
	 * @param guid - Guid of the business object
	 * @return Array of all associations from the business object
	 */
	public Relation[] getAssociatedBOs(String guidBO)
				throws DataAccessException;
	/**
	 * if the given business object has a aggregations to other
	 * business objects, these are returned.
	 * @param guid - Guid of the business object
	 * @return Array of all associations from the business object
	 */
	public Relation[] getAggregatedBOs(String guidBO)
				throws DataAccessException;	
	/**
	 * returns a dataholder class of the requested business object instance
	 * @param guidBO - guid of the business object
	 * @param guidInstance - guid of the instance
	 * @return dataholder class of the requested bo instance
	 */
	public BOInstance getBOInstance(String guidBO, String guidInstance)
				throws DataAccessException, CAFPermissionException;	
	/**
	 * Returns all instances of a business object whose guid starts with the
	 * given part
	 * @param guidBO - guid of the business object
	 * @param guidPart - part of the guid with which the requested instances have to start
	 * @return Array of guids
	 */
	public String[] getInstancesByGUIDPart(String guidBO, String guidPart)
				throws DataAccessException;		
	/**
	 * Returns all instances of a referenced business object whose guid starts with the
	 * given part.
	 * @param guidBO - guid of the business object where the reference starts
	 * @param guidRelation - guid of the relation between the business objects
	 * @param guidInstance - guid of the instance for the business object where the reference starts
	 * @param guidPart - part of the guid with which the requested instances have to start
	 * @return Array of guids
	 */				
	public String[] getRelatedInstancesByGUIDPart(String guidBO, String guidRelation, String guidInstance, String guidPart)
				throws DataAccessException;		
	
	/**
	 * Returns a business object whose guid matches the given one.
	 * @param guid - guid of the business object
	 * @return Returns the business object
	 */				
	public BO getBOByGUID(String guid)
				throws DataAccessException;
				
	/**
	 * Returns a the metadata of a business object by its name
	 * @param prov Name of the provider who created the business object
	 * @param app Name of the application of which the business object is part of
	 * @param bo Name of the business object
	 * @return Metadata of the requested business object
	 * @throws QueryException
	 * @throws ConnectionException
	 * @throws BONotFoundException
	 */
	public BO getBO(String prov, String app, String bo)
				throws DataAccessException;
				
	public BO getRelatedBOByRelationGUID(String relationGuid)
				throws DataAccessException;

	/**
	 * Returns a list of all partition elements which are populated for a business object.
	 * @param guid - guid of the business object
	 * @param noDigits - Length of the guids which are returned
	 * @param startGuid - Part of the guid with which the returned guid should start
	 * @return Array of strings with the names of the folders which are populated
	 */				
	public String[] getPopulatedFolders(String guid, int noDigits, String startGuid)
				throws DataAccessException;
				
	/**
	 * Returns a list of all partition elements which are populated for a related business object
	 * @param guidBO - guid of the business object
	 * @param guidRelation - guid of the relation to the specific business object
	 * @param guidInstance - guid of the instance of the source business object
	 * @param noDigits - Length of the guids which are returned
	 * @param startGuid - Part of the guid with which the returned guid should start
	 * @return Array of strings with the names of the folders which are populated
	 */
	public String[] getRelatedPopulatedFolders(String guidBO, String guidRelation, String guidInstance, int noDigits, String startGuid)
				throws DataAccessException;

	/**
	 * Returns all depenpendent objects of a business object
	 * @param guidBO - guid of the business object whose dependent objects are required
	 * @return Array of the guids of the dependend objects as strings
	 */	
	public Relation[] getComplexAttributes(String guidBO, String[] guidRelation)
				throws DataAccessException;

	/**
	 * Returns all instances of a dependent object which belongs to a business object instance
	 * @param guidBO - guid of the business object whose dependent objects are required
	 * @param guidInstance - guid of the business object instance
	 * @param bo - true if guidBO refers to a business object, false if it refers to a dependent object
	 * @param guidRelation - guid of 
	 * @return 
	 */	
	public String[] getComplexAttributeInstances(String guidBO, String guidInstance, String[] guidRelation, String[] guidDOInstance)
				throws DataAccessException;

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getDOByRelationGUID(String)
	 */	
	public BO getComplexAttribute(String guidDO)
				throws DataAccessException;

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getDOInstance(String, String, String, String)
	 */	
	public BOInstance getComplexAttributeInstance(String guidBO, String guidBOInstance, String[] guidDO, String[] guidDOInstance)
				throws DataAccessException, CAFPermissionException;

	public String[] getRelatedKMObjects(String fullBOName)  
				throws DataAccessException;

	
	/**
	 * This method performs a permission check for a specific business object using the caf security framework.
	 * @param boName Full qualified name of the business object
	 * @param userName Name of the user for whom the check should be performed
	 * @param guidInstance Guid of the instance
	 * @param rightsToCheck Deprecated. Is not used internally. Pass an empty string array
	 * @return Array of booleans. Each component indicates if the user has a certain permission {read, update, write, remove}
	 * @throws QueryException
	 * @throws ConnectionException
	 * @throws BONotFoundException
	 * @throws BOPermissionException
	 */
	public boolean[] checkRights(String boName, String userName, String guidInstance, String[] rightsToCheck)
				throws DataAccessException, CAFPermissionException;						
}
