package com.sap.caf.km.datasvc;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Vector;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.TransactionManager;
import javax.transaction.UserTransaction;

import com.sap.caf.km.datasvc.data.BO;
import com.sap.caf.km.datasvc.data.BODescription;
import com.sap.caf.km.datasvc.data.BOInstance;
import com.sap.caf.km.datasvc.data.Relation;
import com.sap.caf.km.datasvc.instreader.InstanceReader;
import com.sap.caf.km.datasvc.util.TypeConstants;
import com.sap.caf.metamodel.Application;
import com.sap.caf.metamodel.Attribute;
import com.sap.caf.metamodel.BaseObject;
import com.sap.caf.metamodel.BusinessEntityInterface;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.metamodel.DataStructure;
import com.sap.caf.rt.bol.IBusinessEntityService;
import com.sap.caf.rt.bol.IBusinessObject;
import com.sap.caf.rt.bol.IClassificationExt;
import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.bol.da.IDataAccessService;
import com.sap.caf.rt.bol.util.IntQueryFilter;
import com.sap.caf.rt.bol.util.QueryFilter;
import com.sap.caf.rt.exception.CAFBaseException;
import com.sap.caf.rt.exception.CAFPermissionException;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.metamodel.MetaModel;
import com.sap.caf.rt.metamodel.RepositoryConnection;
import com.sap.caf.rt.security.util.CAFPermissionName;
import com.sap.caf.rt.security.util.CAFPermissionUtil;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.caf.km.da.DocumentDataAccessService;
import com.sap.dictionary.runtime.IDataType;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
import com.sap.tc.webdynpro.repository.DataTypeBroker;

/**
 * @author d040882
 *
 * Implementation of the business object browser. This class contains methods
 * to handle the connection to the model repository as well as methods to browse
 * the business objects, its instances and aggregations/associations between them.
 */
public class CAFDataServiceImpl implements ICAFDataService {

	private static final String jARMRequest="CAF:rt:cafdataservice";
	private static final Location location = Location.getLocation(CAFDataServiceImpl.class);
	final static private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
	private Context ctx = null;
	private MetaModel mmrProxy = null;
	private InstanceReader instProxy = null;

	private final static int kmBackEnd = 2;

	private final static String JNDI_PREFIX = "localejbs/";
	private final static String BE_DOCUMENT_NAME = "sap.com/caf.core/Document";
	private final static String VO_DOCUMENT = "com.sap.caf.core.besrv.document.Document";
	
	/**
	 * BE Document Service Name
	 * @see getDocumentBEServiceName 
	 */
	private static String m_beDocumentServiceName = null; 	
	
	

	public CAFDataServiceImpl() throws DataAccessException {
		
		String method = "CAFDataServiceImpl()";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[0]);

		//initialize the connection to the business object factory
		//used to get information about single business object instances
		try {		
			ctx = new InitialContext();
			location.debugT("Initialization of jndi context successfull", new Object[0]);
		} catch (NamingException ex) {
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, method, "Error in initializing jndi context",
							new Object[0], ex);
			throw new DataAccessException("JNDI_CONTEXT_ERROR", new Object[0], ex); //$NON-NLS-1$
		}

		//initialize the connection to the model repository
		//used for queries to the metadata api
		mmrProxy = new MetaModel();
		location.debugT("Initialization of connection to metadata repository successfull", new Object[0]);

		//initialize the connection to the service for reading the business object instances
		instProxy = new InstanceReader();

		CAFPublicLogger.exiting(null, jARMRequest, method, location);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getAllBOs()
	 */
	public BODescription[] getAllBOs() throws DataAccessException {
		String method = "getAllBOs()";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[0]);

		//stores the dataholder classes of all business objects which are returned
		//by the query
		Collection col = new Vector();
		Collection tmp = null;

		// get a list of all objects
		Iterator iterator = null;
		//business objects are shareableobjects of objectType 0
		tmp = mmrProxy.getAllBusinessEntityInterfaces(0);
		
		location.debugT("Reading data from mmr successfull", new Object[0]);
		
		//all objects whose source is km are removed from the collection
		//filtering is disabled as all business entities should be visible in repository manager
		//tmp = filterObjects(tmp, true);
		//location.debugT("Filtering of km objects successfull", new Object[0]);
		
		//loop through the results and extract the guid (mofid) of each object
		Iterator it = tmp.iterator();
		while (it.hasNext()) {
			Object obj = it.next();
			if (obj instanceof BusinessEntityInterface) {
				BusinessEntityInterface so = (BusinessEntityInterface) obj;
				BaseObject bo = so.getBusinessEntity();								
				
				if (so.getBackend().intValue() == kmBackEnd)
					col.add(new BODescription(bo.getObjectId(), true));
				else
					col.add(new BODescription(bo.getObjectId(), false));
			}
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (BODescription[]) col.toArray(new BODescription[col.size()]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getAssociatedBOs(String)
	 */
	public Relation[] getAssociatedBOs(String guidBO) throws DataAccessException {
		String method = "getAssociatedBOs(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] { guidBO });

		Collection tmp = new Vector();
		Map map = null;

		DataObject so = mmrProxy.getDataObjectByGUID(guidBO);
//		Application app = so.getApplication();
		
		map = mmrProxy.getRelatedBEsByGUID( so.getBusinessEntityInterface().refMofId() );
		
		location.debugT("Reading data from mmr successfull", new Object[0]);
		
		//TODO: related km objects are not filtered. they are only not shown on the first level which is queried via getAllBOs 
		//remove all objects which don´t come from km
		//map=removeKMObjects(map);
		//filtering is disabled as all business entities should be visible in repository manager
		//tmp = filterObjects(tmp, true);
		//map = filterObjects(map, false);
		
		location.debugT("Filtering of km objects successfull", new Object[0]);
		
		//loop through the results and extract the guid of the business object and the guid of the attribute object
		Iterator it = map.keySet().iterator();
		while (it.hasNext()) {
			String key = (String) it.next();
			BusinessEntityInterface relatedSo = (BusinessEntityInterface) map.get(key);

			if (relatedSo!=null) {
				BODescription bo = null;
				if (relatedSo.getBackend().intValue() == kmBackEnd)
					bo = new BODescription(relatedSo.getBusinessEntity().refMofId(), true);
				else
					bo = new BODescription(relatedSo.getBusinessEntity().refMofId(), false);

				Relation rel = new Relation(bo, key);

				tmp.add(rel);				
			}
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (Relation[]) tmp.toArray(new Relation[0]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getAggregatedBOs(String)
	 */
	public Relation[] getAggregatedBOs(String guidBO) throws DataAccessException {
		String method = "getAggregatedBOs(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO});

		Collection tmp = new Vector();
		Map map = null;

//		DataObject so = mmrProxy.getDataObjectByGUID(guidBO);
//		Application app = so.getApplication();

		map = mmrProxy.getRelatedBEsByGUID(guidBO);
		
		location.debugT("Reading data from mmr successfull", new Object[0]);
		
		//remove all objects which don´t come from km
		//		TODO: related km objects are not filtered. they are only not shown on the first level which is queried via getAllBOs
		//		map=removeKMObjects(map);
		//filtering is disabled as all business entities should be visible in repository manager
		//tmp = filterObjects(tmp, true);		
		//map = filterObjects(map, false);
		
		location.debugT("Filtering of km objects successfull", new Object[0]);
		
		Iterator it = map.keySet().iterator();
		while (it.hasNext()) {
			String key = (String) it.next();
			DataObject relatedSo = (DataObject) map.get(key);

			BODescription bo = null;
			if (relatedSo.getBusinessEntityInterface().getBackend().intValue() == kmBackEnd)
				bo = new BODescription(relatedSo.refMofId(), true);
			else
				bo = new BODescription(relatedSo.refMofId(), false);

			Relation rel = new Relation(bo, key);
			tmp.add(rel);
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);
		
		return (Relation[]) tmp.toArray(new Relation[0]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getBOByGUID(String)
	 */
	public BO getBOByGUID(String guid) throws DataAccessException {
		String method = "getBOByGUID(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guid});

		DataObject so = null;

		so = mmrProxy.getDataObjectByGUID(guid);

		location.debugT("Reading data from mmr successfull", new Object[0]);
		//if the object comes from km, it mustn´t be returned
		/*		if(checkKMObject(so))
					throw new BONotFoundException();*/

		BO tmp = convertBO(so);

		location.debugT("Converting metadata successfull", new Object[0]);
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return tmp;
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getBOInstance(String,String)
	 */
	public BOInstance getBOInstance(String guidBO, String guidInstance)
		throws DataAccessException, CAFPermissionException {
		String method = "getBOIstance(String, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidInstance});

		BOInstance tmp = null;
		String fullObjectName = null;
		IBusinessObject obj = null;
		TransactionManager tr = null;

		try {
	
			tr = getResourceTransaction();
			tr.begin();	

			BO tmpBO = getBOByGUID(guidBO);
		
			location.debugT("Reading data from mmr successfull", new Object[0]);
			
			obj = getLocalInterface(tmpBO, guidInstance);
			tmp = convertBOInstance(tmpBO.getGuidBO(), tmpBO.getProviderBO(), tmpBO.getAppBO(), tmpBO.getIdBO(), obj);
			tr.rollback();
			tr = null;
			
			location.debugT("Converting of business entity instance successfull", new Object[0]);			

			return tmp;
		} catch(CAFPermissionException ex) {
			String[] args = { guidBO, guidInstance };
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, method, "Cannot get BO Instance. BO: {0} Instance: {1}",
							new Object[0], ex);
			throw ex;						
		} catch (Throwable ex) {
				//			$JL-EXC$
				String[] args = { fullObjectName, guidInstance };
				throw new DataAccessException("CANNOT_READ_BO", args, ex); //$NON-NLS-1$
		}
		finally {		
			try {
				if (tr!=null) {
					tr.rollback();	
				}				
			} catch (Exception ex) {
				CAFPublicLogger.categoryCAF.logThrowableT(Severity.ALL, location, method, "Cannot rollback resource transaction",
								new Object[0], ex);
			}
			CAFPublicLogger.exiting(null, jARMRequest, method, location);
		}

	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getInstancesByGUIDPart(String,String)
	 */
	public String[] getInstancesByGUIDPart(String guidBO, String guidPart) throws DataAccessException {
		String method = "getInstancesByGUIDPart(String, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidPart});
		
		Collection guids = null;

		try {
			guids = instProxy.getInstances(guidBO, guidPart);
			location.debugT("Reading data from database successfull", new Object[0]);
		} catch (Exception ex) {
			String[] args = { guidBO };
			throw new DataAccessException("CANNOT_GET_BO", args, ex); //$NON-NLS-1$
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (String[]) guids.toArray(new String[0]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getRelatedInstancesByGUIDPart(String,String,String,String)
	 */
	public String[] getRelatedInstancesByGUIDPart(
		String guidBO,
		String guidRelation,
		String guidInstance,
		String guidPart)
		throws DataAccessException {
		String method = "getRelatedInstancesByGUIDPart(String, String, String, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidRelation, guidInstance, guidPart});
			
		Collection guids = null;
		TransactionManager tr = null;


		try {			

			//related km object have to treated different from caf objects 
			BusinessEntityInterface so = mmrProxy.getRelatedBOByRelation(guidRelation);
			location.debugT("Reading data from mmr successfull", new Object[0]);
			
			tr = getResourceTransaction();
			tr.begin();	

			if (so.getBackend().intValue() == kmBackEnd) {

				BO tmpBO = getBOByGUID(guidBO);
				location.debugT("Reading data from mmr successfull", new Object[0]);
				
				Object obj = getLocalInterface(tmpBO, guidInstance);				
				
				Attribute attr = mmrProxy.getAttribute(guidRelation);
				location.debugT("Reading data from mmr successfull", new Object[0]);
				
				String attrName = attr.getObjectName();
				char ch = Character.toUpperCase(attrName.charAt(0));
				attrName = ch + attrName.substring(1, attrName.length());				

				Method m = obj.getClass().getMethod("get" + attrName, null);
				String[] tmp=null;
				Object oRes = m.invoke(obj, null);
				if(oRes instanceof Collection){ 
					Collection col = (Collection)oRes;
					tmp = (String[]) col.toArray(new String[0]);
				} else {
					tmp = new String[] {m.invoke(obj, null).toString()};
				}
				tr.rollback();
				return tmp;
			} else {
				guids = instProxy.getRelatedInstances(guidBO, guidInstance, guidRelation, guidPart);
				tr.rollback();
				location.debugT("Reading data from database successfull", new Object[0]);
			}
		} catch (Exception ex) {
			String[] args = { guidBO };
			throw new DataAccessException("CANNOT_GET_RELATED_BO", args, ex); //$NON-NLS-1$
		}	

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (String[]) guids.toArray(new String[0]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getRelatedBOByRelationGUID(String)
	 */
	public BO getRelatedBOByRelationGUID(String relationGuid)
		throws DataAccessException {
		String method = "getRelatedBOByRelationGUID(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {relationGuid});
		
		BusinessEntityInterface so = null;
		int relationType = 0;

		//resolve the shareable object for the guid
		so = mmrProxy.getRelatedBOByRelation(relationGuid);
		
		location.debugT("Reading metadata from mmr successfull", new Object[0]);
		
		//resolve the relation object (attribute) itself to get the type of relation			
		Attribute attr = mmrProxy.getAttribute(relationGuid);
		
		location.debugT("Reading data from database successfull", new Object[0]);
		relationType = 0; //attr.getRelation().getRelationType1();

		BO tmp = convertBOWithRelation(so.getBusinessEntity(), relationGuid, relationType);

		location.debugT("Converting metadata successfull", new Object[0]);
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return tmp;
	}

	/**
	 * Extracts all necessary data from a dataobject/shareableobject and returns it as
	 * a business object dataholder class
	 * @param so DataObject/BusinessEntityInterface which should be converted
	 * @return Returns a new instance of an business object
	 */
	private BO convertBO(DataObject so) {
		String method = "convertBO(DataObject)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {so});
		
		//call the more generic version of this method. relation attributes are set to null
		BO tmp = convertBOWithRelation(so, null, 0);

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return tmp;
	}

	/**
	 * @param so
	 * @param relationRole
	 * @param relationType
	 * @return
	 */
	/**
	 * @param so
	 * @param relationRole
	 * @param relationType
	 * @return
	 */
	/**
	 * Extracts all necessary data from the dataobject/shareableobject and returns it as
	 * a business object dataholder class
	 * @param so DataObject which should be converted
	 * @param relationRole Guid of the relation which points to the shareable object. Can be null
	 * @param relationType Indicates the type of relation which is described by relationRole (0=>Assoc, 1=>Aggr)
	 * @return Returns a new instance of an business object
	 */
	private BO convertBOWithRelation(DataObject so, String relationRole, int relationType) {
		String method = "convertBOWithRelation(DataObject, String, int)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {so, relationRole, new Integer(relationType)});
		
		//copy all static attributes
		Application app = so.getApplication();
		if(app==null)
			app = so.getBusinessEntityInterface().getApplication();				

		//use the mofid of the business entity interface instead of the business entity
		BO tmp = new BO(so.refMofId(), app.getProviderName(), app.getObjectName(), so.getObjectName(), so.getBusinessEntityInterface().getObjectName());
		tmp.setRoleBO(relationRole);
		tmp.setRelationType(relationType);
		tmp.setCreatedBy(so.getCreatedBy());
		tmp.setLastChangedBy(so.getLastChangedBy());
		tmp.setLongText(so.getLongText());
		tmp.setShortText(so.getShortText());

		//all time values are converted to milliseconds
		try {
			if (so.getCreatedAt() != null)
				tmp.setCreatedAt(dateFormat.parse(so.getCreatedAt()).getTime());
			if (so.getLastChangedAt() != null)
				tmp.setLastChangedAt(dateFormat.parse(so.getLastChangedAt()).getTime());
		} catch (ParseException ex) {
			//logging
			CAFPublicLogger.traceThrowableT(Severity.ERROR, location, method, "Created/changed Date cannot be parsed",
							new Object[0], ex);
			
		}
		//if it is a shareableobject, the object type can be taken directly from the shareableobject
		if (so instanceof BusinessEntityInterface) {
			tmp.setType(0);
			tmp.setBackend(((BusinessEntityInterface) so).getBackend().intValue());
		}
		//otherwise it´s a dependent object
		else {
			tmp.setType(99);
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return tmp;
	}

	/**
	 * Extracts all necessary data from an business object instance and returns it as
	 * a dataholder class of type BOInstance
	 * @param guid Guid of the business object
	 * @param provider Name of the business object provider
	 * @param application Name of the business object application
	 * @param id Id of the business object
	 * @param bo Reference to the local interface of an ejb instance
	 * @return Returns an instance of BOInstance which contains all necessary data
	 */
	private BOInstance convertBOInstance(
		String guid,
		String provider,
		String application,
		String id,
		IDependentObject bo)
		throws CAFPermissionException, CAFBaseException {
		String method = "convertBOInstance(String, String, String, String, IDependentObject)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guid, provider, application, id, bo});
		
		BOInstance instance = new BOInstance(guid, provider, application, id, bo.getKey());
		
		//BusinessEntityInterface so = mmrProxy.getDataObjectByGUID(guid).getBusinessEntityInterface();
		DataObject so = mmrProxy.getDataObjectByGUID(guid);
		
		//the following attributes are only valid for real business objects
		if (bo instanceof IBusinessObject) {
			//use only the display names
			instance.setCreatedBy(((IBusinessObject) bo).getCreatedBy());
			instance.setLastChangedBy(((IBusinessObject) bo).getLastChangedBy());

			//time is converted to milliseconds
			Date tmp = ((IBusinessObject) bo).getCreatedAt();
			if (tmp != null)
				instance.setCreatedAt(tmp.getTime());

			tmp = ((IBusinessObject) bo).getLastChangedAt();
			if (tmp != null)
				instance.setLastChangedAt(tmp.getTime());
				
			Attribute desc = so.getDefaultLongText();
			if (desc!=null) 
				instance.setDescription((String)((IBusinessObject)bo).getProperty(desc.getObjectName()));
			else
				instance.setDescription("");
			
			Attribute dispName = so.getDefaultShortText();
			if (dispName!=null)
				instance.setDisplayName((String)((IBusinessObject)bo).getProperty(dispName.getObjectName()));
			else
				instance.setDisplayName(bo.getObjectType());
		}

		instance.setBackend(so.getBusinessEntityInterface().getBackend().intValue());

		//loop through the attributes defined for the business object and read their values
		/*Iterator itMetaAttributes = so.getAttributes().iterator();
		while(itMetaAttributes.hasNext()) {
			Attribute metaAttr = (Attribute)itMetaAttributes.next();*/

		instance.setAttributes(fillAttributes(so.getBusinessEntityInterface(), bo));
		location.debugT("Converting attributes for business entity successfull", new Object[0]);
				
		instance.setCategories(fillCategories(so.getBusinessEntityInterface(), bo));
		location.debugT("Converting categories for business entity successfull", new Object[0]);
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return instance;
	}

	/**
	 * This methods will extract all categories and classifications that are assigned to a business object.
	 * The method uses reflection to access the business object category. The reason is that the business
	 * object category cannot be used because of a cyclic reference between caf~runtime~core and caf~core 
	 * @param so metadata of the business object
	 * @param obj ejb instance of the business object
	 * @return
	 * @throws BOException
	 */
	private com.sap.caf.km.datasvc.data.Attribute[] fillCategories(BusinessEntityInterface so, IDependentObject obj)
		throws CAFBaseException {

		String method = "fillCategories(BusinessEntityInterface, IDependentObject)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {so, obj});
		
		Collection col = new ArrayList();
		
		try {
			Object home = ctx.lookup(JNDI_PREFIX + "sap.com/caf.tc/CategoryService");

			location.debugT("Jndi lookup for business entity {0} succesfull", new Object[] {"sap.com/caf.tc/CategoryService"});
		
			Map mapValues = new HashMap();
				
			Application app = so.getApplication();
			String provider = app.getProviderName();
			String appName = app.getObjectName();
		
			//get all categories
			Method cr_meth = home.getClass().getMethod("create", new Class[0]);
			Object createdService = cr_meth.invoke(home, new Object[0]);
			Method findCatByName_meth = createdService.getClass().getMethod("findCategoryByBOName", new Class[] {String.class});
			Collection categories = new ArrayList(1);
			if(findCatByName_meth != null) {
				categories = (Collection) findCatByName_meth.invoke(createdService,new Object[] {
					provider + "/" + appName + "/" +so.getBusinessEntity().getObjectName()});
			}
													
			location.debugT("Found {0} assigned categories", new Object[] {new Integer(categories.size())});

			//get getClassification method for BusinessEntityInterface service
			Object classif = ctx.lookup(JNDI_PREFIX + provider + "/" + appName + "/" + so.getObjectName());
			Method cr_meth_classif = classif.getClass().getMethod("create", new Class[0]);
			Object createdServiceClassif = cr_meth_classif.invoke(classif, new Object[0]);
			Method getclassif_meth = createdServiceClassif.getClass().getMethod("getClassification",
				new Class[]{String.class, String.class});
								
			//get read method for CategoryValueService
			Object categoryValueHome = ctx.lookup(JNDI_PREFIX + "sap.com/caf.tc/CategoryValueService");
			Method createCategoryValueHome_meth = categoryValueHome.getClass().getMethod("create", new Class[0]);
			Object categoryValueService = createCategoryValueHome_meth.invoke(categoryValueHome, new Object[0]);
			Method categoryValueService_read_meth = categoryValueService.getClass().getMethod("read", new Class[] {String.class});
			
			Iterator itCat = categories.iterator();
			//loop on all categories
			while(itCat.hasNext()) {
				Collection values = new ArrayList();
				Object category = itCat.next();

				Method getKey = category.getClass().getMethod("getKey", new Class[0]);
				String categoryKey = (String)getKey.invoke(category, new Object[0]);
				Collection allClass = (Collection)getclassif_meth.invoke(createdServiceClassif,
					new Object[] {obj.getKey(), categoryKey});

				Iterator itClass = allClass.iterator();
				//loop on all category values 
				while(itClass.hasNext()) {						
					String categoryValueKey = (String)itClass.next();
					Object categoryValue = categoryValueService_read_meth.invoke(
						categoryValueService, new Object[] {categoryValueKey});
					
					Collection shortTexts=new ArrayList();
					Collection longTexts=new ArrayList();
								
					Method shortText = categoryValue.getClass().getMethod("getShortText", new Class[0]);
					HashMap map1 = (HashMap)shortText.invoke(categoryValue, new Object[0]);
					if(map1!=null) {
						Iterator it1 = map1.keySet().iterator();
						while(it1.hasNext()) {
							String language = (String)it1.next();
							String value = (String)map1.get(language);
							shortTexts.add( language + "###" + value);	
						}
					}

					Method longText = categoryValue.getClass().getMethod("getLongText", new Class[0]);
					Object v2 = longText.invoke(categoryValue, new Object[0]);
					HashMap map2 = (HashMap)shortText.invoke(categoryValue, new Object[0]);
					if(map2!=null) {
						Iterator it2 = map2.keySet().iterator();
						while(it2.hasNext()) {
							String language = (String)it2.next();
							String value = (String)map2.get(language);
							longTexts.add( language + "###" + value);	
						}
					}
											
					Method name = categoryValue.getClass().getMethod("getName", new Class[0]);
					String value = (String)name.invoke(categoryValue, new Object[0]);
			
					//concatenate short texts for all languages
					Iterator shortTextIterator = shortTexts.iterator();
					while ( shortTextIterator.hasNext() ) {
				 		value = value +";;" + (String)shortTextIterator.next();
					}
					//concatenate long texts for all languages					
					Iterator longTextIterator = longTexts.iterator();
					while ( longTextIterator.hasNext() ) {
				 		value = value +";;" + (String)longTextIterator.next();
					}						
					values.add(value);								
				}
					
				Method name = category.getClass().getMethod("getId", new Class[0]);
				String value = (String)name.invoke(category, new Object[0]);
				value = URLEncoder.encode(value);

				if(values.size()==0)
					values.add("");
															
				com.sap.caf.km.datasvc.data.Attribute attr = new com.sap.caf.km.datasvc.data.Attribute(value, 
																(String[])values.toArray(new String[0]));
				attr.setCollection(true);
				attr.setType(TypeConstants.TYPE_STRING);
					
				location.debugT("Converting of assigned category {0} successfull", new Object[] {value});
				col.add(attr);																											
			}				
		}
		catch(Exception ex) {
			CAFPublicLogger.traceThrowableT(Severity.ERROR, location, method, "Categories could not be resolved for business entity",
							new Object[0], ex);
		}
		finally {
			CAFPublicLogger.exiting(null, jARMRequest, method, location);
		}
				
		return (com.sap.caf.km.datasvc.data.Attribute[])col.toArray(new com.sap.caf.km.datasvc.data.Attribute[0]);
	}

	private com.sap.caf.km.datasvc.data.Attribute[] fillAttributes(BusinessEntityInterface so, IDependentObject obj)
		throws CAFBaseException {
		String method = "fillAttributes(BusinessEntityInterface, IDependentObject)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {so, obj});
			
		Collection attributes = new Vector();
		List list = so.getBusinessEntity().getAttributes();

		//loop through all attributes of the business object
		for (int l = 0; l < list.size(); l++) {
			Attribute metaAttr = (Attribute) list.get(l);
			
			//simple and complex attributes have to be treaten different
			if (metaAttr.getReferencedObject() == null) {
				//handle a simple attribute
				attributes.add(createAttribute(obj, metaAttr));
			}
			else {
				//handle a complex attribute
				DataStructure dataObj = metaAttr.getReferencedObject();
				if((dataObj instanceof BusinessEntityInterface) && !(dataObj instanceof DataObject))
					continue;
					
				IDependentObject obj2=null;
				try {
					Method getMethod = obj.getClass().getMethod("get" + metaAttr.getName(), new Class[0]);
					obj2 = (IDependentObject)getMethod.invoke(obj, new Object[0]);
				}
				catch(Exception e) {
					continue;
				}
				
				//handle all simple attributes of a complex attribute
				List list2 = dataObj.getAttributes();
				for(int i=0;i<list2.size();i++) {
					Attribute complexAttr = (Attribute) list2.get(i);
					if (metaAttr.getReferencedObject() != null) {
						com.sap.caf.km.datasvc.data.Attribute attr = createAttribute(obj2, complexAttr);
						attr.setName(metaAttr.getName() + "." + attr.getName());						
						attributes.add(attr);
					}
				}
			}
			location.debugT("Converting of attribute {0} successfull", new Object[] {metaAttr.getObjectName()});
		}
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);
		
		return (com.sap.caf.km.datasvc.data.Attribute[]) attributes.toArray(
			new com.sap.caf.km.datasvc.data.Attribute[0]);
	}
	
	private com.sap.caf.km.datasvc.data.Attribute createAttribute(IDependentObject obj, Attribute metaAttr) throws CAFBaseException {
		String method = "createAttribute(IDependentObject, Attribute)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {obj, metaAttr});
		try {		
			Locale locale = Locale.getDefault(); 
			ClassLoader objClassLoader = obj.getClass().getClassLoader();
			Class attrClass = DataTypeBroker.getDataType("ddic:"+metaAttr.getTypeJavaDdic(), locale, objClassLoader).getAssociatedClass();
			String attrName = metaAttr.getObjectName();
			Object property = obj.getProperty(attrName);
			com.sap.caf.km.datasvc.data.Attribute kmAttr = null;
			if ((metaAttr.getMaxOccurs().intValue()==0) || (metaAttr.getMaxOccurs().intValue()==1)) {
			  // if cardinality is 1, than the return type is a simple type
				String type = TypeConstants.getType(attrClass);
				String value = TypeConstants.convertToString(type, property);
				kmAttr = new com.sap.caf.km.datasvc.data.Attribute(attrName, new String[] {value});
				kmAttr.setType(type);
				kmAttr.setCollection(false);
			}
			else { 
				// otherwise the return type is a collection
				String type = TypeConstants.getType(attrClass);
				List values = new ArrayList();
				if (property==null) {
					values.add(TypeConstants.convertToString(type, null));				
				}
				else {
					for (Iterator itr = ((Collection)property).iterator(); itr.hasNext(); ) {
						values.add(TypeConstants.convertToString(type, itr.next()));
					}
				}
				kmAttr = new com.sap.caf.km.datasvc.data.Attribute(attrName, (String[])values.toArray(new String[values.size()]));
				kmAttr.setType(type);
				kmAttr.setCollection(true);
			}
			return kmAttr;
		}
		finally {		
			CAFPublicLogger.exiting(null, jARMRequest, method, location);
		}		
	}
	

	/**
	 * Removes all shareable objects from the collection whose source is km.
	 * @param col Collection of dataobjects
	 * @param bKM Determines whether km objects are filtered or not
	 */
//	TODO: remove this method if repository manager in caf 1.0 os tested
	private Collection filterObjects(Collection col, boolean bKM) {
		String method = "filterObjects(Collection, boolean)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {col, new Boolean(bKM)});
		
		Collection newObjects = new ArrayList();
		Iterator objects = col.iterator();

		//loop through the collection
		while (objects.hasNext()) {
			BusinessEntityInterface so = (BusinessEntityInterface) objects.next();

			//remove the object if it comes from km
			if (!checkRemoveObject(so, bKM))
				newObjects.add(so);
		}
		return newObjects;
	}

	/**
	 * Removes all data objects from the map whose source is km.
	 * @param map HashMap of shareable objects. Index of the map is the guid of
	 * 			  the relation which points to the data object. The value
	 * 			  of the map is the shareable object itself.
	 * @param bKM Determines whether km objects are filtered or not
	 */
	//TODO: remove this method if repository manager in caf 1.0 os tested
	private Map filterObjects(Map map, boolean bKM) {
		String method = "filterObjects(Map, boolean)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {map, new Boolean(bKM)});
		
		Map newObjects = new HashMap();
		Iterator keys = map.keySet().iterator();

		//loop through all keys
		while (keys.hasNext()) {
			String key = (String) keys.next();

			//get the object for this key and check its source
			BusinessEntityInterface so = (BusinessEntityInterface) map.get(key);
			if (!checkRemoveObject(so, bKM))
				newObjects.put(key, so);
		}
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);
		
		return newObjects;
	}

	/**
	 * Checks if the source of a shareable object is km.
	 * @param so BusinessEntityInterface whose source should be checked
	* @param bKM Determines whether km objects are filtered or not
	 * 	 * @return True if the object comes from another source than km, otherwise false
	 */
//	TODO: remove this method if repository manager in caf 1.0 os tested
	private boolean checkRemoveObject(BusinessEntityInterface so, boolean bKM) {
		String method = "checkRemoveObject(BusinessEntityInterface, boolean)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {so, new Boolean(bKM)});
		
		boolean result=false;
		if (so.getBackend().intValue() == kmBackEnd)
			//km objects
			result=bKM;
		else if (so.getBackend().intValue() == 0)
			//objects with local persistency
			result= false;
		else
			result=true;
			
		CAFPublicLogger.exiting(null, jARMRequest, method, location);
		return result;
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getPopulatedFolders(String, int, String)
	 */
	public String[] getPopulatedFolders(String guid, int noDigits, String startGuid)
		throws DataAccessException {
		String method = "getPopulatedFolders(String, int, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guid, new Integer(noDigits), startGuid});
						
		Collection guids = null;

		try {
			guids = instProxy.getInstancePrefixes(guid, noDigits, startGuid);
		} catch (Exception ex) {
			String[] args = { guid, startGuid };
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, method, "Cannot get populated folders for business object {0} with prefix {1}",
							new Object[0], ex);
			throw new DataAccessException("CANNOT_GET_POPULATED_FOLDERS", args, ex); //$NON-NLS-1$
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (String[]) guids.toArray(new String[0]);
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getRelatedPopulatedFolders(String, String, String, int, String)
	 */
	public String[] getRelatedPopulatedFolders(
		String guidBO,
		String guidRelation,
		String guidInstance,
		int noDigits,
		String startGuid)
		throws DataAccessException {

		String method = "getRelatedPopulatedFolders(String, String, String, int, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidRelation, guidInstance, new Integer(noDigits)});
		
		Collection guids = null;

		try {
			Attribute attr = mmrProxy.getAttribute(guidRelation);
			location.debugT("Reading metadata from mmr successfull", new Object[0]);
			guids = instProxy.getRelatedInstancePrefixes(guidBO, guidInstance, attr.refMofId(), noDigits, startGuid);
			location.debugT("Reading data from database successfull", new Object[0]);
		} catch (Exception ex) {
			String[] args = { guidBO, startGuid };
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, method, "Cannot get populated folders for business object {0} with prefix {1}",
							new Object[0], ex);
			throw new DataAccessException("CANNOT_GET_RELATED_POPULATED_FOLDERS", args, ex); //$NON-NLS-1$
		}

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (String[]) guids.toArray(new String[0]);
	}

	/**
	 * Returns all dependent objects of a business object
	 * @param guidBO - guid of the business object whose dependent objects are required
	 * @return Array of the guids of the dependend objects as strings
	 */
	public Relation[] getComplexAttributes(String guidBO, String[] guidRelation)
		throws DataAccessException {
		String method = "getComplexAttributes(String, String[])";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidRelation});

		//stores the dataholder classes of all business objects which are returned
		//by the query
		Collection tmp = new Vector();
		Attribute attr = null;
		DataObject obj = null;
		Iterator it = null;
		BusinessEntityInterface so = null;

		/*so = mmrProxy.getBusinessEntityInterfaceByGUID(guidBO);

		if ((guidRelation == null) || (guidRelation.length == 0)) {
			it = so.getBusinessEntity().getAttributes().iterator();
		} else {
			obj = mmrProxy.getDataObjectByGUID(guidRelation[guidRelation.length - 1]);
			it = obj.getAttributes().iterator();
		}

		while (it.hasNext()) {
			attr = (Attribute) it.next();
			Object refObj = attr.getReferencedObject();
			if ((refObj instanceof DataObject) && !(refObj instanceof BusinessEntityInterface)) {
				BODescription bo = null;
				if (so.getBackend().intValue() == kmBackEnd)
					bo = new BODescription(obj.refMofId(), true);
				else
					bo = new BODescription(obj.refMofId(), false);

				Relation rel = new Relation(bo, attr.refMofId());
				tmp.add(rel);
			}
		}*/

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return (Relation[]) tmp.toArray(new Relation[0]);
	}

	/**
	 * Returns all instances of a dependent object which belongs to a business object instance
	 * @param guidBO - guid of the business object whose dependent objects are required
	 * @param guidInstance - guid of the business object instance
	 * @param guidRelation - guid of 
	 * @return 
	 */
	public String[] getComplexAttributeInstances(
		String guidBO,
		String guidInstance,
		String[] guidRelation,
		String[] guidDOInstance)
		throws DataAccessException {

		String method = "getComplexAttributeInstances(String, String, String[], String[])";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidInstance, guidRelation, guidDOInstance});
			
		String[] guids = null;
		String attrName = null;
		Attribute attr = null;
		String fullObjectName = null;
		IDependentObject obj = null;
	

		/*try {
			ut = getUserTransaction();
			beginTransaction(ut);

			//get the metadata of the business object
			BO tmpBO = getBOByGUID(guidBO);
			obj = (IDependentObject) getLocalInterface(tmpBO, guidInstance);

			//walk through the whole list of dependent objects  
			for (int i = 0; i < guidRelation.length - 1; i++) {
				//get the metadata of the relation to the next dependent object
				attr = mmrProxy.getAttribute(guidRelation[i]);
				attrName = attr.getObjectName();

				//invoke the method to get an instance of the related dependent object
				Method m = obj.getClass().getMethod("get" + attrName, null);

				//the multiplicity tells what return type has to be expected
				if ((attr.getMaxOccurs().intValue() >= 1) || (attr.getMaxOccurs().intValue()==-1)) {
					//return type is collection
					Iterator it = ((Collection) m.invoke(obj, null)).iterator();
					while (it.hasNext()) {
						obj = (IDependentObject) it.next();
						if (obj.equals(guidDOInstance[i]))
							break;

						rollbackTransaction(ut);
						throw new DataAccessException("Wrong parameters", new Object[0]);
					}
				} else {
					obj = (IDependentObject) m.invoke(obj, null);
					if (!obj.getKey().equals(guidDOInstance[i])) {
						rollbackTransaction(ut);
						throw new DataAccessException("Wrong parameters", new Object[0]);
					}
				}
			}

			//get the metadata of the relation to the last dependent object
			attr = mmrProxy.getAttribute(guidRelation[guidRelation.length - 1]);
			attrName = attr.getObjectName();

			//invoke the method to get an instance of the related dependent object
			Method m = obj.getClass().getMethod("get" + attrName, null);

			//the multiplicity tells what return type has to be expected
			if((attr.getMaxOccurs().intValue()>=2) && (attr.getMaxOccurs().intValue()!=-1)) {
				//return type is collection
				Collection col = (Collection) m.invoke(obj, null);
				Iterator it = col.iterator();
				guids = new String[col.size()];
				int k = 0;

				while (it.hasNext()) {
					obj = (IDependentObject) it.next();
					guids[k] = obj.getKey();
					k++;
				}
			} else {
				obj = (IDependentObject) m.invoke(obj, null);

				guids = new String[1];
				guids[0] = obj.getKey();
			}

		} catch (CAFBaseException ex) {
			String[] args = {
			};
			rollbackTransaction(ut);
			throw new DataAccessException("", args, ex);
		} catch (Exception ex) {
			rollbackTransaction(ut);
			throw new DataAccessException(
				"Cannot read dataobject for business object {0}",
				new Object[] { fullObjectName },
				ex);
		}

		commitTransaction(ut);*/
		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return guids;
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getDOByRelationGUID(String)
	 */
	public BO getComplexAttribute(String guidDO) throws DataAccessException {

		String method = "getComplexAttribute(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidDO});
		
		DataObject so = null;
		Attribute attr = null;

		/*attr = mmrProxy.getAttribute(guidDO);
		so = (DataObject)attr.getReferencedObject();

		BO tmp = convertBO(so);*/

		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return null;
	}

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getDOInstance(String, String)
	 */
	public BOInstance getComplexAttributeInstance(
		String guidBO,
		String guidBOInstance,
		String[] guidRelation,
		String[] guidDOInstance)
		throws DataAccessException, CAFPermissionException {

		String method = "getComplexAttributeInstance(String, String, String, String[], String[])";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {guidBO, guidBOInstance, guidRelation, guidDOInstance});
			
		BOInstance tmp = null;
		String attrName = null;
		Attribute attr = null;
		BO bo = null;
		String fullObjectName = null;
		IDependentObject obj = null;

		/*try {
			ut = getUserTransaction();
			beginTransaction(ut);

			//get the metadata of the business object
			BO tmpBO = getBOByGUID(guidBO);
			obj = (IDependentObject) getLocalInterface(tmpBO, guidBOInstance);

			//walk through the whole list of dependent objects  
			for (int i = 0; i < guidRelation.length; i++) {
				//get the metadata of the relation to the next dependent object
				attr = mmrProxy.getAttribute(guidRelation[i]);
				attrName = attr.getObjectName();

				//invoke the method to get an instance of the related dependent object
				Method m = obj.getClass().getMethod("get" + attrName, null);

				//the multiplicity tells what return type has to be expected
				if ((attr.getMaxOccurs().intValue()>=2) && (attr.getMaxOccurs().intValue()!=-1)) {
					//return type is collection
					Iterator it = ((Collection) m.invoke(obj, null)).iterator();
					while (it.hasNext()) {
						obj = (IDependentObject) it.next();
						if (obj.equals(guidDOInstance[i]))
							break;

						rollbackTransaction(ut);
						throw new DataAccessException("Wrong parameters", new Object[0]);
					}
				} else {
					obj = (IDependentObject) m.invoke(obj, null);
					if (!obj.getKey().equals(guidDOInstance[i])) {
						rollbackTransaction(ut);
						throw new DataAccessException("Wrong parameters", new Object[0]);
					}
				}
			}

			tmp = convertBOInstance(bo.getGuidBO(), bo.getProviderBO(), bo.getAppBO(), bo.getIdBO(), obj);
		} catch (CAFBaseException ex) {
			String[] args = {
			};
			rollbackTransaction(ut);
			throw new DataAccessException("", args, ex);
		} catch (Exception ex) {
			rollbackTransaction(ut);
			throw new DataAccessException(
				"Cannot read dataobject for business object {0}",
				new Object[] { fullObjectName },
				ex);
		}

		commitTransaction(ut);*/
		CAFPublicLogger.exiting(null, jARMRequest, method, location);

		return tmp;
	}

	/**
	 * Closes the connection to the model repository
	 */
	public void closeConnection() throws DataAccessException {
		String method = "closeConnection()";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[0]);
		
		RepositoryConnection.closeConnection();

		CAFPublicLogger.exiting(null, jARMRequest, method, location);
	}

	/*	public String[] getRelatedKMObjects(String guidBO)
			throws QueryException, ConnectionException, BONotFoundException {
			BOPublicLogger.LOC_CAF.entering("closeConnection");
	
			Map map = null;
			Set rids = new HashSet();
	
			//get metadata of object and its relations
			try {
				ShareableObject so = mmrProxy.getShareableObjectByGUID(guidBO);
				if (so == null)
					throw new BONotFoundException();
	
				Application app = so.getApplication();
				map = mmrProxy.getRelatedBOs(app.getProviderName(), app.getID(), so.getID());
				if (map == null)
					throw new BONotFoundException();
			} catch (BONotFoundException ex) {
				throw new BONotFoundException();
			}
	
			//extract all relations to km objects
			Relation[] objects = getAssociatedBOs(guidBO);
			Collection assKMObjects = new Vector();
			for (int i = 0; i < objects.length; i++) {
				if (objects[i].getBo().isKMObject())
					assKMObjects.add(objects[i].getGuidRelation());
			}
	
			//get all instances and loop through them
			String[] instances = getInstancesByGUIDPart(guidBO, "");
	
			for (int i = 0; i < instances.length; i++) {
				//loop through all roles of km objects
				Iterator it = assKMObjects.iterator();
				while (it.hasNext()) {
					String[] relInstances = getRelatedInstancesByGUIDPart(guidBO, (String) it.next(), instances[i], "");
					for (int k = 0; k < relInstances.length; k++)
						rids.add(relInstances[k]);
				}
			}
	
			BOPublicLogger.LOC_CAF.exiting();
			return (String[]) rids.toArray(new String[0]);
		}
		*/

	public String[] getRelatedKMObjects(String fullBOName)
		throws DataAccessException {
		String method = "getRelatedKMObjects(String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {fullBOName});
		
		Map map = null;
		Collection rids = new HashSet();
		//get metadata of object and its relations
		try {
			rids = instProxy.getRelatedInstances(fullBOName, kmBackEnd);
		} catch (Exception e) {
			throw new DataAccessException("CANNOT_READ_RELATED_BE", new Object[] {fullBOName}); //$NON-NLS-1$
		}
		
		CAFPublicLogger.exiting(null, jARMRequest, method, location);
		return (String[]) rids.toArray(new String[0]);
	}

	private IBusinessObject getLocalInterface(BO tmpBO, String guidInstance) throws DataAccessException {
		String method = "getLocalInterface(BO, String)";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {tmpBO, guidInstance});
		
		String fullObjectName = null;
		try {
			Object home = ctx.lookup(JNDI_PREFIX + tmpBO.getFullServiceName());
			location.debugT("Jndi lookup for business entity {0} successfull", new Object[] {fullObjectName});
						
			Method m = home.getClass().getMethod("create", null);
			Object local = m.invoke(home, null);			
			
			m = local.getClass().getMethod("read", new Class[] {String.class} );
			location.debugT("Calling read method of business entity service successfull", new Object[0]);			
			
			IBusinessObject obj = (IBusinessObject) m.invoke(local, new Object[] { guidInstance });
			location.debugT("Instantiating of business entity successfull", new Object[0]);
			return obj;
		} catch (Exception ex) {
			String[] args = { fullObjectName, guidInstance };
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, method, "Cannot read business entity: bo={0} instance={1}",
							args, ex);						
			throw new DataAccessException("CANNOT_READ_BE", args, ex); //$NON-NLS-1$
		}
	}
	
	public boolean[] checkRights(String boName, String userName, String guidInstance, String[] rightsToCheck) 
		throws DataAccessException {

		String method = "checkRights(String, String, String, String[])";
		CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {boName, userName, guidInstance, rightsToCheck});
			
		IBusinessObject obj =null;
		boolean[] results = new boolean[4];
		
		TransactionManager tr = null;
		try {
			if(BE_DOCUMENT_NAME.equals(boName)) {				
																
				Object objEJB = ctx.lookup( getDocumentBEServiceName() );
				Class cl = objEJB.getClass().getClassLoader().loadClass(VO_DOCUMENT);
				String parentFolder = guidInstance.substring(0, guidInstance.lastIndexOf('/'));
				String documentId = guidInstance.substring(guidInstance.lastIndexOf('/')+1);							

				//load document from JDO only to avoid circularity  (load -> loadFromKM -> checkRights)
				DocumentDataAccessService dataAccessService = DocumentDataAccessService.getInstance();
				
				/*
				 * Resource transaction is neccessary for correct work of current version of JDODataAccessService
				 * which is used in DocumentDataAccessService. If there is no active Resource transaction  
				 * JDODataAccessService closes resultSet immediatly after method complition.
				 */
				tr = getResourceTransaction();
				tr.begin();
				Object obj2 = dataAccessService.loadFromJDO(cl, documentId, parentFolder);
				tr.rollback();
				tr = null;
									
				try {
					results[0] = CAFPermissionUtil.remoteCheckAclPermission(obj2, userName , CAFPermissionName.read, boName);
				}
				catch(CAFPermissionException ex) {
					results[0] = false;
				}
				
				try {
					results[1] = CAFPermissionUtil.remoteCheckAclPermission(obj2, userName , CAFPermissionName.update, boName);
				}
				catch(CAFPermissionException ex) {
					results[1] = false;
				}
				
				try {
					results[2] = CAFPermissionUtil.remoteCheckAclPermission(obj2, userName , CAFPermissionName.write, boName);
				}
				catch(CAFPermissionException ex) {
					results[2] = false;
				}
				try {
					results[3] = CAFPermissionUtil.remoteCheckAclPermission(obj2, userName , CAFPermissionName.remove, boName);
				}
				catch(CAFPermissionException ex) {
					results[3] = false;
				}										
			}
		} catch (Exception ex) {				
			String[] args = { boName, guidInstance };
			CAFPublicLogger.categoryCAF.logThrowableT(Severity.ERROR, location, null, "Cannot read business entity for permission check: bo={0} instance={1}",
							args, ex);
			throw new DataAccessException("CANNOT_READ_BO", args, ex);
		}		
		finally {
			try {
					if (tr!=null) {
						tr.rollback();
					}
			} catch (Exception ex) {
				CAFPublicLogger.categoryCAF.logThrowableT(Severity.ALL, location, null, "Cannot rollback Resource trqansaction",
								null, ex);
			}
			CAFPublicLogger.exiting(null, jARMRequest, method, location);
		}
		
		return results;
	}	

	/**
	 * @see com.sap.caf.km.datasvc.ICAFDataService#getBO(java.lang.String, java.lang.String, java.lang.String)
	 */
	public BO getBO(String prov, String app, String bo)	throws DataAccessException {
			String method = "checkRights(String, String, String, String[])";
			CAFPublicLogger.entering(null, jARMRequest, method, location, new Object[] {prov, app, bo});
			
			DataObject so = mmrProxy.getDataObject(prov, app, bo);
			
			location.debugT("Reading metadata from mmr successfull", new Object[0]);

			BO tmp = convertBO(so);

			location.debugT("Converting metadata successfull", new Object[0]);
			
			CAFPublicLogger.exiting(null, jARMRequest, method, location);

			return tmp;
	}
	
	/**
	 * Gets Document BE Service name
	 */
	private String getDocumentBEServiceName() throws DataAccessException {
		if (m_beDocumentServiceName == null) {
			BusinessEntityInterface bei = mmrProxy.getDataObject(BE_DOCUMENT_NAME).getBusinessEntityInterface();

			Application app = bei.getApplication();

			m_beDocumentServiceName = JNDI_PREFIX + app.getProviderName() 
											+ "/" + app.getObjectName() 
											+ "/" + bei.getObjectName();
		}
		return m_beDocumentServiceName;
	}
	
	/**
	 * Gets Resource Transaction Manager.
	 * This transaction type is used in <i>checkRights()</i> method.
	 */
	private TransactionManager getResourceTransaction() throws NamingException {		
		return (TransactionManager)ctx.lookup( "TransactionManager" ); 
	}
}