/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_eclipse/src/com/sap/tc/jtools/jlint/jom/eclipse/JomRequestUtil.java#2 $
 */
 
package com.sap.tc.jtools.jlint.jom.eclipse;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;

/**
 * 
 * @author I024087 Denitsa Evtimova
 * @version 7.0.0
 */
public class JomRequestUtil {
  
  private static final IWorkspaceRoot wsRoot =
    ResourcesPlugin.getWorkspace().getRoot();

  private static Set searchPathClassDirs = new HashSet();
  private static Set searchPathJars = new HashSet();
  private static Set searchPathSrcDirs = new HashSet();

  private static Set allProjects = new HashSet();

  private JomRequestUtil() {
    // no instantiation
  }
  


 /**
  * gets all classpath entries (recursing into dependent projects)
  * of this project and converts them to JLin request parameters
  * 
  * @param project the Java project to read the classpath entries from 
  * @return JLin request parameters for source folders, class folders and jars
  */  
  public static ParameterInterface[] getClassPathParams(IJavaProject project) {
    reset();
    ParameterInterface[] params = new ParameterInterface[3];
    try {
      addToSearchPath(project, false);
    } catch (JavaModelException e) {
      ExceptionHandler.handle(e);
    }
    // construct classpath for parser:
    // 1. jars
    params[0] =
      ParameterTool.createParameter(
        JomTestVisitor.PARAMETER_JAVA_LIBRARIES,
        ParameterTool.PAR_TYPE_STRING_ARRAY,
        (String[]) searchPathJars.toArray(new String[searchPathJars.size()]));
    // 2. src folders
    params[1] =
      ParameterTool.createParameter(
        JomTestVisitor.PARAMETER_SOURCE_FOLDERS,
        ParameterTool.PAR_TYPE_STRING_ARRAY,
        (String[]) searchPathSrcDirs.toArray(
          new String[searchPathSrcDirs.size()]));
    // 3. class folders
    params[2] =
      ParameterTool.createParameter(
        JomTestVisitor.PARAMETER_CLASS_FOLDERS,
        ParameterTool.PAR_TYPE_STRING_ARRAY,
        (String[]) searchPathClassDirs.toArray(
          new String[searchPathClassDirs.size()]));
    return params;
  }

  private static void addToSearchPath(IJavaProject project, boolean isDepProject)
    throws JavaModelException {
    allProjects.add(project.getElementName());
    // recursively scan all dependent projects
    String[] dependendProjs = project.getRequiredProjectNames();
    if (dependendProjs != null) {
      for (int i = 0; i < dependendProjs.length; i++) {
        // avoid StackOverflow caused by cyclic project dependencies! 
        if (allProjects.contains(dependendProjs[i]))
          continue;
        addToSearchPath(
          JavaCore.create(wsRoot).getJavaProject(dependendProjs[i]),
          true);
      }
    }
    // add binary output folder of this project to searchpath
    IResource binOutputDir = wsRoot.findMember(project.getOutputLocation());
    if (binOutputDir != null) {
      IPath absoluteOutputDir = binOutputDir.getLocation();
      if (absoluteOutputDir != null) {
        searchPathClassDirs.add(absoluteOutputDir.toString());
      }
    }
    // add source- and library- classpathEntries to searchpath
    IClasspathEntry[] cpEntries = project.getResolvedClasspath(true);
    for (int i = 0; i < cpEntries.length; i++) {
      switch (cpEntries[i].getEntryKind()) {
        case IClasspathEntry.CPE_SOURCE :
          if (!isDepProject) {
            //source only for project under test, not dependent projects
            addSourceDir(project, cpEntries[i]);
          }
          break;
        case IClasspathEntry.CPE_LIBRARY :
          addLibrary(project, cpEntries[i]);
          break;
      }
    }
  }
  
  private static void addSourceDir(IJavaProject project, IClasspathEntry entry)
    throws JavaModelException {
    IResource cpEntryResource = wsRoot.findMember(entry.getPath());
    if (cpEntryResource == null)
      return;
    IPath absolutePath = cpEntryResource.getLocation();
    if (absolutePath == null)
      return;
    if (new File(absolutePath.toString()).exists()) {
      searchPathSrcDirs.add(absolutePath.toString());
    } else {
      throw new JavaModelException(
        new FileNotFoundException("file " + absolutePath.toString() + " doesn\'t exist!"),
        IStatus.ERROR);
    }
  }
  
  private static void addLibrary(IJavaProject project, IClasspathEntry entry) throws JavaModelException {
    IResource cpEntryResource = wsRoot.findMember(entry.getPath());
    boolean isClassesDir = false;
    String absPath = null;
    if (cpEntryResource == null) {
      // path is not in workspace, should be an absolute jar path
      absPath = entry.getPath().toString();
    } else {
      // path is jar or classes dir relative to the workspace root
      switch (cpEntryResource.getType()) {
        case IResource.FILE :
          isClassesDir = false;
          break;
        case IResource.FOLDER :
        case IResource.PROJECT :
          isClassesDir = true;
          break;
        default :
          throw new IllegalStateException(
            "unknown library type: " + cpEntryResource.getType());
      }
      IPath absolutePath = cpEntryResource.getLocation();
      if (absolutePath != null) { 
        absPath = absolutePath.toString();
      }
    }
    if (absPath != null) {
      if (new File(absPath).exists()) {
        if (isClassesDir) {
          searchPathClassDirs.add(absPath);
        } else {
          searchPathJars.add(absPath);
        }
      } else {
        throw new JavaModelException(
          new Exception("file " + absPath + " doesn\'t exist!"),
          IStatus.ERROR);
      }
    }
  }
  
  private static void reset() {
    allProjects.clear();
    searchPathClassDirs.clear();
    searchPathJars.clear();
    searchPathSrcDirs.clear();
  }



}
