/**
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_eclipse/src/com/sap/tc/jtools/jlint/jom/eclipse/JomRequest.java#3 $
 * Last Change: $Author: perforce $
 * $DateTime: 2004/04/29 10:11:12 $
 * 
 * Copyright(c) 2003, SAP AG, All Rights Reserved.
 */
package com.sap.tc.jtools.jlint.jom.eclipse;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;

import com.sap.tc.jtools.jlint.eclipse.launch.EclipseRequest;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.java.JavaFileArrayTestObject;
import com.sap.tc.jtools.jtci.FilePosition;
import com.sap.tc.jtools.jtci.PerformUnit;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestObjectSpecification;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;

/**
 * @author D037913
 * 
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class JomRequest extends EclipseRequest {

  private static final int MAX_TESTOBJECT_SIZE = 200;
  private IJavaElement[] javaElements;
  private Map            fileName2JavaElement = new HashMap();
  private Map            baseDir2FileListMap  = new HashMap();
  protected IJavaProject javaProject;
  protected List         relativeFileNames;

  /**
   * @param name
   * @param tests
   */
  public JomRequest(String name, Test[] tests, IJavaElement[] jElements) {
    super(name, tests);
    // sanity check
    IJavaProject javaProject = jElements[0].getJavaProject();
    for (int i = 0; i < jElements.length; i++) {
      if (!jElements[i].getJavaProject().equals(javaProject)) {
        throw new IllegalArgumentException(
            "all java elements must belong to the same project");
      }
    }
    this.javaProject = javaProject;
    this.javaElements = jElements;
    setSearchPathParameters();
    addSources();
  }

  private void addSources() {
    for (int i = 0; i < javaElements.length; i++) {
      try {
        switch (javaElements[i].getElementType()) {
          case IJavaElement.JAVA_PROJECT :
            traverseSources((IJavaProject) javaElements[i]);
            break;
          case IJavaElement.PACKAGE_FRAGMENT_ROOT :
          case IJavaElement.PACKAGE_FRAGMENT :
          case IJavaElement.COMPILATION_UNIT :
            traverseSources(javaElements[i]);
            break;
          case IJavaElement.TYPE :
            traverseSources(((IType) javaElements[i]).getCompilationUnit());
            break;
        }
      } catch (JavaModelException e) {
        ExceptionHandler.handle(e);
        return;
      }
    }
    Set entries = baseDir2FileListMap.entrySet();
    for (Iterator iter = entries.iterator(); iter.hasNext();) {
      Map.Entry entry = (Map.Entry) iter.next();
      String baseDir = (String) entry.getKey();
      String[] files = (String[]) ((List) entry.getValue())
          .toArray(new String[0]);
      // generate testobjects of maximum size
      for (int position = 0; position < files.length; position += MAX_TESTOBJECT_SIZE) {
        int length = (position + MAX_TESTOBJECT_SIZE > files.length) 
           ? files.length - position : 
             MAX_TESTOBJECT_SIZE;
        String[] filesChunk = new String[length];
        System.arraycopy(files, position, filesChunk, 0, length);
	      try {
	        addSource(new JavaFileArrayTestObject(baseDir, filesChunk));
	      } catch (InvalidTestObjectSpecification e) {
	        ExceptionHandler.handle(e);
	        return;
	      }
      }
    }
  }

  private void setSearchPathParameters() {
    ParameterInterface[] params = JomRequestUtil
        .getClassPathParams(javaProject);
    for (int i = 0; i < params.length; i++) {
      addParameter(params[i]);
    }
  }

  private void traverseSources(IJavaProject project) throws JavaModelException {
    IPackageFragmentRoot[] pkgFrgRoots = project.getPackageFragmentRoots();
    for (int i = 0; i < pkgFrgRoots.length; i++) {
      traverseSources(pkgFrgRoots[i]);
    }
  }

  private void traverseSources(IJavaElement el) throws JavaModelException {
    //pre-check
    if (el == null)
      return;
    if (el.getElementType() == IJavaElement.PACKAGE_FRAGMENT_ROOT) {
      // add source packagefragment roots only
      if (((IPackageFragmentRoot) el).isArchive()
          || ((IPackageFragmentRoot) el).getKind() != IPackageFragmentRoot.K_SOURCE) {
        return;
      }
    } else if (el.getElementType() == IJavaElement.PACKAGE_FRAGMENT) {
      // add source package fragments only
      if (((IPackageFragment) el).getKind() != IPackageFragmentRoot.K_SOURCE) {
        return;
      }
    } else if (el.getElementType() == IJavaElement.COMPILATION_UNIT) {
      ICompilationUnit cu = (ICompilationUnit) el;
      // compilation units may be working copies, we need the originals
      // (with underlying resource)
      if (cu.isWorkingCopy()) {
        el = cu.getOriginalElement();
      }
    }
    String baseDir = el.getAncestor(IJavaElement.PACKAGE_FRAGMENT_ROOT)
        .getResource().getLocation().toString();
    String[] sources = getAllRelativeFileNames(el);
    if (sources == null || sources.length == 0) {
      return;
    }
    for (int i = 0; i < sources.length; i++) {
      fileName2JavaElement.put(sources[i], el);
    }
    List fileList = (List) baseDir2FileListMap.get(baseDir);
    if (fileList == null) {
      fileList = new ArrayList(sources.length);
      baseDir2FileListMap.put(baseDir, fileList);
    }
    fileList.addAll(Arrays.asList(sources));
  }

  private String[] getAllRelativeFileNames(IJavaElement el) {
    relativeFileNames = new ArrayList();
    doElement(el);
    return (String[]) relativeFileNames.toArray(new String[relativeFileNames
        .size()]);
  }

  protected void doElement(IJavaElement el) {
    if (el instanceof ICompilationUnit) {
      ICompilationUnit cUnit = (ICompilationUnit) el;
      int baseDirSegments = 0;
      try {
        baseDirSegments = cUnit.getAncestor(IJavaElement.PACKAGE_FRAGMENT_ROOT)
            .getCorrespondingResource().getLocation().segmentCount();
        relativeFileNames.add(el.getCorrespondingResource().getLocation()
            .removeFirstSegments(baseDirSegments).setDevice(null).toString());
      } catch (JavaModelException e) {
        ExceptionHandler.handle(e);
      }
      return;
    }
    if (el instanceof IParent) {
      IJavaElement[] children = null;
      try {
        children = (IJavaElement[]) ((IParent) el).getChildren();
      } catch (JavaModelException e) {
        ExceptionHandler.handle(e);
      }
      if (null == children) {
        return;
      }
      for (int i = 0; i < children.length; i++) {
        doElement(children[i]);
      }
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.eclipse.extension.IEclipseRequest#testObjectCount()
   */
  public int testObjectCount() {
    int noOfSrc = 0;
    PerformUnit[] units = this.getPerformUnits();
    for (int i = 0; i < units.length; i++) {
      TestObject[] testObjs = units[i].getSources();
      for (int j = 0; j < testObjs.length; j++) {
        noOfSrc += testObjs[j].getComponentCount();
      }
    }
    return noOfSrc;
  }

  /*
   * (non-Javadoc)
   * 
   * @see com.sap.tc.jtools.jlint.eclipse.extension.IEclipseRequest#getProject()
   */
  public IProject getProject() {
    return javaProject.getProject();
  }

  /**
   * @param testObjectPosition
   * @return
   */
  public Object getData(String testObjectID,
      TestObjectPosition testObjectPosition) {
    if (!(testObjectPosition instanceof FilePosition))
      return null;
    return fileName2JavaElement.get(((FilePosition) testObjectPosition)
        .getFile());
  }
}
