/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.events;

import com.sapportals.wcm.WcmException;

/**
 * The event broker is a mediator between event senders ({@link IEventSender})
 * and receivers ({@link IEventReceiver}). All senders and receivers must
 * register themselfs with a broker. All receivers will receive events from all
 * senders but it is possible to register for different event "templates" (event
 * class and type). Usually a sender will instanciate its own broker and
 * register itself. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Markus Breitenfelder
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/events/IEventBroker.java#8
 *      $
 */
public interface IEventBroker {

  public final static int PRIO_MAX = 1;
  public final static int PRIO_MIN = 99;

  /**
   * Sends the specified event to all receivers that have registered for this
   * kind of event.
   *
   * @param event The event so send.
   * @param sender The event sended instance (the caller).
   * @exception WcmException If the event sender is not registered.
   */
  public void send(IEvent event, IEventSender sender)
    throws WcmException;

  /**
   * Registers an event sender instance.
   *
   * @param sender The event sender.
   * @exception WcmException Exception raised in failure situation
   */
  public void register(IEventSender sender)
    throws WcmException;

  /**
   * Registers an event receiver. The receiver will only receive events which
   * match the template event. This depends on the matching rules imlemented in
   * the events {@link IEvent#isLike(IEvent)} method. Usually the event's instance and
   * type are relevant. <p>
   *
   * It is allowed to register the same event receiver instance several times
   * with different templates.
   *
   * @param receiver The event receiver to register
   * @param template The event template
   * @exception WcmException
   */
  public void register(IEventReceiver receiver, IEvent template)
    throws WcmException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has an additional {@link IEventMapper}
   * argument.
   *
   * @param receiver The event receiver to register
   * @param template The event template
   * @param mapping The event mapper
   * @exception WcmException
   * @see IEventMapper
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping)
    throws WcmException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has two additional arguments for priority
   * and asynchronous events. <p>
   *
   * The priority controls the order of multible receivers which have registered
   * for the same event(s). Receivers with higher priority (smaller values) will
   * receive an event before receivers with lower priority (greater values). <p>
   *
   * An event receiver can choose to receive events asynchronously. This means
   * that a dedicated event queue and sender thread is created for each receiver
   * and template. The event sender will not be blocked, that means the send()
   * method puts the event into the queue and returns immediatelly.
   *
   * @param receiver The event receiver to register
   * @param template The event template
   * @param priority The priority of this receiver and template.
   * @param async If true, the receiver will receive the events on a seperate
   *      thread.
   * @exception WcmException
   */
  public void register(IEventReceiver receiver, IEvent template, int priority, boolean async)
    throws WcmException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method.
   *
   * @param receiver The event receiver to register
   * @param template The event template
   * @param mapping The event mapper
   * @param priority The priority of this receiver and template.
   * @param async If true, the receiver will receive the events on a seperate
   *      thread.
   * @exception WcmException
   * @see #register(IEventReceiver, IEvent, IEventMapper)
   * @see #register(IEventReceiver, IEvent, int, boolean)
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping, int priority, boolean async)
    throws WcmException;

  /**
   * Unregisteres the event sender instance.
   *
   * @param sender The event sender to unregister
   * @exception WcmException
   */
  public void unregister(IEventSender sender)
    throws WcmException;

  /**
   * Unregisters an event receiver/template tuple.
   *
   * @param receiver The event receiver
   * @param template The event template
   * @exception WcmException
   */
  public void unregister(IEventReceiver receiver, IEvent template)
    throws WcmException;

  /**
   * Suspend the event receiver. The broker will stop delivering any events
   * until {@link #resume(IEventReceiver)} is called.
   *
   * @param receiver The receiver to suspend
   */
  public void suspend(IEventReceiver receiver);

  /**
   * Resumes a previously suspended event receiver. The broker will continue
   * delivering events to the receiver.
   *
   * @param receiver The receiver to resume
   */
  public void resume(IEventReceiver receiver);

  public void hold(IEventReceiver receiver, boolean collect);

  public SenderMode getMode(IEventReceiver receiver);

  public IEventList getHoldEvents(IEventReceiver receiver);

  public IEventList clearHoldEvents(IEventReceiver receiver);

  public IEventList getEvents();

}
