/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.service;

import com.sapportals.wcm.WcmObject;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.ResourceException;

import com.sapportals.wcm.repository.manager.IRepositoryManager;
import com.sapportals.wcm.service.IService;

import java.util.Collection;

/**
 * Defines an interface for a repository service factory. This factory creates
 * instances of repository dependent services. It also provides access to
 * repository independet services. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author frank.renkes@sapportals.com
 * @version $Id:$
 */
public interface IRepositoryServiceFactory {

  /**
   * Get a list of all registered service managers for a given repository
   * manager.
   *
   * @param repository TBD: Description of the incoming method parameter
   * @return Collection of <code>IRepositoryService</code> references.
   * @exception ResourceException If there was a problem while allocating the
   *      managers.
   */
  public Collection getAllRepositoryServices(IRepositoryManager repository)
    throws ResourceException;

  /**
   * Returns a specific service manager instance for a given repository manager.
   *
   * @param repository A repository manager reference
   * @param servicetype A service type constant
   * @return A <code>IRepositoryService</code> reference.
   * @exception ResourceException If there was a problem while allocating the
   *      manager.
   */
  public IRepositoryService getRepositoryService(IRepositoryManager repository, String servicetype)
    throws ResourceException;

  /**
   * Checks if the service is avaible.
   *
   * @param repository TBD: Description of the incoming method parameter
   * @param servicetype TBD: Description of the incoming method parameter
   * @return repositoryServiceAvailable
   * @exception ResourceException If there was a problem while allocating the
   *      manager.
   */
  public boolean isRepositoryServiceAvailable(IRepositoryManager repository, String servicetype)
    throws ResourceException;

  /**
   * Get all services for a given resource.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return A Collection of <code>IRepositoryService</code> s.
   * @exception ResourceException If there was a problem while allocating the
   *      managers.
   */
  public Collection getAllRepositoryServices(IResource resource)
    throws ResourceException;

  /**
   * Get a specific service for a given resource.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param servicetype TBD: Description of the incoming method parameter
   * @return A <code>IRepositoryService</code> .
   * @exception ResourceException If there was a problem while allocating the
   *      manager.
   */
  public IRepositoryService getRepositoryService(IResource resource, String servicetype)
    throws ResourceException;

  /**
   * Checks if the service is avaible.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param servicetype TBD: Description of the incoming method parameter
   * @return repositoryServiceAvailable
   * @exception ResourceException If there was a problem while allocating the
   *      manager.
   */
  public boolean isRepositoryServiceAvailable(IResource resource, String servicetype)
    throws ResourceException;

  /**
   * Get a service instance by ID. The method will return the service instance
   * for a manager this service's type is registered.
   *
   * @param id The service manager ID
   * @return A <code>IRepositoryService</code> .
   * @exception ResourceException If there was a problem while allocating the
   *      manager.
   */
  public IRepositoryService getRepositoryService(String id)
    throws ResourceException;

  /**
   * Returns the instance of a service.
   *
   * @param serviceID The service ID as specified in configuration.
   * @return The service
   * @exception ResourceException Exception raised in failure situation
   */
  public IService getService(String serviceID)
    throws ResourceException;

  /**
   * Returns a collection of all defined services
   *
   * @return A Collection with <code>IService</code> instances - empty
   *      collection if no global services are available
   * @exception ResourceException Exception raised in failure situation
   */
  public Collection getAllServices()
    throws ResourceException;
}
