/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.security;

import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceList;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.manager.IAclSecurityManager;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.uri.RID;

/**
 * An IResourceAclManager administers and persists Access Control Lists (ACL)
 * <br>- IResourceAclManagers are used by {@link IAclSecurityManager}s
 * <br>- IResourceAclManagers use {@link IAclManager}s (one to one)
 * <br>- IResourceAclManagers operate on {@link IResource}s
 * <br>- {@link IAclManager}s operate on arbitrary objects which are identified by an unique
 * ID
 * <br>- IResourceAclManagers pass the {@link RID} of the {@link IResource}s as object ID to the
 * {@link IAclManager}s
 * <br>- IResourceAclManagers pass the logged in user (from the
 * resource context) as the 'caller' to {@link IAclManager}s
 * <br>- IResourceAclManagers use
 * the wrappers {@link IResourceAcl} (for {@link IAcl}), {@link IResourceAclEntry} (for {@link IAclEntry}) in
 * order to hide the calls of the {@link IAclManager} (so that nobody can pass a fake
 * 'caller')
 * <p>
 * Abbreviations
 * <br>ACL: Access Control List
 * <br>ACE: Access Control List Entry
 * <p>
 * Copyright (c) SAP AG 2001-2004
 */
public interface IResourceAclManager {
  /**
   * create a new ACE for an ACL to grant or deny a permission to a principal
   *
   * @param principal the principal
   * @param negative true if the entry denies a permission, false if it grants
   *      (denials are currently unsupported)
   * @param permission the permission
   * @param sortIndex the position of the ACE in an ACL (important only with
   *      denials)
   * @return the newly created ACE
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the permission was created by a foreign
   *      IResourceAclManager
   * @exception UnsupportedOperationException negative ACEs are currently not
   *      supported (request support by using the
   *      areNegativeAclEntriesSupported() method)
   */
  public IResourceAclEntry createAclEntry(IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex)
    throws AclPersistenceException, UnsupportedOperationException, InvalidClassException;

  /**
   * check whether negative ACEs are supported
   *
   * @return true iff negative ACEs are supported
   */
  public boolean areNegativeAclEntriesSupported();

  /**
   * create a new ACL for a resource - initial owner is the user in the resource
   * context - inherited ACEs are added - if the resource inherits an ACL the
   * user must be an owner - fails, if the resource already has an ACL assigned
   *
   * @param resource the resource
   * @return the newly created ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of an inherited ACL
   * @exception AclExistsException an ACL already exists for the resource
   * @exception ResourceException the resource URI could not be determined
   */
  public IResourceAcl createAcl(IResource resource)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException;

  /**
   * assign an ACL from a foreign IResourceAclManager to a local resource -
   * foreign ACEs which hold permissions that are not supported by the local
   * IResourceAclManager are ignored - locally inherited ACEs are added - the
   * owners are maintained - if the resource inherits an ACL the user must be an
   * owner - fails, if the resource already has an ACL assigned
   *
   * @param foreignAcl the foreign ACL
   * @param resource the local resource
   * @return true iff the ACL has been assigned
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of an inherited ACL
   * @exception AclExistsException an ACL already exists for the resource
   * @exception ResourceException the resource URI could not be determined
   */
  public boolean assignForeignAcl(IResourceAcl foreignAcl, IResource resource)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException;

  /**
   * get the ACL of a specific resource
   *
   * @param resource the resource
   * @return the ACL of the resource or null in case none is assigned yet
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception ResourceException the resource URI could not be determined
   */
  public IResourceAcl getAcl(IResource resource)
    throws AclPersistenceException, ResourceException;

  /**
   * get the ACL that a resource inherits by its ancestors
   *
   * @param resource the resource
   * @return the inherited ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception ResourceException the resource URI could not be determined
   */
  public IResourceAcl getInheritedAcl(IResource resource)
    throws AclPersistenceException, ResourceException;

  /**
   * remove the ACL of a specific resource
   *
   * @param resource the resource
   * @return true iff an ACL did exist and was removed
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL
   * @exception ResourceException the resource URI could not be determined
   */
  public boolean removeAcl(IResource resource)
    throws AclPersistenceException, NotAuthorizedException, ResourceException, ResourceException;

  /**
   * remove an ACL
   *
   * @param acl the ACL
   * @return true iff the ACL was removed
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner of the access
   *      control list
   * @exception InvalidClassException the acl was created by a foreign resource
   *      acl manager
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean removeAcl(IResourceAcl acl)
    throws AclPersistenceException, NotAuthorizedException, InvalidClassException, ResourceException;

  /**
   * get a list of object types which are supported by the IResourceAclManager
   * for resources
   *
   * @return the object types
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IObjectTypeList getSupportedObjectTypes()
    throws AclPersistenceException;

  /**
   * get the object type of a resource
   *
   * @param resource the resource
   * @return the object type
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception ResourceException the resource URI could not be determined
   */
  public IObjectType getObjectType(IResource resource)
    throws AclPersistenceException, ResourceException;

  /**
   * get a list of permissions which are supported by the IResourceAclManager
   * for a specific resource
   *
   * @param resource the resource
   * @return the permissions
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception ResourceException the resource URI could not be determined
   */
  public IAclPermissionList getSupportedPermissions(IResource resource)
    throws AclPersistenceException, ResourceException;

  /**
   * get the IAclPermission object of the (supported) permission with a given
   * name
   *
   * @param name the name of the permission
   * @return the permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAclPermission getPermission(String name)
    throws AclPersistenceException;

  /**
   * assign a permission to an object type
   *
   * @param objectType the object type
   * @param permission the permission
   * @return true iff the permission was assigned successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   */
  public boolean addSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException;

  /**
   * remove the assignment of a permission to an object type - a supported
   * permission can only be removed if it is not predefined and not used in an
   * ACL
   *
   * @param objectType the object type
   * @param permission the permission
   * @return true iff the assignment was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean removeSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException;

  /**
   * create a new permission (the permission can not be used until it is
   * assigned to an object type by calling the addSupportedPermission() method)
   *
   * @param name the permission name
   * @return the newly created permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PermissionExistsException a permission with that name exists
   *      already
   */
  public IAclPermission createPermission(String name)
    throws AclPersistenceException, PermissionExistsException;

  /**
   * remove a permission - a permission can only be removed if it is not
   * predefined and not used in an ACL
   *
   * @param permission the permission
   * @return true iff the permission was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   * @exception PermissionUsedException the permission is used in some acl and
   *      thus not allowed to be changed
   */
  public boolean removePermission(IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException;

  /**
   * check whether a permission is used in an ACL
   *
   * @param permission the permission
   * @return true iff the permission is used in an ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isPermissionUsedInAcl(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * check whether the IResourceAclManager is read only (no creation or
   * modification of ACLs is allowed)
   *
   * @return true iff the IResourceAclManager is read only
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;

  /**
   * check whether the ACLs are up to date (compared to the database)
   *
   * @param acls the ACLs to check
   * @return a boolean array containing true for the ACLs which are up to date
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean[] areAclsUpToDate(IResourceAcl[] acls)
    throws AclPersistenceException;

  /**
   * remove the ACLs of the descendants of the resource
   *
   * @return null if all acls could be removed successfully otherwise a list of
   *      all Resources which failed operation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @exception NoAclException Exception raised in failure situation
   * @exception InvalidClassException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public IResourceList propagateAcl_Remove(IResource resource)
    throws AclPersistenceException, ResourceException, NoAclException, InvalidClassException, NotAuthorizedException;

  /**
   * Returns a value representing the state (timestamp or modifycounter) of all
   * persisted acl entries. If a negative number is returned this method is not
   * supported or an error occured.
   *
   * @return dBVersion
   */
  public long getDBVersion();

}
