/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.security;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.repository.*;

import com.sapportals.wcm.util.acl.*;

import java.util.*;

/**
 * An IResourceAcl is a wrapper for an IAcl which uses an URI as object ID for
 * the IAcl and the user in resource context as 'callers' in the IAcl methods -
 * IResourceAcls are assigned to resources - IResourceAcls hold
 * IResourceAclEntries Abbreviations ACL Access Control List ACE Access Control
 * List Entry
 */
public interface IResourceAcl {
  /**
   * add a new owner to the ACL (only owners are allowed to change the ACL)
   *
   * @param owner the new owner
   * @return true iff the owner was added successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL (only owners are allowed to change the owner list)
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean addOwner(IUMPrincipal owner)
    throws AclPersistenceException, NotAuthorizedException, ResourceException;

  /**
   * remove an owner from the ACL (only owners are allowed to change the ACL)
   *
   * @param owner the owner to be removed
   * @return true iff the owner was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL (only owners are allowed to change the owner list)
   * @exception LastOwnerException attempt to remove the last owner (which is
   *      forbidden)
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean removeOwner(IUMPrincipal owner)
    throws AclPersistenceException, NotAuthorizedException, LastOwnerException, ResourceException;

  /**
   * check whether a principal is an owner of the ACL
   *
   * @param principal the principal
   * @return true iff the principal is an owner of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isOwner(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * check whether a principal is authorized to change the ACL
   *
   * @param principal the principal
   * @return true iff the principal is an owner of the ACL or a member of an
   *      owner of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isAuthorized(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * get a list of the owners of the ACL
   *
   * @return a list of the owners of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IUMPrincipalList getOwners()
    throws AclPersistenceException;

  /**
   * add an ACE to the ACL
   *
   * @param aclEntry the ACE
   * @return true iff the ACE was added successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the ACE was created by a foreign
   *      IResourceAclManager
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL
   * @exception AlreadyAssignedToAclException the ACE is already assigned to
   *      another ACL
   * @exception PermissionNotSupportedException the permission of the ACE is not
   *      supported for the object type of the ACL's resource
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean addEntry(IResourceAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException, AlreadyAssignedToAclException, PermissionNotSupportedException, ResourceException;

  /**
   * remove an ACE from the ACL
   *
   * @param aclEntry the ACE
   * @return true iff the ACE was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the ACE was created by a foreign
   *      IResourceAclManager
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean removeEntry(IResourceAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException, ResourceException;

  /**
   * get a list of the ACEs of the ACL
   *
   * @return a list of the ACEs of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IResourceAclEntryList getEntries()
    throws AclPersistenceException;

  /**
   * get a list of the ACEs of the ACL concerning a specific principal
   *
   * @param principal the principal
   * @return a list of the ACEs
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IResourceAclEntryList getEntries(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * check whether the ACL grants a specific permission to a user
   *
   * @param permission the permission
   * @param principal TBD: Description of the incoming method parameter
   * @return true iff the permission is granted
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean checkPermission(IUMPrincipal principal, IAclPermission permission)
    throws AclPersistenceException;

  /**
   * check whether the ACL is read only
   *
   * @return true iff the ACL is read only
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;

  /**
   * get the resource the ACL is assigned to
   *
   * @return the resource
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IResource getResource()
    throws AclPersistenceException;

  /**
   * lock the ACL (the lock is cooperative and affects only the result of
   * subsequent calls to lock() or isLocked())
   *
   * @return true iff the ACL was locked successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      an owner of the ACL
   */
  public boolean lock()
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * check whether the ACL is locked
   *
   * @return true iff the ACL is locked
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isLocked()
    throws AclPersistenceException;

  /**
   * get the user which locked the ACL
   *
   * @return the user which locked the ACL or null in case the ACL is not locked
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IUMPrincipal getLockingUser()
    throws AclPersistenceException;

  /**
   * unlock the ACL
   *
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context is not
   *      the user which locked the ACL
   */
  public void unlock()
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * check whether the ACL is up to date (check change level of persisted ACL)
   * (used to invalidate ACL caches)
   *
   * @return true iff the persisted ACL has the same change level
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isUpToDate()
    throws AclPersistenceException;
}
