/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;

import com.sapportals.wcm.crt.configuration.IConfiguration;
import java.util.*;

/**
 * Provides a tool for making diffs of configuration of child components.
 */
public final class CmConfigDiffTool {

  /**
   * Helper method for reconfigure() implementation of components that are
   * containers: Returns a diff object for the configuration data of all added,
   * removed and modified child components.
   *
   * @param childConfigClassName The path of the configuration framwork URI of
   *      the child component to diff, e.g. "/cm/utilities/caches/MemoryCache".
   * @param config The current configuration data of the component.
   * @param newConfig The new configuration data received as argument of the
   *      reconfigure() method.
   * @return A diff object with configuration instances of added, removed and
   *      modified child components. The name of the configuration ({@link
   *      IConfiguration#getName()}) is the value of the ID property of the child
   *      component's configurable.
   */
  public final static ChildConfigDiff diffChildConfigurations(
    String childConfigClassName,
    IConfiguration config,
    IConfiguration newConfig) {
    ArrayList added = new ArrayList();
    ArrayList removed = new ArrayList();
    ArrayList modified = new ArrayList();
    HashMap attDiffs = new HashMap();

    IConfiguration oldParent = config.getChild(childConfigClassName);
    IConfiguration newParent = newConfig.getChild(childConfigClassName);
    IConfiguration[] oldChildren = oldParent.getChildren();
    IConfiguration[] newChildren = newParent.getChildren();

    for (int i = 0; i < oldChildren.length; i++) {
      if (null == newParent.getChild(oldChildren[i].getName(), false)) {
        removed.add(oldChildren[i]);
      }
    }
    for (int i = 0; i < newChildren.length; i++) {
      IConfiguration old = oldParent.getChild(newChildren[i].getName(), false);
      if (old == null) {
        added.add(newChildren[i]);
      } else {
        // diff attribute list
        String[] newNames = newChildren[i].getAttributeNames();
        String[] oldNames = old.getAttributeNames();

        ArrayList addedAttr = new ArrayList();
        ArrayList removedAttr = new ArrayList();
        ArrayList modifiedAttr = new ArrayList();
        HashMap oldValues = new HashMap();
        HashMap newValues = new HashMap();

        for (int n = 0; n < oldNames.length; n++) {
          if (null == newChildren[i].getAttribute(oldNames[n], null) && null != old.getAttribute(oldNames[n], null)) {
            removedAttr.add(oldNames[n]);
          }
        }
        for (int n = 0; n < newNames.length; n++) {
          String oldValue = old.getAttribute(newNames[n], null);
          String newValue = newChildren[i].getAttribute(oldNames[n], null);
          if (oldValue == null && newValue != null) {
            addedAttr.add(newNames[n]);
          } else if (newValue != null && oldValue != null) {
            if (!newValue.equals(oldValue)) {
              modifiedAttr.add(newNames[n]);
              oldValues.put(newNames[n], oldValue);
              newValues.put(newNames[n], newValue);
            }
          }
        }

        if (removedAttr.size() > 0 || addedAttr.size() > 0 || modifiedAttr.size() > 0) {
          modified.add(newChildren[i]);
          attDiffs.put(
            newChildren[i].getName(),
            new CmConfigDiffTool.AttributeDiff(removedAttr, addedAttr, modifiedAttr, oldValues, newValues));
        }
      }
    }

    return new CmConfigDiffTool.ChildConfigDiff(removed, added, modified, attDiffs);
  }

  /**
   * The diff result of {@link CmConfigDiffTool#diffChildConfigurations(String, IConfiguration, IConfiguration)}.
   */
  public final static class ChildConfigDiff {
    private final IConfiguration[] removed;
    private final IConfiguration[] added;
    private final IConfiguration[] modified;
    private final Map attributeDiffs;

    ChildConfigDiff(Collection removed, Collection added, Collection modified, Map attributeDiffs) {
      this.attributeDiffs = attributeDiffs;
      this.removed = (IConfiguration[])removed.toArray(new IConfiguration[removed.size()]);
      this.added = (IConfiguration[])added.toArray(new IConfiguration[added.size()]);
      this.modified = (IConfiguration[])modified.toArray(new IConfiguration[modified.size()]);
    }

    /**
     * Returns an array of removed child configurations.
     *
     * @return removedChildren
     */
    public IConfiguration[] getRemovedChildren() {
      return this.removed;
    }

    /**
     * Returns an array of added child configurations.
     *
     * @return addedChildren
     */
    public IConfiguration[] getAddedChildren() {
      return this.added;
    }

    /**
     * Returns an array of modified child configurations
     *
     * @return modifiedChildren
     */
    public IConfiguration[] getModifiedChildren() {
      return this.modified;
    }

    /**
     * Returns the diff object for the attributes of the configuration child
     * node with the specified name (Configurable ID).
     *
     * @param name TBD: Description of the incoming method parameter
     * @return attributeDiff
     */
    public AttributeDiff getAttributeDiff(String name) {
      return (AttributeDiff)this.attributeDiffs.get(name);
    }
  }

  /**
   * The diff result for the attributes of new and old child configuration node.
   */
  public final static class AttributeDiff {
    private final String[] removed;
    private final String[] added;
    private final String[] modified;
    private final Map oldValues;
    private final Map newValues;

    public AttributeDiff(Collection removed, Collection added, Collection modified, Map oldValues, Map newValues) {
      this.removed = (String[])removed.toArray(new String[removed.size()]);
      this.added = (String[])added.toArray(new String[added.size()]);
      this.modified = (String[])modified.toArray(new String[modified.size()]);
      this.oldValues = oldValues;
      this.newValues = newValues;
    }

    public String[] getRemovedAttributeNames() {
      return this.removed;
    }

    public String[] getAddedAttributeNames() {
      return this.added;
    }

    public String[] getModifiedAttributeNames() {
      return this.modified;
    }

    public String getOldValue(String attributeName) {
      return this.oldValues.get(attributeName).toString();
    }

    public String getNewValue(String attributeName) {
      return this.newValues.get(attributeName).toString();
    }
  }

}
