/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.IResourceFactory;
import com.sapportals.wcm.repository.IResourceListIterator;
import com.sapportals.wcm.repository.IVersionHistory;
import com.sapportals.wcm.repository.NotSupportedException;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.repository.enum.SupportedOption;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;

import java.util.Date;

/**
 * Some utils to handle resources from an event... Copyright (c) SAP Portals
 * Europe GmbH 2001
 *
 * @author paul.goetz@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/repository/framework/api/com/sapportals/wcm/repository/manager/ResourceEventUtils.java#10
 *      $
 * @review-reader Hansi Richstein
 * @review-date 09.01.2002
 */
public class ResourceEventUtils {

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static Location g_Log = Location.getLocation(com.sapportals.wcm.repository.manager.ResourceEventUtils.class);

  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  public static boolean isVisible(IResource resource,
    IResourceContext context)
    throws WcmException {

    if (resource == null) {
      return false;
    }

    RID rid = resource.getRID();

    IResource visibleResource = ResourceFactory.getInstance().getResource(rid, context);
    if (visibleResource == null) {
      g_Log.debugT("isVisible(70)", "ResourceEventUtils: resource " + rid + " not visible");
      return false;
    }
    else {
	  if( !resource.isCollection() ) {	  // this only works for files, so check for folders
		  if ((resource.getRepositoryManager().getSupportedOptions(visibleResource)).isSupported(SupportedOption.VERSIONING)) {
			Date visibleResourceLMDate = visibleResource.getLastModified();
			if (visibleResourceLMDate != null) {
			  Date resourceLMDate = resource.getLastModified();
			  if (resourceLMDate != null) {
				if( visibleResourceLMDate.before(resourceLMDate) ) {
			      return false;
				}
			  }
			}
		  }
		  return true;
      } else {
		  return true;   // Versioning doesn't work for folders
	  }
    }

  }

  /*
   * d036774: Check this. IMHO VersionHistoryList is stored from oldest to newest Version,
   * not as supposted here from newest to oldest.
   */
  public static boolean isVisibleWithLaterVersion(IResource resource,
    IResourceContext context)
    throws WcmException {

    if (resource == null) {
      return false;
    }

    RID rid = resource.getRID();
    try {
      IVersionHistory versionHistory = resource.getVersionHistory();
      if (versionHistory == null) {
        g_Log.debugT("isVisibleWithLaterVersion(104)", "ResourceEventUtils: no version history for " + rid);
        return false;
      }
      IResourceListIterator iterator = versionHistory.listIterator();
      if (!iterator.hasNext()) {
        g_Log.debugT("isVisibleWithLaterVersion(109)", "ResourceEventUtils: no current version for " + rid);
        return false;
      }
      /*
       * IResource currentVersion =
       */
      iterator.next();
      if (!iterator.hasNext()) {
        g_Log.debugT("isVisibleWithLaterVersion(117)", "ResourceEventUtils: no later version for " + rid);
        return false;
      }
      IResource laterVersion = iterator.next();
      if (laterVersion == null) {
        g_Log.debugT("isVisibleWithLaterVersion(122)", "ResourceEventUtils: no later version uri for " + rid);
        return false;
      }
      rid = laterVersion.getRID();
    }
    catch (NotSupportedException e) {
      //$JL-EXC$
      g_Log.debugT("isVisibleWithLaterVersion(128)", "ResourceEventUtils: no versioning for " + rid);
      return false;
    }

    IResource visibleResource = ResourceFactory.getInstance().getResource(rid, context);
    if (visibleResource == null) {
      g_Log.debugT("isVisibleWithLaterVersion(134)", "ResourceEventUtils: resource " + rid + " not visible");
      return false;
    }

    return true;
  }

}
