/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import com.sap.netweaver.bc.rf.common.context.*;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.events.IEvent;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.RID;

import com.sapportals.wcm.util.resource.ResourceBundles;

import java.util.Locale;

/**
 * Implements the events known for resources. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author markus.breitenfelder@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/repository/framework/api/com/sapportals/wcm/repository/manager/ResourceEvent.java#14
 *      $
 * @review-reader Hansi Richstein
 * @review-date 09.01.2002
 */

public class ResourceEvent implements IResourceEvent {

  /**
   * any kind of event... that means: an undefined event!
   */
  public final static int ALL = 0;

  /**
   * signals a LockManager.lock()
   */
  public final static int UNLOCK = 101;

  /**
   * signals a LockManager.unlock()
   */
  public final static int LOCK = 102;

  /**
   * signals a PropertyManager.delete()
   */
  public final static int PROPERTY_DELETE = 201;

  /**
   * signals a PropertyManager.set()
   */
  public final static int PROPERTY_SET = 202;

  /**
   * signals a PropertyManager.get()
   */
  public final static int PROPERTY_GET = 203;

  /**
   * signals a NamespaceManager.delete()
   */
  public final static int DELETE = 301;

  /**
   * signals a NamespaceManager.createChild()
   */
  public final static int CREATE_CHILD = 311;

  /**
   * signals a NamespaceManager.createCollection()
   */
  public final static int CREATE_COLLECTION = 321;

  /**
   * signals a NamespaceManager.createLink()
   */
  public final static int CREATE_LINK = 331;

  /**
   * signals a NamespaceManager.updateLink()
   */
  public final static int UPDATE_LINK = 332;

  /**
   * signals a NamespaceManager.updateContent()
   */
  public final static int SET = 302;

  /**
   * signals a NamespaceManager.getContentInputStream()
   */
  public final static int GET = 313;

  /**
   * signals a NamespaceManager.getChildren()
   */
  public final static int GET_CHILDREN = 323;

  /**
   * signals a NamespaceManager.move()
   */
  public final static int MOVE = 340;

  /**
   * signals a NamespaceManager.move()
   */
  public final static int RENAME = 341;

  /**
   * signals a NamespaceManager.copy()
   */
  public final static int COPY = 342;

  /**
   * signals a VersioningManager.checkOut()
   */
  public final static int CHECKOUT = 350;

  /**
   * signals a VersioningManager.checkIn()
   */
  public final static int CHECKIN = 351;

  /**
   * Signals a VersioningManager.undoCheckOut()
   */
  public final static int UNDO_CHECKOUT = 352;

  /**
   * Signals that versioning was activated for a resource
   */
  public final static int ENABLE_VERSIONING = 353;

  /**
   * Before IContentManager.updateContent()
   */
  public final static int PRE_SET = 400;

  /**
   * Before INamespaceManager.delete()
   */
  public final static int PRE_DELETE = 401;

  /**
   * Before INamespaceManager.rename()
   */
  public final static int PRE_RENAME = 402;

  /**
   * Before INamespaceManager.delete()
   */
  public final static int PRE_MOVE = 403;

  /**
   * Before IVersioningManager.checkIn()
   */
  public final static int PRE_CHECKIN = 404;

  /**
   * signals a migration
   */
  public final static int MIGRATION = 501;

  /**
   * constants for event templates
   */
  public static final ResourceEvent ALL_TEMPLATE                = new ResourceEvent(ALL, null);
  public static final ResourceEvent UNLOCK_TEMPLATE             = new ResourceEvent(UNLOCK, null);
  public static final ResourceEvent LOCK_TEMPLATE               = new ResourceEvent(LOCK, null);
  public static final ResourceEvent PROPERTY_DELETE_TEMPLATE    = new ResourceEvent(PROPERTY_DELETE, null);
  public static final ResourceEvent PROPERTY_SET_TEMPLATE       = new ResourceEvent(PROPERTY_SET, null);
  public static final ResourceEvent PROPERTY_GET_TEMPLATE       = new ResourceEvent(PROPERTY_GET, null);
  public static final ResourceEvent DELETE_TEMPLATE             = new ResourceEvent(DELETE, null);
  public static final ResourceEvent CREATE_CHILD_TEMPLATE       = new ResourceEvent(CREATE_CHILD, null);
  public static final ResourceEvent CREATE_COLLECTION_TEMPLATE  = new ResourceEvent(CREATE_COLLECTION, null);
  public static final ResourceEvent CREATE_LINK_TEMPLATE        = new ResourceEvent(CREATE_LINK, null);
  public static final ResourceEvent UPDATE_LINK_TEMPLATE        = new ResourceEvent(UPDATE_LINK, null);  
  public static final ResourceEvent SET_TEMPLATE                = new ResourceEvent(SET, null);
  public static final ResourceEvent GET_TEMPLATE                = new ResourceEvent(GET, null);
  public static final ResourceEvent GET_CHILDREN_TEMPLATE       = new ResourceEvent(GET_CHILDREN, null);
  public static final ResourceEvent MOVE_TEMPLATE               = new ResourceEvent(MOVE, null);
  public static final ResourceEvent RENAME_TEMPLATE             = new ResourceEvent(RENAME, null);
  public static final ResourceEvent COPY_TEMPLATE               = new ResourceEvent(COPY, null);
  public static final ResourceEvent CHECKOUT_TEMPLATE           = new ResourceEvent(CHECKOUT, null);
  public static final ResourceEvent CHECKIN_TEMPLATE            = new ResourceEvent(CHECKIN, null);
  public static final ResourceEvent UNDO_CHECKOUT_TEMPLATE      = new ResourceEvent(UNDO_CHECKOUT, null);
  public static final ResourceEvent ENABLE_VERSIONING_TEMPLATE  = new ResourceEvent(ENABLE_VERSIONING, null);
  public static final ResourceEvent PRE_SET_TEMPLATE            = new ResourceEvent(PRE_SET, null);
  public static final ResourceEvent PRE_DELETE_TEMPLATE         = new ResourceEvent(PRE_DELETE, null);
  public static final ResourceEvent PRE_RENAME_TEMPLATE         = new ResourceEvent(PRE_RENAME, null);
  public static final ResourceEvent PRE_MOVE_TEMPLATE           = new ResourceEvent(PRE_MOVE, null);
  public static final ResourceEvent PRE_CHECKIN_TEMPLATE        = new ResourceEvent(PRE_CHECKIN, null);
  public static final ResourceEvent MIGRATION_TEMPLATE          = new ResourceEvent(MIGRATION, null);
  
  /**
   * resource bundles.
   */
  private final static ResourceBundles resBundle = ResourceBundles.getBundle("com.sapportals.wcm.repository.manager.Res");

  private final static String DEFAULT_DESCRIPTION = "event #";
  private final static String RESOURCE_PREFIX = "ResourceEvent.";

  /**
   * This events type.
   */
  protected final int type;

  /**
   * This events associated resource.
   */
  protected final IResource resource;

  /**
   * Resource context asscociated to the event
   */
  protected IResourceContext resourceContext;

  /**
   * Client context asscociated to the event
   */
  protected final String clientId;

  /**
   * Unique id of the event
   */
  protected final String uniqueId;

  /**
   * Correlationid of the event
   */
  protected final String correlationId;

  /**
   * This events parameter(s)
   */
  protected final Object param;

  /**
   * Properties which can be stored by receivers and read by subsequent
   * receivers of this event.
   */
  private java.util.HashMap props;

  private static long fastUniqueID = 0L;

  /**
   * This is a simple counter for unique event identifiers. Usage of UUID class
   * was to expensive.
   *
   * @return The next unique ID value.
   */
  private static String nextID() {
    synchronized (ResourceEvent.class) {
      ResourceEvent.fastUniqueID = ResourceEvent.fastUniqueID + 1;
    }
    return String.valueOf(ResourceEvent.fastUniqueID);
  }

  // ------------
  // Constructors -------------------------------------------------------------
  // ------------


  /**
   * Create resource event from type and resource.
   *
   * @param type The type for this event (action).
   * @param resource The resource that caused the event.
   */
  public ResourceEvent(int type,
    IResource resource) {

    this(resource, type, null, null);
  }


  /**
   * Create resource event from type and resource.
   *
   * @param type The type for this event (action).
   * @param resource The resource that caused the event.
   * @param param Additional parameter(s):<br>
   *      LOCK, UNLOCK: ILockInfo of the lock<br>
   *      PROPERTY_DELETE: IPropertyDef of the deleted property<br>
   *      PROPERTY_GET, PROPERTY_SET: IProperty<br>
   *      CREATE_CHILD, CREATE_LINK: IResource<br>
   *      CREATE_COLLECTION: ICollection<br>
   *      COPY, MOVE: IResource The new Resource<br>
   *      RENAME: String The old {@link RID}<br>
   *      DELETE: In case the deleted resource was a collection the parameter
   *      might contain a {@link IRidList} of all deleted RIDa - that is the
   *      identifier of the collection and all its children. The repository
   *      manager is not required to provide this list.<br>
   *      CHECKIN: The new revision {@link RID}<br>
   *      PRE_RENAME: The new name as String<br>
   *      PRE_MOVE: The destination {@link RID}<br>
   *
   */
  public ResourceEvent(int type,
    IResource resource,
    Object param) {

    this(resource, type, null, param);
  }

  /**
   * Create resource event based on resource event type and resource.
   *
   * @param resource resource that caused the event
   * @param type type for this event (action)
   */
  public ResourceEvent(IResource resource, int type) {
    this(resource, type, null, null);
  }

  /**
   * Create resource event based on resource event type and resource.
   *
   * @param resource resource that caused the event
   * @param type type for this event (action)
   * @param correlationId (unique event) id to which this event is related
   */
  public ResourceEvent(IResource resource, int type, String correlationId) {
    this(resource, type, correlationId, null);
  }

  /**
   * Create resource event based on resource event type and resource.
   *
   * @param resource resource that caused the event
   * @param type type for this event (action)
   * @param param additional parameter(s):<br>
   *      LOCK, UNLOCK: ILockInfo of the lock<br>
   *      PROPERTY_DELETE: IPropertyDef of the deleted property<br>
   *      PROPERTY_GET, PROPERTY_SET: IProperty<br>
   *      CREATE_CHILD, CREATE_LINK: IResource<br>
   *      CREATE_COLLECTION: ICollection<br>
   *      COPY, MOVE: IResource The new Resource<br>
   *      RENAME: String The old {@link RID}<br>
   *      DELETE: In case the deleted resource was a collection the parameter
   *      might contain a {@link IRidList} of all deleted RIDa - that is the
   *      identifier of the collection and all its children. The repository
   *      manager is not required to provide this list.<br>
   *      CHECKIN: The new revision {@link RID}<br>
   *      PRE_RENAME: The new name as String<br>
   *      PRE_MOVE: The destination {@link RID}<br>
   *
   */
  public ResourceEvent(IResource resource, int type, Object param) {
    this(resource, type, null, param);
  }

  /**
   * Create resource event from type and resource.
   *
   * @param resource resource that caused the event
   * @param type type for this event (action)
   * @param correlationId (unique event) id to which this event is related
   * @param param additional parameter(s):<br>
   *      LOCK, UNLOCK: ILockInfo of the lock<br>
   *      PROPERTY_DELETE: IPropertyDef of the deleted property<br>
   *      PROPERTY_GET, PROPERTY_SET: IProperty<br>
   *      CREATE_CHILD, CREATE_LINK: IResource<br>
   *      CREATE_COLLECTION: ICollection<br>
   *      COPY, MOVE: IResource The new Resource<br>
   *      RENAME: String The old {@link RID}<br>
   *      DELETE: In case the deleted resource was a collection the parameter
   *      might contain a {@link IRidList} of all deleted RIDa - that is the
   *      identifier of the collection and all its children. The repository
   *      manager is not required to provide this list.<br>
   *      CHECKIN: The new revision {@link RID}<br>
   *      PRE_RENAME: The new name as String<br>
   *      PRE_MOVE: The destination {@link RID}<br>
   *
   */
  public ResourceEvent(IResource resource, int type, String correlationId, Object param) {
    this.resource = resource;
    this.uniqueId = ResourceEvent.nextID();
    this.correlationId = correlationId;
    this.type = type;
    this.param = param;

    IAccessContext accessContext = AccessContextFactory.getInstance().getContextOrNullIfUnavailable();
    if (accessContext != null) {
      this.clientId = accessContext.getClientId();
    }
    else {
      this.clientId = null;
    }
  }


  // --------------
  // IResourceEvent -----------------------------------------------------------
  // --------------


  public IResource getResource() {
    return this.resource;
  }


  /**
   * Returns the resource context used when the event was build.
   *
   * @return the resource context used when the event was build.
   */
  public IResourceContext getResourceContext() {
    if (this.resourceContext == null) {
      if (resource != null) {
        IResourceContext ctxt = resource.getContext();
        if (ctxt != null) {
          this.resourceContext = ((ResourceContext)ctxt).cloneContext();
        }
      }
    }
    return this.resourceContext;
  }

  /**
   * Returns the client identifier used when the event was build.
   *
   * @return the client identifier used when the event was build.
   */
  public String getClientId() {
    return this.clientId;
  }

  /**
   * Get unique id of event.
   *
   * @return unique id of event
   */
  public String getUniqueId() {
    return this.uniqueId;
  }

  /**
   * Get correlation id of event.
   *
   * @return correlation id of event
   */
  public String getCorrelationId() {
    return this.correlationId;
  }


  public int getType() {
    return this.type;
  }


  public boolean isLike(IEvent template) {

    // if other event is <null> --> don't care
    if( template == null ) return true;

    // if other event is not a resource event --> doesn't match
    if( !(template.getClass().equals(ResourceEvent.class)) ) return false;

    // this resource event's type is valid for all resource event types
    if( this.type == ResourceEvent.ALL ) return true;
    // other resource event's type is valid for all resource event types
    if( template.getType() == ResourceEvent.ALL ) return true;

    // return true if this resource event's type and the other resource event's type match
    return( template.getType() == this.type );
    
  }


  public String getDescription() {
    return getDescription(Locale.getDefault());
  }


  public String getDescription(Locale locale) {

    String description = null;
    try {
      description = ResourceEvent.resBundle.getString(RESOURCE_PREFIX + this.type, locale);
    }
    catch (Exception e) {
      description = null;
    }

    if (description == null) {
      description = DEFAULT_DESCRIPTION + this.type;
    }

    return description;
  }

  /**
   * Returns the property with the specified name. See {@link #setReceiverProperty(String, Object)}.
   *
   * @param name TBD: Description of the incoming method parameter
   * @return the property with the specified name.
   */
  public Object getReceiverProperty(String name) {
    if (this.props == null) {
      return null;
    }
    else {
      return this.props.get(name);
    }
  }

  /**
   * Stores a property (name/value pair) at this event. This method can be used
   * by event receivers to attach information to the event in addition to the
   * parameter(s) provided by the event sender. Subsequent receivers can read
   * these properties with {@link #getReceiverProperty(String)}. Note: "subsequent
   * receivers" of a synchronous reveicer R are:<br>
   * - sychronous receivers with lower priority than R<br>
   * - all asynchronous receivers<br>
   *
   *
   * @param name receiverProperty to be set
   * @param value receiverProperty to be set
   */
  public void setReceiverProperty(String name, Object value) {
    if (this.props == null) {
      this.props = new java.util.HashMap();
    }
    this.props.put(name, value);
  }

  public Object getParameter() {
    return this.param;
  }


  public String toString() {
    return this.eventName(this.type);
  }

  private String eventName(int type) {
    switch (type) {
      case ResourceEvent.ALL:
        return "ALL";
      case ResourceEvent.UNLOCK:
        return "UNLOCK";
      case ResourceEvent.LOCK:
        return "";
      case ResourceEvent.PROPERTY_DELETE:
        return "PROPERTY_DELETE";
      case ResourceEvent.PROPERTY_SET:
        return "PROPERTY_SET";
      case ResourceEvent.PROPERTY_GET:
        return "PROPERTY_GET";
      case ResourceEvent.DELETE:
        return "DELETE";
      case ResourceEvent.CREATE_CHILD:
        return "CREATE_CHILD";
      case ResourceEvent.CREATE_COLLECTION:
        return "CREATE_COLLECTION";
      case ResourceEvent.CREATE_LINK:
        return "CREATE_LINK";
      case ResourceEvent.UPDATE_LINK:
        return "UPDATE_LINK";        
      case ResourceEvent.SET:
        return "SET";
      case ResourceEvent.GET:
        return "GET";
      case ResourceEvent.GET_CHILDREN:
        return "GET_CHILDREN";
      case ResourceEvent.MOVE:
        return "MOVE";
      case ResourceEvent.RENAME:
        return "RENAME";
      case ResourceEvent.COPY:
        return "COPY";
      case ResourceEvent.CHECKOUT:
        return "CHECKOUT";
      case ResourceEvent.CHECKIN:
        return "CHECKIN";
      case ResourceEvent.UNDO_CHECKOUT:
        return "UNDO_CHECKOUT";
      case ResourceEvent.ENABLE_VERSIONING:
        return "ENABLE_VERSIONING";
      case ResourceEvent.PRE_SET:
        return "PRE_SET";
      case ResourceEvent.PRE_DELETE:
        return "PRE_DELETE";
      case ResourceEvent.PRE_RENAME:
        return "PRE_RENAME";
      case ResourceEvent.PRE_MOVE:
        return "PRE_MOVE";
      case ResourceEvent.PRE_CHECKIN:
        return "PRE_CHECKIN";
      default:
        return String.valueOf(type);
    }
  }
}
