/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import com.sap.tc.logging.*;
import com.sapportals.wcm.*;
import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.events.*;
import com.sapportals.wcm.util.logging.*;

import java.util.*;

/**
 * Class implementing overall resource event broker, which can be used to send
 * and register for events for all repositories.
 *
 * @created 4. April 2003
 */
public class OverallResourceEventBroker
   implements IResourceEventBroker, IComponentListener {
  // Logging location to log messages to
  private static Location logger =
    Location.getLocation(
    com.sapportals.wcm.repository.manager.OverallResourceEventBroker.class);

  // Singleton instance of this class
  private static OverallResourceEventBroker inst = null;

  // List of all resource event brokers of all repositories
  private List resourceEventBrokers = new ArrayList();

  private boolean sendGet = false;
  private boolean sendGetChildren = false;
  private boolean sendGetProperty = false;
  private boolean sendSetProperty = false;
  private boolean sendDeleteProperty = false;

  /**
   * Construct instance of class OverallResourceEventBroker.
   */
  private OverallResourceEventBroker() {
    // Get all reporitory managers
    Collection repositoryManagers = null;
    try {
      repositoryManagers =
        ResourceFactory.getInstance()
        .getManagerFactory()
        .getAllRepositoryManagers();
    }
    catch (Exception exception) {
      // Handle caught exception
      logger.errorT(
        "OverallResourceEventBroker(62)",
        "Caught exception while retrieving all repository managers: "
         + LoggingFormatter.extractCallstack(exception));
    }

    // Check for returned managers
    if (repositoryManagers != null) {
      for (Iterator iter = repositoryManagers.iterator();
        iter.hasNext();
        ) {
        IRepositoryManager repositoryManager =
          (IRepositoryManager)iter.next();
        if ((repositoryManager != null)
           && (repositoryManager.getEventBroker() != null)) {
          // Store event broker of repository
          resourceEventBrokers.add(
            repositoryManager.getEventBroker());
        }
      }
    }
  }

  /**
   * Get singleton instance of this class.
   *
   * @return singleton instance of this class
   */
  public static synchronized OverallResourceEventBroker getInstance() {
    // Check if singleton wasn't instantiated yet
    if (inst == null) {
      // Instantiate singleton instance now
      try {
        inst = new OverallResourceEventBroker();
      }
      catch (Throwable throwable) {
        // Nothing to do except for logging/tracing since inst wasn't assigned yet
        // Rethrow the caught throwable or a new one so that no null is returned
        logger.errorT(
          "getInstance(100)",
          "Caught exception while creating singleton instance of overall resource event broker: "
           + LoggingFormatter.extractCallstack(throwable));
      }
    }

    // Return singleton instance
    return inst;
  }

  /**
   * Handle component add, reconfiguration or removal. Extract the resource
   * event broker in case the event relates to an repository manager. Note:
   * Component reconfiguration or removal events are not yet available.
   *
   * @param event CRT component event
   */
  public synchronized void notify(ComponentEvent event) {
    // Handle component add
    // Note: Component reconfiguration or removal events are not yet available
    if (event.getType().equals(ComponentEvent.Type.COMPONENT_ADDED)) {
      // Check for repository manager
      if (IRepositoryManager.class
        .isAssignableFrom(event.getComponentClass())) {
        try {
          // Look up repository manager
          IRepositoryManager repositoryManager =
            (IRepositoryManager)CrtSystem.getInstance()
            .lookupComponentByUri(
            event.getComponentUri());

          // Store event broker of repository
          resourceEventBrokers.add(
            repositoryManager.getEventBroker());
        }
        catch (Throwable throwable) {
          // Caught exception while looking up repository manager or resource event broker for repository
          logger.errorT(
            "notify(138)",
            "Caught exception while looking up repository manager or resource event broker for repository "
             + event.getComponentUri().toString()
             + ": "
             + LoggingFormatter.extractCallstack(throwable));
        }
      }
    }
  }

  /**
   * Sends the specified event to all receivers that have registered for this
   * kind of event.
   *
   * @param event event to be sent
   * @param sender event sender instance sending the event
   * @exception WcmException when the event sender is not registered
   */
  public boolean send(IFrameworkTransaction ft, IEvent event, IEventSender sender) {
    boolean ok = true;
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        if (ft.willBeRollbacked() || !resourceEventBroker.send(ft, event, sender)) return false;
      }
      catch (Exception exception) {
        // Handle caught exception
        ok = false;
        logger.errorT(
          "send(168)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
    return ok;
  }
  
  /**
   * Sends the specified event to all receivers that have registered for this
   * kind of event.
   *
   * @param event event to be sent
   * @param sender event sender instance sending the event
   * @exception WcmException when the event sender is not registered
   */
  public synchronized void send(IEvent event, IEventSender sender)
    throws WcmException {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.send(event, sender);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "send(168)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Registers an event sender instance.
   *
   * @param sender event sender to be registered
   * @exception WcmException when the registration fails
   */
  public synchronized void register(IEventSender sender)
    throws WcmException {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.register(sender);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "register(193)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Registers an event receiver. The receiver will only receive events which
   * match the template event. This depends on the matching rules implemented in
   * the events {@link IEvent#isLike(IEvent)} method. Usually the event's instance and
   * type are relevant. It is allowed to register the same event receiver
   * instance several times with different templates.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @exception WcmException when the registration fails
   */
  public synchronized void register(IEventReceiver receiver, IEvent template)
    throws WcmException {
    this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
    this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
    this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
    this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
    this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.register(receiver, template);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "register(224)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has an additional {@link IEventMapper}
   * argument, i.e. that the mapper will be called prior to giving the event to
   * the receiver.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @exception WcmException when the registration fails
   * @see IEventMapper
   */
  public synchronized void register(
    IEventReceiver receiver,
    IEvent template,
    IEventMapper mapper)
    throws WcmException {
    this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
    this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
    this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
    this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
    this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.register(receiver, template, mapper);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "register(260)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has two additional arguments for priority
   * and asynchronous events. <p>
   *
   * The priority controls the order of multible receivers which have registered
   * for the same event(s). Receivers with higher priority (smaller values) will
   * receive an event before receivers with lower priority (greater values). <p>
   *
   * An event receiver can choose to receive events asynchronously. This means
   * that a dedicated event queue and sender thread is created for each receiver
   * and template. The event sender will not be blocked, that means the send()
   * method puts the event into the queue and returns immediatelly.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception WcmException when the registration fails
   */
  public synchronized void register(
    IEventReceiver receiver,
    IEvent template,
    int priority,
    boolean async)
    throws WcmException {
    this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
    this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
    this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
    this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
    this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.register(receiver, template, priority, async);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "register(305)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent,
   * IEventMapper)} and {@link #register(IEventReceiver, IEvent, int, boolean)}
   * method.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception WcmException when the registration fails
   * @see IEventMapper
   */
  public synchronized void register(
    IEventReceiver receiver,
    IEvent template,
    IEventMapper mapper,
    int priority,
    boolean async)
    throws WcmException {
    this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
    this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
    this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
    this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
    this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.register(receiver, template, mapper, priority, async);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "register(345)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Unregisteres an event sender instance.
   *
   * @param sender event sender to be unregistered
   * @exception WcmException when the unregistration fails
   */
  public synchronized void unregister(IEventSender sender)
    throws WcmException {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.unregister(sender);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "unregister(370)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Unregisteres an event receiver instance.
   *
   * @param receiver event receiver to be unregistered
   * @param template event template on which the event receiver will no longer
   *      receive events
   * @exception WcmException when the unregistration fails
   */
  public synchronized void unregister(IEventReceiver receiver, IEvent template)
    throws WcmException {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.unregister(receiver, template);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "unregister(397)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Suspend the event receiver. The broker will stop delivering any events
   * until {@link #resume(IEventReceiver)} is called.
   *
   * @param receiver event receiver
   */
  public synchronized void suspend(IEventReceiver receiver) {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.suspend(receiver);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "suspend(421)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Resumes a previously suspended event receiver. The broker will continue
   * delivering events to the receiver.
   *
   * @param receiver event receiver
   */
  public synchronized void resume(IEventReceiver receiver) {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.resume(receiver);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "resume(445)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * TBD: Unknown function.
   *
   * @param receiver TBD: Unknown parameter
   * @param collect TBD: Unknown parameter
   */
  public synchronized void hold(IEventReceiver receiver, boolean collect) {
    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resourceEventBroker.hold(receiver, collect);
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "hold(469)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }
  }

  /**
   * Get the receiver mode of the receiver, i.e. off, sending and so on.
   *
   * @param receiver event receiver
   * @return receiver mode of the receiver
   * @see SenderMode
   */
  public synchronized SenderMode getMode(IEventReceiver receiver) {
    // Throw unsupported operation exception, because
    // Operation can't be supported for all resource event brokers identically
    throw new UnsupportedOperationException("Operation can't be supported for all resource event brokers identically!");
  }

  /**
   * Get all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public synchronized IEventList getHoldEvents(IEventReceiver receiver) {
    // Create empty result list
    IEventList resultList = new EventList();

    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resultList.addAll(resourceEventBroker.getHoldEvents(receiver));
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "getHoldEvents(509)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }

    // Return result list
    return resultList;
  }

  /**
   * Remove all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public synchronized IEventList clearHoldEvents(IEventReceiver receiver) {
    // Create empty result list
    IEventList resultList = new EventList();

    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resultList.addAll(resourceEventBroker.clearHoldEvents(receiver));
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "clearHoldEvents(539)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }

    // Return result list
    return resultList;
  }

  /**
   * Returns a list of event templates for all possible events this broker can
   * send (determined by querying all registered senders). The event instances
   * contained in the list are "templates", that means the event parameter is
   * not relevant.
   *
   * @return list of event templates for all possible events this broker can
   *      send
   */
  public synchronized IEventList getEvents() {
    // Create empty result list
    IEventList resultList = new EventList();

    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resultList.addAll(resourceEventBroker.getEvents());
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "getEvents(572)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }

    // Return result list
    return resultList;
  }

  /**
   * Returns a list of event templates for all possible events this broker can
   * send for the given resource (determined by querying all registered
   * senders). The resource event instances contained in the list are
   * "templates", that means the resource event parameter is not relevant.
   *
   * @param resource resource for which event templates should be checked
   * @return list of resource event templates for all possible resource events
   *      this broker can send
   */
  public synchronized IEventList getEvents(IResource resource) {
    // Create empty result list
    IEventList resultList = new EventList();

    // Iterate over all resource event brokers
    for (Iterator iter = resourceEventBrokers.iterator(); iter.hasNext(); ) {
      // Call operation on resource event broker
      IResourceEventBroker resourceEventBroker = (IResourceEventBroker)iter.next();
      try {
        resultList.addAll(resourceEventBroker.getEvents(resource));
      }
      catch (Exception exception) {
        // Handle caught exception
        logger.errorT(
          "getEvents(606)",
          "Failed to call operation on resource event broker " + resourceEventBroker + ": "
           + LoggingFormatter.extractCallstack(exception));
      }
    }

    // Return result list
    return resultList;
  }


  /**
   * Check if the given type of a <code>ResourceEvent</code> has to be
   * sent at all.
   * @param type the ResourceEvent-type to check for.
   * @return <code>true</code> if at least one receiver is registered for
   *        this <code>ResourceEvent</code> type.
   */
  public boolean mustSendResourceEventType(int type) {
    switch( type ) {
      case ResourceEvent.PROPERTY_GET: return this.sendGetProperty;
      case ResourceEvent.GET_CHILDREN: return this.sendGetChildren;
      case ResourceEvent.GET: return this.sendGet;
      case ResourceEvent.PROPERTY_SET: return this.sendSetProperty;
      case ResourceEvent.PROPERTY_DELETE: return this.sendDeleteProperty;
      default: return true;
    }
  }

}
