/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.IRidSet;

import java.util.*;

/**
 * An ISecurityManager is a repository submanager which controls the access to
 * resources - an ISecurityManager might be an IAclSecurityManager (using Access
 * Control Lists)
 */
public interface ISecurityManager {
  /**
   * Returns a a list of permissions which are supported by the security manager
   * for a specific resource (the result depends on wether the resource is a
   * collection or not)
   *
   * @param resource the resource
   * @return the supported permissions
   * @exception ResourceException the resource RID could not be determined
   */
  public IPermissionList getSupportedPermissions(IResource resource)
    throws ResourceException;

  /**
   * Returns the IPermission object for a supported permission with a given name
   *
   * @param name the permission name
   * @return the IPermission object
   */
  public IPermission getPermission(String name);

  /**
   * Checks, whether a user has a specific permission on a specific resource
   *
   * @param resource the resource
   * @param user the user ID
   * @param permission the permission
   * @return true if the permission is granted
   * @exception ResourceException the resource RID could not be determined
   */
  public boolean isAllowed(IResource resource, IUser user, IPermission permission)
    throws ResourceException;

  /**
   * Checks, whether a user has some permissions on a specific resource
   *
   * @param resource the resource
   * @param user the user ID
   * @param permissions A list of permission
   * @return true if the permission is granted
   * @exception ResourceException the resource RID could not be determined
   */
  public boolean isAllowed(IResource resource, IUser user, IPermissionList permissions)
    throws ResourceException;


  /**
   * Checks, whether a user has some permissions on all specified resources. A
   * RID from the input list must be contained int the returned list if the user
   * has all the permissions in the permission list.
   *
   * @param ridList A list of resource identifiers to check
   * @param user the user ID
   * @param permissions A list of permission
   * @param context The resource context
   * @return A set of RIDs
   * @exception ResourceException the resource RID could not be determined
   */
  public IRidSet isAllowed(IRidList ridList, IUser user, IPermissionList permissions, IResourceContext context)
    throws ResourceException;

  /**
   * Returns the type attribute of this security manager.
   *
   * @return type
   */
  public String getType();
}
