/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.filter;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.*;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.crt.configuration.*;
import com.sapportals.wcm.repository.enum.FilterType;
import com.sapportals.wcm.repository.runtime.ICmRuntimeConst;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.regex.PathPatternMatcher;
import java.lang.reflect.*;
import java.util.*;

/**
 * Abstract base class for all filter managers. <p>
 *
 * Copyright (c) SAP AG 2001 - 2002
 *
 * @author Markus Breitenfelder
 * @version $Id:$
 */
public abstract class AbstractFilterManager
   implements IComponent, IConfigurable, IStartable, IComponentInfo, ILifecycleInfo, IThreadSafe {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.filter.AbstractFilterManager.class);

  private static final Map TYPE_NAMES;
  static {
    TYPE_NAMES = new HashMap();
    TYPE_NAMES.put(FilterType.CONTENT, "ContentFilter");
    TYPE_NAMES.put(FilterType.NAMESPACE, "NamespaceFilter");
    TYPE_NAMES.put(FilterType.PROPERTY, "PropertyFilter");
    TYPE_NAMES.put(FilterType.URI, "UriFilter");
  }

  private String id;
  protected Properties properties;
  protected IConfiguration config;

  protected final ComponentStateHandler stateHandler = new ComponentStateHandler(this);

  /**
   * The default constructor
   */
  public AbstractFilterManager() { }


  /**
   * @param properties TBD: Description of the incoming method parameter
   * @deprecated As of EP 5.0 SP5, replaced by default (public, no arguments)
   *      constructor.
   */
  protected AbstractFilterManager(Properties properties) {
    if (properties != null) {
      this.properties = (Properties)(properties.clone());
    }
    this.id = (String)this.properties.get(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
  }

  public final String getID() {
    return this.id;
  }

  // ---------------------------------------------------------------------------
  // interface IConfigurable
  // ---------------------------------------------------------------------------

  public final void configure(IConfiguration config)
    throws ConfigurationException {
    this.stateHandler.preConfigure();

    this.config = config;
    this.id = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID, null);

    this.stateHandler.postConfigure();
  }

  // ---------------------------------------------------------------------------
  // interface IStartable
  // ---------------------------------------------------------------------------

  public final void start()
    throws StartupException {
    this.stateHandler.preStart();

    try {
      this.startUpImpl();
    }
    catch (ConfigurationException ex) {
      throw this.stateHandler.postStart(ex);
    }
    catch (StartupException ex) {
      throw this.stateHandler.postStart(ex);
    }
    catch (Exception ex) {
      throw this.stateHandler.postStart(new StartupException("Runtime exception during start up: " + ex.getMessage(), ex));
    }

    this.stateHandler.postStart();
  }

  public final void stop() {
    this.stateHandler.preStop();
    this.shutDownImpl();
    this.stateHandler.postStop();
  }

  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return this.id;
  }

  public String getDescription(Locale locale) {
    return null;
  }

  public Properties getProperties() {
    Properties props = AbstractFilterManager.convertConfigToProperties(this.config, "config-");
    if (this instanceof IFilterManager) {
      String typeName = (String)TYPE_NAMES.get(((IFilterManager)this).getType());
      props.setProperty("type", typeName);
    }
    return props;
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public final ComponentState getState() {
    return this.stateHandler.getState();
  }

  public ConfigurationException getLastConfigurationException() {
    return this.stateHandler.getLastConfigurationException();
  }

  public StartupException getStartupException() {
    return this.stateHandler.getStartupException();
  }

  public final Date getCreationDate() {
    return this.stateHandler.getCreationDate();
  }

  public Date getLastReconfigurationDate() {
    return this.stateHandler.getLastReconfigurationDate();
  }

  public Date getNextAutoRestartDate() {
    return this.stateHandler.getNextAutoRestartDate();
  }

  // ---------------------------------------------------------------------------

  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc).
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #shutDownImpl()}.
   */
  protected void startUp()
    throws WcmException {
    log.infoT("startUp(171)", "The filter manager implementation does not overwrite the startUpImpl() method: " + this.id);
  }


  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown.
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #startUpImpl()}.
   */
  protected void shutDown()
    throws WcmException {
    log.infoT("shutDown(184)", "The filter manager implementation does not overwrite the shutDownImpl() method: " + this.id);
  }


  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc). Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called on systen start up or
   * when a new instance of this component is added to the persistence
   * configuration.
   *
   * @exception ConfigurationException Exception raised in failure situation
   * @exception StartupException Exception raised in failure situation
   */
  protected void startUpImpl()
    throws ConfigurationException, StartupException {
    try {
      this.startUp();
    }
    catch (WcmException x) {
      throw new StartupException(x.getMessage(), x);
    }
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during startup. Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called when the component is
   * removed from the persistent configuration or when the system shuts down.
   */
  protected void shutDownImpl() {
    try {
      this.shutDown();
    }
    catch (WcmException x) {
      log.errorT("shutDownImpl(222)", x.getMessage() + ": " + LoggingFormatter.extractCallstack(x));
    }
  }

  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config to be converted
   * @return all attributes of the config object as properties
   */
  protected final static Properties convertConfigToProperties(IConfiguration config) {
    return convertConfigToProperties(config, null);
  }
  
  /**
   * Helper method for conversion of configuration data into properties. Prefixes
   * all attribute name with given prefix.
   *
   * @param config to be converted
   * @param prefix to prepend to config's attribute names
   * @return all attributes of the config object as properties
   */
  protected final static Properties convertConfigToProperties(IConfiguration config, String prefix) {
    Properties props = new Properties();
    String[] attributeNames = config.getAttributeNames();
    for (int i = 0; i < attributeNames.length; i++) {
      String name = attributeNames[i];
      String value = config.getAttribute(name, null);
      if (value != null) {
        if (prefix != null) {
          name = prefix + name;
        }
        props.setProperty(name, value);
      }
    }
    return props;
  }

  /**
   * Changes the system properties of a filter manager. Must be used by filter
   * managers in the implementation of the reconfigure() method.
   *
   * @param newConfig The new configuration.
   * @exception ConfigurationException If the new system properties are invalid
   */
  protected abstract void changeSystemProperties(IConfiguration newConfig)
    throws ConfigurationException;

  protected final void throwConfigurationException(String text, String location, String propName)
    throws ConfigurationException {
    throw new ConfigurationException(text + ", location=" + location + ", property=" + propName);
  }

  /**
   * Helper method for reconfigure() implementations. Throws a configuration
   * exception if any of the specified attributes or any of the not
   * reconfigurable framework attributes (e.g. prefix) has changed.
   *
   * @param notReconfigurable
   * @param newConfig
   * @exception ConfigurationException
   * @notReconfigurable An array of names of attributes that cannot be canged
   *      wihout restarting the component/server.
   * @newConfig The new configuration data received as argument of the
   *      reconfigure() method.
   */
  protected final void checkNotReconfigurableAttributes(String[] notReconfigurable, IConfiguration newConfig)
    throws ConfigurationException {
    // Note all system properties are mutable - check only component specific properties
    this.checkNotReconfigurableAttributeNames(notReconfigurable, newConfig);
  }

  /**
   * Update the filter properties of runtime.
   *
   * @param id TBD: Description of the incoming method parameter
   * @param prio TBD: Description of the incoming method parameter
   * @param prefix TBD: Description of the incoming method parameter
   * @param mimePattern TBD: Description of the incoming method parameter
   * @param extPattern TBD: Description of the incoming method parameter
   * @param pathPattern TBD: Description of the incoming method parameter
   * @param resourceType TBD: Description of the incoming method parameter
   * @param managerIDs TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @exception Exception Exception raised in failure situation
   */
  protected final void callUpdateFilter(
    String id, Integer prio, String prefix,
    PathPatternMatcher mimePattern, PathPatternMatcher extPattern,
    PathPatternMatcher pathPattern, String resourceType, ArrayList managerIDs, int type
    )
    throws Exception {
    //
    // Calls the runtime via reflection because the runtime classes are in CORE
    // TODO: Move abstract repository manager stuff and runtime in API !!
    //
    Class filterPropertiesClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.repository.runtime.CmFilterProperties");
    Class systemClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.repository.runtime.CmSystem");
    Class registryClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.repository.runtime.CmRegistry");

    Constructor filterPropertiesConst = filterPropertiesClass.getConstructor(
      new Class[]{Integer.class, String.class, PathPatternMatcher.class, PathPatternMatcher.class, PathPatternMatcher.class, ArrayList.class, Integer.TYPE});
    Object filterProperties = filterPropertiesConst.newInstance(
      new Object[]{prio, prefix, mimePattern, extPattern, pathPattern, resourceType, managerIDs, new Integer(type)});

    Method getInstance = systemClass.getMethod("getInstance", new Class[0]);
    Object cmSystem = getInstance.invoke(null, new Object[0]);
    Method getRegistry = systemClass.getMethod("getRegistry", new Class[0]);
    Object registry = getRegistry.invoke(cmSystem, new Object[0]);

    Method updateFilter = registryClass.getMethod("updateFilter", new Class[]{String.class, filterPropertiesClass});
    updateFilter.invoke(registry, new Object[]{id, filterProperties});

    AbstractFilterManager.log.infoT("callUpdateFilter(320)", "Filter system properties changed: " + id + ", " + filterProperties.toString());
  }

  //
  // TODO: Remove and use runtime constants if moved to CORE !
  //

  /**
   * TBD: Description of the class.
   */
  protected static interface TYPE {
    public final static int CONTENT = 0;
    public final static int PROPERTY = 1;
    public final static int NAMESPACE = 2;
    public final static int URI = 3;
  }

  /**
   * TBD: Description of the class.
   */
  public interface CFG {
    String PROPERTY_PREFIX = "prefix";
    String ACTIVE_KEY = "active";
    String PROPERTY_CLASSNAME = "class";
    String MANAGER_CLASSNAME = "RepositoryManager";
    String SERVICE_CLASSNAME = "Service";
    String REPSERVICE_CLASSNAME = "RepositoryService";
    String MANAGER_SERVICES_KEY = "services";
    String FILTER_CLASSNAME = "RepositoryFilter";
    String FILTERMGR_REPMGRS = "managers";
    String FILTERMGR_URIPREFIX = "prefix";
    String FILTERMGR_EXTENSIONS = "extensions";
    String FILTERMGR_MIMETYPES = "mimetypes";
    String FILTERMGR_RESOURCETYPE = "resourcetype";
    String FILTERMGR_PATH = "path";
    String FILTERMGR_PRIORITY = "priority";
    String SERVICE_CCLIST = "cclist";
  }


  /**
   * TBD: Description of the class.
   */
  public interface FILTER_PRIO {
    Integer MIN = new Integer(0);
    Integer MAX = new Integer(99);
  }

  private void checkNotReconfigurableAttributeNames(String[] names, IConfiguration newConfig)
    throws ConfigurationException {
    for (int i = 0; i < names.length; i++) {
      String name = names[i];
      String newValue = newConfig.getAttribute(name, null);
      String oldValue = this.config.getAttribute(name, null);
      if (newValue != null && oldValue != null) {
        if (!newValue.equals(oldValue)) {
          throw new ConfigurationException("Property cannot be changed at runtime: " + name);
        }
      }
    }
  }

  /**
   * Helper method: Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final ArrayList parseValueList(String values, String separator) {
    ArrayList list = new ArrayList();
    if (values == null) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }
}
