package com.sapportals.wcm.repository;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * Copyright SAP AG 2004
 * @author andreas.heix@sap.com
 * @since EP5, SP6, Patch1, Hotfix1; EP6, SP2_CP
 * Selector allows to specify hierarchical queries to define which set of
 * resources should be retrieved from the repository framework. A selector
 * might hold a list of other selector instances as well as 
 * {@link PropertySelectorItem} instances. Its type constants define
 * how selection instances are combined.
 */
public final class Selector {

  
  final static public int SELECTOR_TYPE_AND  = 1;
  final static public int SELECTOR_TYPE_OR   = 2;

  public static final Type SELECTOR_AND = new Type(1, "Selector <AND>");
  public static final Type SELECTOR_OR  = new Type(2, "Selector <OR>");

  private Type selectorType = SELECTOR_AND;

  //private int               selectorType = SELECTOR_TYPE_AND;
  private List              list          = new ArrayList();
  private PropertyNameList  nameList      = new PropertyNameList();
 
  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(Selector.class.getName());  

  /**
   * creates a new class instance with the specified selectortype.
   * @param type. If the type is <code>null</code>, the default
   * selectortype <code>SELECTOR_AND</code> will be used
   */
  public Selector(Type type) {
    if (type != null) {
      this.selectorType = type;
    }
  }
  
  /**
   * standard constructor - will create a new class instance with selectortype
   * <code>SELECTOR_AND</code>.
   */
  public Selector() {
  }

  /**
   * creates a new <code>PropertySelectorItem</code> using the specified parameters
   * @param include the include mode for the <code>PropertySelectorItem</code>
   * @param property the property for the <code>PropertySelectorItem</code>
   *  - must not be <code>null</code>
   */
  public void addSelector(boolean include, IProperty property) {
    if (property == null) {
      throw new NullPointerException("The parameter property must not be <null>");
    }
    PropertySelectorItem ps = new PropertySelectorItem(include, property);
    this.addSelector(ps);
  }

  /**
   * @param selector is added to the list of selectors - must not be <code>null</code>
   */
  public void addSelector(PropertySelectorItem selector) {
    if (selector == null) {
      throw new NullPointerException("The parameter selector must not be <null>");
    }
    this.list.add(selector);
    this.nameList.add(selector.getPropertyName());
  }

  /**
   * @param selector is added to the list of selectors - must not be <code>null</code>
   */
  public void addSelector(Selector selector) {
    if (selector == null) {
      throw new NullPointerException("The parameter selector must not be <null>");
    }
    this.list.add(selector);
    for (int i = 0; i < selector.getPropertyNameList().size(); i++) {
      this.nameList.add(selector.getPropertyNameList().get(i));
    }
  }

  public Type getSelectorType() {
    return this.selectorType;
  }

  /**
   * @return flag, stating, if any <code>Selector</code> or <code>PropertySelectorItem</code>
   * has been added to this class. If not, the method will return <code>true</code>.
   */
  public boolean isEmpty() {
    return this.list.size() == 0;
  }

  /**
   * @return a list containing all the names of the properties that are used within
   * the <code>PropertySelectorItem</code> instances that have been added to this instance.
   * Also the Property names of <code>PropertySelectorItem</code> instances that are included 
   * in the <code>Selector</code> instances added to this instance are added. The current
   * implmentation does not check for duplicates within the list. <br>
   * Might return an empty list, but never <code>null</code>
   */
  public PropertyNameList getPropertyNameList() {
    return nameList;
  }

  public String toString() {
    return this.getSelectorType().toString() + this.list.toString();   
  }

  private boolean checkAnd(IPropertyMap properties) throws ResourceException {
    int sel_count = list.size() - 1;
  
    boolean result = true;
  
    while (result && sel_count >= 0) {
      Object selectorItem = list.get(sel_count--);

      if (selectorItem instanceof Selector) {
        Selector  selItem = (Selector)selectorItem;
        result = ((Selector)selectorItem).check(properties);
      } else {
        PropertySelectorItem  selItem = (PropertySelectorItem)selectorItem;
        IProperty             resProp = properties.get(selItem.getPropertyName());
        result = selItem.pass(resProp);
      }
    }

    return result;
  }

  private boolean checkOr(IPropertyMap properties) throws ResourceException {
    int sel_count = list.size() - 1;

    boolean result = false;

    while ((!result) && (sel_count >= 0)) {
      Object selectorItem = list.get(sel_count--);

      if (selectorItem instanceof Selector) {
        result = ((Selector)selectorItem).check(properties);
      } else {
        PropertySelectorItem  selItem = (PropertySelectorItem)selectorItem;
        IProperty             resProp = properties.get(selItem.getPropertyName());
        result                        = selItem.pass(resProp);
      }
    }

    return result;
  }

  /**
   * checks, if the provided map of properties will pass all  <code>PropertySelectorItem</code>
   * instances defined within this class. Depending on the combination of these instances and
   * other <code>Selector</code> instances within this class, not all tests will be executed
   * to calculate the result of this method. If, for example, already the first test fails and
   * the <code>Selector.Type</code> of this class is set to <code>SELECTOR_AND</code>, the 
   * method will return <code>false</code> without evaluating further properties
   * @param properties the properties to check, must not be <code>null</code>, but might be empty
   * @return flag, stating if the properties passed this check or not
   * @throws ResourceException
   */
  public boolean check(IPropertyMap properties) throws ResourceException {
    if (properties == null) {
      throw new NullPointerException("The parameter properties must not be <null>");
    }

    switch( this.selectorType.type) {
      case SELECTOR_TYPE_AND: return this.checkAnd(properties);
      case SELECTOR_TYPE_OR:  return this.checkOr(properties);
      default:                return false;
    }
  }
  
  private static class Type {
    
    int    type = 0;
    String desc = null;
    
    private Type(int type, String desc) {
      this.type = type;
      this.desc = desc;
    }
    
    public String toString() {
      return this.desc;
    }
  }
}

