/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/repository/ResourcePropertyComparator.java#5 $
 */
package com.sapportals.wcm.repository;

import com.sapportals.portal.security.usermanagement.IUserFactory;

import com.sapportals.wcm.repository.enum.PropertyType;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.text.CollationKey;
import java.text.Collator;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;


/**
 * ResourcePropertyComparator <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author jan.heiler@sap.com
 */
public class ResourcePropertyComparator implements Comparator {
  private static IPropertyName displayNameProperty = PropertyName.createDisplayname();
  private static String LOCK_PROPERTY_NAME = "lock";
  private static String COLLABORATION_STATUS_PROPERTY_NAME = "collaborationstatus";
  private final static String CHILDCOUNT_BOTH_PROPERTY_NAME = "childcount_both";
  private final static String CHILDCOUNT_SUM_PROPERTY_NAME = "childcount_sum";
  private final static String CHILDCOUNT_FOLDERS_PROPERTY_NAME = "childcount_folders";
  private final static String CHILDCOUNT_FILES_PROPERTY_NAME = "childcount_files";
  private final static String CHILDCOUNT_PROPERTY_NAME = "childcount";
  public final static String CM_RND_COLLABORATION_STATUS = "rnd:collaborationstatus";
  public final static String CM_ID = "id";
  private final static int COMPARE_CASE_DISPLAY_NAME = 1;
  private final static int COMPARE_CASE_LOCKED = 2;
  private final static int COMPARE_CASE_COLL_STATUS = 3;
  private final static int COMPARE_CASE_CHILD_COUNT = 4;
  private final static int COMPARE_CASE_USER = 5;
  private final static int COMPARE_CASE_OTHER = 6;
  private final static int COMPARE_CASE_ID = 7;
  private final static Long NullProperty = new Long(0);
  private IPropertyName property = null;
  private boolean ascending = true;
  private boolean displayNameAsSecondProperty = false;
  private Collator collator = null;
  private int compareCase = 0;
  private Locale locale = null;
  private Map resourceInfoCache = null;

  /**
   * Constructor for the ResourcePropertyComparator object
   *
   * @param property Description of the param
   * @param ascending Description of the param
   */
  public ResourcePropertyComparator(IPropertyName property, boolean ascending) {
    this(property, ascending, false);
  }

  public ResourcePropertyComparator(IPropertyName property, boolean ascending, boolean useDisplayNameAsSecondProperty) {
    this.property = property;
    this.ascending = ascending;
    this.displayNameAsSecondProperty = useDisplayNameAsSecondProperty;
    this.compareCase = this.setInitialCompareCase(property);
    this.resourceInfoCache = new HashMap();
  }

  /* even faster, but requires changes on caller side
     public ResourcePropertyComparator(IPropertyName property, boolean ascending, boolean useDisplayNameAsSecondProperty,
             IResourceList resourceList) {
       this.property = property;
       this.ascending = ascending;
       this.displayNameAsSecondProperty = useDisplayNameAsSecondProperty;
       this.compareCase = this.setInitialCompareCase(property);
       if(resourceList != null){
         this.resourceArray = this.createResourceArray(resourceList);
         this.resourceInfoArray = new ResourceInfo[this.resourceArray.length];
       }else{
         this.resourceInfoCache = new HashMap();
       }
     }
     private IResource[] createResourceArray(IResourceList resourceList){
       int numberResources = resourceList.size();
       IResource[] array = new IResource[numberResources];
       for (int i = 0; i < numberResources; i++) {
         array[i] = resourceList.get(i);
       }
       return array;
     }
     private ResourceInfo getResourceInfo(Integer pos){
       int intPos = pos.intValue();
       ResourceInfo resInfo = this.resourceInfoArray[intPos];
       if(resInfo == null){
         IResource res = this.resourceArray[intPos];
         resInfo = new ResourceInfo(res,this.getCollator(this.getLocale(res)));
         this.resourceInfoArray[intPos] = resInfo;
       }
       return resInfo;
     }
   */
  private ResourceInfo getResourceInfo(IResource res) throws ResourceException {
    String key = res.getRID().toString();
    ResourceInfo resInfo = (ResourceInfo) this.resourceInfoCache.get(key);
    if (resInfo == null) {
      resInfo = new ResourceInfo(res, this.getCollator(this.getLocale(res)));
      this.resourceInfoCache.put(key, resInfo);
    }

    return resInfo;
  }

  private int setInitialCompareCase(IPropertyName propertyName) {
    int compareCase = COMPARE_CASE_OTHER;
    if (propertyName.equals(displayNameProperty)) {
      compareCase = COMPARE_CASE_DISPLAY_NAME;
    } else if (LOCK_PROPERTY_NAME.equals(propertyName.getName())) {
      compareCase = COMPARE_CASE_LOCKED;
    } else if (COLLABORATION_STATUS_PROPERTY_NAME.equals(propertyName.getName())) {
      compareCase = COMPARE_CASE_COLL_STATUS;
    } else if (propertyName.getName().startsWith(CHILDCOUNT_PROPERTY_NAME)) {
      compareCase = COMPARE_CASE_CHILD_COUNT;
    } else if (PropertyName.createCreatedBy().equals(propertyName) || PropertyName.createLastModifiedBy().equals(propertyName)) {
      compareCase = COMPARE_CASE_USER;
    } else if (this.CM_ID.equals(propertyName.getName())) {
      compareCase = COMPARE_CASE_ID;
    }

    return compareCase;
  }

  /**
   * Description of the Method
   *
   * @param o1 Description of the param
   * @param o2 Description of the param
   * @return Description of the Return Value
   */
  public int compare(Object o1, Object o2) {
    int value = 0;
    try {
      /*
         ResourceInfo res1Info = null;
         ResourceInfo res2Info = null;
         if(o1 instanceof IResource){
           IResource r1 = (IResource)o1;
           IResource r2 = (IResource)o2;
           res1Info = this.getResourceInfo(r1);
           res2Info = this.getResourceInfo(r2);
         }else if(o1 instanceof Integer){
           Integer pos1 = (Integer)o1;
           Integer pos2 = (Integer)o2;
           res1Info = this.getResourceInfo(pos1);
           res2Info = this.getResourceInfo(pos2);
         }
       */
      IResource r1 = (IResource) o1;
      IResource r2 = (IResource) o2;
      ResourceInfo res1Info = this.getResourceInfo(r1);
      ResourceInfo res2Info = this.getResourceInfo(r2);
      value = this.compareIsCollection(res1Info, res2Info);
      if (value == 0) {
        switch (this.compareCase) {
        case COMPARE_CASE_DISPLAY_NAME:
          value = this.compareDisplayName(res1Info, res2Info);

          break;

        case COMPARE_CASE_LOCKED:
          value = this.compareIsLocked(res1Info, res2Info);

          break;

        case COMPARE_CASE_COLL_STATUS:
          value = this.compareCollaborationStatus(res1Info, res2Info);

          break;

        case COMPARE_CASE_CHILD_COUNT:
          value = this.compareChildCount(res1Info, res2Info);

          break;

        case COMPARE_CASE_USER:
          value = this.compareUser(res1Info, res2Info);

          break;

        case COMPARE_CASE_ID:
          value = this.compareID(res1Info, res2Info);

          break;

        default:
          value = this.compareByType(res1Info, res2Info);

          break;
        }
        if (!this.ascending) {
          value *= -1;
        }
        if ((value == 0) && this.displayNameAsSecondProperty) {
          value = this.compareDisplayNameAsSecond(res1Info, res2Info);
        }
      }
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareIsCollection(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    boolean r1isCollection = resInfo1.isCollection();
    boolean r2isCollection = resInfo2.isCollection();
    if (r1isCollection != r2isCollection) {
      if (r2isCollection) {
        value = 1;
      } else {
        value = -1;
      }
    }

    return value;
  }

  private int compareByType(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    IProperty p1 = null;
    IProperty p2 = null;
    try {
      p1 = resInfo1.getProperty();
      p2 = resInfo2.getProperty();
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }
    if ((p1 != null) && (p2 != null)) {
      PropertyType type = null;
      type = p1.getPropertyDef().getType();
      if (type == com.sapportals.wcm.repository.enum.PropertyType.BOOLEAN) {
        value = this.compareBoolean(p1, p2);
      } else if (type == com.sapportals.wcm.repository.enum.PropertyType.DATE) {
        value = this.compareDate(p1, p2);
      } else if (type == com.sapportals.wcm.repository.enum.PropertyType.INTEGER) {
        value = this.compareInt(p1, p2);
      } else if (type == com.sapportals.wcm.repository.enum.PropertyType.LONG) {
        value = this.compareLong(p1, p2);
      } else if (type == com.sapportals.wcm.repository.enum.PropertyType.STRING) {
        value = this.compareString(resInfo1, resInfo2);
      } else if (type == com.sapportals.wcm.repository.enum.PropertyType.XML) {
        value = this.compareXML(p1, p2);
      }
    } else {
      value = this.compareExistence(p1, p2);
    }

    return value;
  }

  private int compareInt(IProperty p1, IProperty p2) {
    int value = 0;
    Integer p1Integer = new Integer(p1.getIntValue());
    Integer p2Integer = new Integer(p2.getIntValue());
    value = p1Integer.compareTo(p2Integer);

    return value;
  }

  private int compareLong(IProperty p1, IProperty p2) {
    int value = 0;
    Long p1Long = new Long(p1.getLongIntValue());
    Long p2Long = new Long(p2.getLongIntValue());
    value = p1Long.compareTo(p2Long);

    return value;
  }

  private int compareString(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      CollationKey collationKey1 = resInfo1.getCollationKeyForProperty();
      CollationKey collationKey2 = resInfo2.getCollationKeyForProperty();
      if ((collationKey1 != null) && (collationKey2 != null)) {
        value = collationKey1.compareTo(collationKey2);
      } else {
        value = this.compareExistence(collationKey1, collationKey2);
      }
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareUser(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      CollationKey collationKey1 = resInfo1.getCollationKeyUserName();
      CollationKey collationKey2 = resInfo2.getCollationKeyUserName();
      value = collationKey1.compareTo(collationKey2);
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareID(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      CollationKey collationKey1 = resInfo1.getCollationKeyID();
      CollationKey collationKey2 = resInfo2.getCollationKeyID();
      value = collationKey1.compareTo(collationKey2);
    } catch (Exception e) {
    }

    return value;
  }

  private int compareXML(IProperty p1, IProperty p2) {
    int value = 0;
    String s1 = p1.getStringValue();
    String s2 = p2.getStringValue();
    if ((s1 != null) && (s2 != null)) {
      value = s1.compareToIgnoreCase(s2);
    } else {
      value = this.compareExistence(s1, s2);
    }

    return value;
  }

  private int compareBoolean(IProperty p1, IProperty p2) {
    int value = 0;
    boolean p1BooleanValue = p1.getBooleanValue();
    boolean p2BooleanValue = p2.getBooleanValue();
    if (p1BooleanValue == p2BooleanValue) {
      value = 0;
    } else if (p1BooleanValue == true) {
      value = 1;
    } else if (p1BooleanValue == false) {
      value = -1;
    }

    return value;
  }

  private int compareDate(IProperty p1, IProperty p2) {
    int value = 0;
    java.util.Date d1 = p1.getDateValue();
    java.util.Date d2 = p2.getDateValue();
    if ((d1 != null) && (d2 != null)) {
      value = d1.compareTo(d2);
    } else {
      value = this.compareExistence(d1, d2);
    }

    return value;
  }

  private int compareExistence(Object o1, Object o2) {
    int value = 0;
    if ((o1 != null) && (o2 != null)) {
      //both not null
      value = 0;
    } else if (o1 != null) {
      //o1 not null, o2 null
      value = 1;
    } else if (o2 != null) {
      //o1 null, o2 not null
      value = -1;
    } else {
      //both null
      value = 0;
    }

    return value;
  }

  private int compareDisplayName(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      value = resInfo1.getCollationKeyDisplayName().compareTo(resInfo2.getCollationKeyDisplayName());
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareDisplayNameAsSecond(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      value = resInfo1.getCollationKeyDisplayNameSecondProperty().compareTo(resInfo2.getCollationKeyDisplayNameSecondProperty());
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareIsLocked(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      value = resInfo1.isLocked().compareTo(resInfo2.isLocked());
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareCollaborationStatus(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      boolean p1BooleanValue = getCollaborationStatus(resInfo1);
      boolean p2BooleanValue = getCollaborationStatus(resInfo2);
      if (p1BooleanValue == p2BooleanValue) {
        value = 0;
      } else if (p1BooleanValue == true) {
        value = -1;
      } else if (p1BooleanValue == false) {
        value = 1;
      }
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private int compareChildCount(ResourceInfo resInfo1, ResourceInfo resInfo2) {
    int value = 0;
    try {
      value = getChildCountProperty(resInfo1, this.property.getName()).compareTo(getChildCountProperty(resInfo2, this.property.getName()));
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return value;
  }

  private String getChildCountProperty(ResourceInfo resInfo, String propertyName) {
    try {
      IResource res = resInfo.getResource();
      if ((res != null) && res.isCollection()) {
        ICollection coll = (ICollection) res;
        int totalChildren = coll.getChildrenCount(false, false, false);
        if (ResourcePropertyComparator.CHILDCOUNT_BOTH_PROPERTY_NAME.equals(propertyName)) {
          int folderChildren = coll.getChildrenCount(false, true, false);
          int resChildren = totalChildren - folderChildren;

          return folderChildren + " / " + resChildren;
        } else if (ResourcePropertyComparator.CHILDCOUNT_SUM_PROPERTY_NAME.equals(propertyName)) {
          return String.valueOf(totalChildren);
        } else if (ResourcePropertyComparator.CHILDCOUNT_FILES_PROPERTY_NAME.equals(propertyName)) {
          int folderChildren = coll.getChildrenCount(false, true, false);
          int resChildren = totalChildren - folderChildren;

          return String.valueOf(resChildren);
        } else if (ResourcePropertyComparator.CHILDCOUNT_FOLDERS_PROPERTY_NAME.equals(propertyName)) {
          int folderChildren = coll.getChildrenCount(false, true, false);

          return String.valueOf(folderChildren);
        }
      }
    } catch (Exception e) {
      //$JL-EXC$ 
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
    }

    return "0";
  }

  private boolean getCollaborationStatus(ResourceInfo resInfo) {
    try {
      IResource res = resInfo.getResource();
      Object obj = res.getContext().getUser().getTransientAttribute(res.getRID().toString() + ResourcePropertyComparator.CM_RND_COLLABORATION_STATUS);
      if ((obj != null) && obj instanceof Boolean) {
        return ((Boolean) obj).booleanValue();
      }
    } catch (Exception e) {
      //$JL-EXC$ 
      return false;
    }

    return false;
  }

  private Locale getLocale(IResource res) {
    if (this.locale == null) {
      this.locale = res.getContext().getLocale();
    }

    return this.locale;
  }

  private Collator getCollator(Locale locale) {
    if (this.collator == null) {
      this.collator = Collator.getInstance(locale);
    }

    return this.collator;
  }

  private class ResourceInfo {
    private Object property = null;
    private CollationKey collationKey1 = null;
    private CollationKey collationKey2 = null;
    private IResource resource = null;
    private IResource target = null;
    private boolean isCollection = false;
    private Integer isLocked = null;
    private Collator collator = null;

    public ResourceInfo(IResource resource, Collator collator) {
      this.resource = resource;
      this.collator = collator;
      this.isCollection = resource.isCollection();
      try {
        this.target = resource.getTargetResource();
      } catch (ResourceException e) {
        //$JL-EXC$ 
        this.target = null; // ignore
      }
    }

    public Integer isLocked() {
      if (this.isLocked == null) {
        this.isLocked = this.isLockedUncached();
      }

      return this.isLocked;
    }

    public IResource getResource() {
      return this.resource;
    }

    private Integer isLockedUncached() {
      try {
        if (this.resource.isLocked()) {
          if (this.resource.isLockedByMe()) {
            return new Integer(0);
          } else {
            return new Integer(1);
          }
        }
      } catch (Exception e) {
        //$JL-EXC$
        com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
      }

      return new Integer(2);
    }

    public boolean isCollection() {
      return this.isCollection;
    }

    public IProperty getProperty() throws ResourceException {
      IProperty foundProperty = null;
      if (this.property != null) {
        if (this.property instanceof IProperty) {
          //return cached property
          foundProperty = (IProperty) this.property;
        }
      } else {
        IProperty property = ((this.target == null) || (ResourcePropertyComparator.this.property.equals(PropertyName.createDisplayname())) ||
          (ResourcePropertyComparator.this.property.equals(PropertyName.createDescription())))
          ? this.resource.getProperty(ResourcePropertyComparator.this.property) : this.target.getProperty(ResourcePropertyComparator.this.property);
        if (property != null) {
          //cache found property
          this.property = property;
          //return found property
          foundProperty = property;
        } else {
          //cache info that property does not exist
          this.property = ResourcePropertyComparator.NullProperty;
        }
      }

      return foundProperty;
    }

    public CollationKey getCollationKeyForProperty() throws ResourceException {
      if (this.collationKey1 == null) {
        IProperty property = this.getProperty();
        if (property != null) {
          this.collationKey1 = this.collator.getCollationKey(property.getStringValue());
        }
      }

      return this.collationKey1;
    }

    public CollationKey getCollationKeyDisplayName() {
      if (this.collationKey1 == null) {
        String displayName = this.resource.getDisplayName(true);
        this.collationKey1 = this.collator.getCollationKey(displayName);
      }

      return this.collationKey1;
    }

    public CollationKey getCollationKeyUserName() throws ResourceException {
      if (this.collationKey1 == null) {
        String userId = this.getProperty().getStringValue();
        String userName = userId;
        IUserFactory userFactory = WPUMFactory.getUserFactory();
        try {
          if (userFactory != null) {
            userName = ((userId != null) ? ((userFactory.getUser(userId).getDisplayName() != null) ? userFactory.getUser(userId).getDisplayName() : userId) : "");
          }
        } catch (Exception e) {
          //$JL-EXC$ 
          com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
        }
        this.collationKey1 = this.collator.getCollationKey(userName);
      }

      return this.collationKey1;
    }

    public CollationKey getCollationKeyDisplayNameSecondProperty() {
      if (this.collationKey2 == null) {
        String displayName = this.resource.getDisplayName(true);
        this.collationKey2 = this.collator.getCollationKey(displayName);
      }

      return this.collationKey2;
    }

    public CollationKey getCollationKeyID() throws ResourceException {
      if (this.collationKey1 == null) {
        String id = this.getResource().getName();

        this.collationKey1 = this.collator.getCollationKey(id);
      }

      return this.collationKey1;
    }
  }
}
