/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;

/**
 * Implements the {@link IResourceList} interface.
 * <p>
 *  Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @see IResourceList
 * @version $Revision: 1.6 $
 */
public class ResourceList implements IResourceList {

  private final HashSet set;
  private final List list;

  /**
   * Construct a new resource list containing the elements of the specified list
   * in the same order.
   *
   * @param resourceList TBD: Description of the incoming method parameter
   */
  public ResourceList(ResourceList resourceList) {
    this.set = new HashSet(resourceList.list);
    this.list = new ArrayList(resourceList.list);
  }

  /**
   * Constructs a new empty resource list.
   */
  public ResourceList() {
    this.set = new HashSet();
    this.list = new ArrayList();
  }


  private ResourceList(List l) {
    this.set = new HashSet(l);
    this.list = new ArrayList(l);
  }

  /**
   * Package-private constructor for framework internal use.<p>
   * Constructs a new list with all of the elements of the specified collection,
   * in the order that they are returned by the collection's iterator.
   * @param c the collection whose elements to use.
   */
  ResourceList(Collection c) {
    this.set = new HashSet(c);
    this.list = new ArrayList(c);
  }

  /**
   * Package-private method for framework internal use.<p>
   * All items in the specified collection are added to the end of this list,
   * in the order that they are returned by the collection's iterator.
   * @param c the collection to add.
   */
  void addAll(Collection c) {
    this.set.addAll(c);
    this.list.addAll(c);
  }

  // ---------------------------------------------------------------------------
  // public
  // ---------------------------------------------------------------------------


  public String toString() {
    return this.list.toString();
  }


  // ---------------------------------------------------------------------------
  // IResourceList interface
  // ---------------------------------------------------------------------------


  public int size() {
    return this.list.size();
  }

  public boolean containsResource(IResource resource) {
    if (resource == null) {
      throw new NullPointerException("resource parameter is null");
    }
    return this.set.contains(resource);
  }

  public boolean add(IResource resource) {
    if (resource == null) {
      throw new NullPointerException("resource parameter is null");
    }
    this.set.add(resource);
    return this.list.add(resource);
  }

  public boolean addAll(IResourceList list) {
    if (list == null) {
      throw new NullPointerException("list parameter is null");
    }
    if (list instanceof ResourceList) {
      this.set.addAll(((ResourceList)list).set);
      return this.list.addAll(((ResourceList)list).list);
    }
    else {
      return false;
    }
  }

  public boolean remove(IResource resource) {
    if (resource == null) {
      throw new NullPointerException("resource parameter is null");
    }
    this.set.remove(resource);
    return this.list.remove(resource);
  }


  public IResource get(int index) {
    return (IResource)this.list.get(index);
  }

  public IResource set(int index, IResource resource) {
    if (resource == null) {
      throw new NullPointerException("resource parameter is null");
    }
    this.set.remove(this.list.get(index));
    this.set.add(resource);
    return (IResource)this.list.set(index, resource);
  }

  public void add(int index, IResource resource) {
    if (resource == null) {
      throw new NullPointerException("resource parameter is null");
    }
    this.set.add(resource);
    this.list.add(index, resource);
  }

  public IResource remove(int index) {
    IResource resource = (IResource)this.list.remove(index); 
    this.set.remove(resource);
    return resource;
  }

  public int indexOf(IResource resource) {
    return this.list.indexOf(resource);
  }

  public int lastIndexOf(IResource resource) {
    return this.list.lastIndexOf(resource);
  }

  public IResourceListIterator listIterator() {
    return new ResourceListIterator(this.set, this.list.listIterator());
  }


  public IResourceListIterator listIterator(int index) {
    return new ResourceListIterator(this.set, this.list.listIterator(index));
  }


  public IResourceList subList(int index1, int index2) {
    return new ResourceList(this.list.subList(index1, index2));
  }

  public void sort(Comparator comp) {
    Collections.sort(this.list, comp);
  }


  public IResourceList union(IResourceList list) {
    if (list == null) {
      return new ResourceList(this);
    }
    ResourceList result = new ResourceList(this);
    IResourceListIterator iterator = list.listIterator();
    while (iterator.hasNext()) {
      IResource resource = iterator.next();
      if (!result.containsResource(resource)) {
        result.add(resource);
      }
    }
    return result;
  }


  public IResourceList intersection(IResourceList list) {
    if (list == null) {
      return new ResourceList();
    }
    ResourceList result = new ResourceList();
    IResourceListIterator iterator = this.listIterator();
    while (iterator.hasNext()) {
      IResource resource = iterator.next();
      if (list.containsResource(resource)) {
        result.add(resource);
      }
    }
    return result;
  }


  public IResourceList difference(IResourceList list) {
    if (list == null) {
      return new ResourceList(this);
    }
    ResourceList result = new ResourceList();
    IResourceListIterator iterator = this.listIterator();
    while (iterator.hasNext()) {
      IResource resource = iterator.next();
      if (!list.containsResource(resource)) {
        result.add(resource);
      }
    }
    return result;
  }
}
