/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: ResourceContext.java,v 1.4 2004/05/27 11:33:52 jre Exp $
 */

package com.sapportals.wcm.repository;
import com.sap.tc.logging.Location;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;

import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.util.*;

import java.util.HashMap;

/**
 * Instances of this class contain information about the context the client is
 * running is (request, session, etc.)
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @version $Revision: 1.4 $
 */
public class ResourceContext implements IResourceContext, java.io.Serializable {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.ResourceContext.class);

  /**
   * testing ?
   */
  protected final static String SESSION = "_resourcecontext_sessionobject";

  /**
   * some properties used by <code>setValue()</code> /<code>getValue()</code>
   */
  protected HashMap properties = null;

  /**
   * the locale object
   */
  protected Locale locale = null;

  /**
   * the current user in that context
   */
  protected transient IUser user = null;

  /**
   * URI placeholder used by resource factory
   */
  protected Properties placeholders = null;

  private final long creationTime;

  public ResourceContext(IUser user) {
    this(user, null);
  }

  public ResourceContext(IUser user, Locale fallbackLocale) {
    this(user, fallbackLocale, false);
  }

  public ResourceContext(IUser user, Locale locale, boolean isPrimaryLocale) {
    this.creationTime = System.currentTimeMillis();

    if (user == null) {
      this.locale = Locale.getDefault();
      // generate user ANONYMOUS
      try {
        IUserFactory uf = WPUMFactory.getUserFactory();
        if (uf == null) {
          ResourceContext.log.fatalT("ResourceContext(83)", "Cannot instantiate user factory");
          return;
        }
        this.user = WPUMFactory.getUserFactory().getUser(IUser.ANONYMOUS);

        if (this.user == null) {
          ResourceContext.log.errorT("ResourceContext(89)", "Cannot instantiate user <" + IUser.ANONYMOUS + ">");
        }
        else if (ResourceContext.log.beInfo()) {
          ResourceContext.log.infoT("ResourceContext(92)", "User in context is <" + IUser.ANONYMOUS + ">");
        }
      }
      catch (Exception ex) {
        //$JL-EXC$ 
        ResourceContext.log.errorT("ResourceContext(96)", "Cannot instantiate user <" + IUser.ANONYMOUS + ">, factory failure: " + LoggingFormatter.extractCallstack(ex));
      }
    }
    else {
      if (ResourceContext.log.beDebug()) {
        ResourceContext.log.debugT("ResourceContext(101)", "User in context is <" + user.getId() + ">");
      }

      this.user = user;

      if (isPrimaryLocale && locale != null) {
        this.locale = locale;
      }
      else {
        if (this.user != null) {
          this.locale = this.user.getLocale();
        }
      }
      if (this.locale == null) {
        this.locale = (locale == null) ? Locale.getDefault() : locale;
      }
    }

    if (user != null && !user.isAuthenticated()) {
      ResourceContext.log.warningT("ResourceContext(120)", "ResourceContext.setUser called with unauthenticated user <" + user.getId() + ">: " + LoggingFormatter.extractCallstack(new Exception("stacktrace for debugging")));
    }
  }

  private ResourceContext(long creationTime, IUser user, Locale locale, HashMap properties, Properties placeholders) {
    this.creationTime = creationTime;
    this.locale = locale;
    this.user = user;
    this.properties = properties;
    this.placeholders = placeholders;
  }

  public long getCreationTime() {
    return this.creationTime;
  }

  // --- IResourceContext ---


  public synchronized void setObjectValue(String name, Object value)
    throws ResourceException {
    if (name == null) {
      throw new ResourceException("Parameter name is null");
    }
    if (name.length() == 0) {
      throw new ResourceException("Parameter name is empty");
    }
    if (this.properties == null) {
      this.properties = new HashMap();
    }
    this.properties.put(name, value);
  }

  public Object getObjectValue(String name) {
    if (this.properties == null || name == null) {
      return null;
    }
    return this.properties.get(name);
  }


  public synchronized void setValue(String name, String value)
    throws ResourceException {
    setObjectValue(name, (Object)value);
  }

  public String getValue(String name) {
    return (String)getObjectValue(name);
  }

  public void setValues(HashMap properties) {
    this.properties = properties;
  }

  public HashMap getValues() {
    return this.properties;
  }

  public Locale getLocale() {
    return (this.locale == null) ? Locale.getDefault() : this.locale;
  }

  public void setLocale(Locale locale) {
    this.locale = locale;
  }

  public void setUser(IUser user) {
    if (user == null) {
      ResourceContext.log.warningT("setUser(188)", "ResourceContext.setUser called with null user: " + LoggingFormatter.extractCallstack(new Exception("stacktrace for debugging")));
    }
    else if (!user.isAuthenticated()) {
      ResourceContext.log.warningT("setUser(191)", "ResourceContext.setUser called with unauthenticated user <" + user.getId() + ">: " + LoggingFormatter.extractCallstack(new Exception("stacktrace for debugging")));
    }
    this.user = user;
  }

  public IUser getUser() {
    return this.user;
  }


  /**
   * Clone resource context.
   */
  public IResourceContext cloneContext() {
    return new ResourceContext(creationTime, user, locale, properties, placeholders);
  }
}

