package com.sapportals.wcm.repository;

import java.util.List;

/**
 * 
 * @author d032991
 * @since EP5, SP6, Patch1, Hotfix1; EP6, SP2_CP
 * This class defines a selection criteria based on a single property. Depending on 
 * its <code>include</code>-Attribute, it will either let each resource that fulfills
 * the property pass this filter or fail this filter. This selector class can also 
 * handle multi valued properties. 
 */
public final class PropertySelectorItem {

  private static com.sap.tc.logging.Location log =
    com.sap.tc.logging.Location.getLocation(PropertySelectorItem.class.getName());

  private boolean include = true;
  private IProperty property = null;

  /**
   * @param include flag to define, how the #pass(IProperty prop) method will behave:
   * if it is set to <code>true</code>, only resources that have the required property
   * will pass. If it is set to false, only resources that have the required property
   * will not pass.
   * @param property the property to check. Must not be <code>null</code>
   */
  public PropertySelectorItem(boolean include, IProperty property) {
    if (property == null) {
      throw new NullPointerException("The parameter <property> must not be null");
    }
    this.include = include;
    this.property = property;
  }

  private boolean getInclude() {
    return include;
  }

  private IProperty getProperty() {
    return property;
  }

  public IPropertyName getPropertyName() {
    return this.property.getPropertyName();
  }

  //  public boolean equals(Object o) {
  //    if (o instanceof PropertySelectorItem) {
  //      PropertySelectorItem ps = (PropertySelectorItem)o;
  //      return ((this.include == ps.include) && (this.property.equals(ps.property)));
  //    }
  //    return false;
  //  }

  public String toString() {
    return "PropertySelectorItem with Include-Mode set to <"
      + this.getInclude()
      + ">, Property <"
      + this.getProperty().getPropertyName()
      + "> with value <"
      + this.getProperty().toString()
      + ">";
  }

  /**
   * checks, if the given property will pass this selector or not. Behavior varies, depending on <br>
   * the value of the global include-attribute set in the constructor <br>
   * if it is a multi valued property or a single valued property. <br>
   * If the property-types of this class and the provided property do not match, this method will
   * always return <code>false</code>. <br>
   * Include mode set to <code>true</code>:
   * if the checkProp-parameter is <code>null</code>, the method will return <code>false</code>
   * If the checkProp equals the property of this class or one of the values of the checkProp equals
   * one of the values of the property of this class, the method will return <code>true</code>. <br>
   * Include mode set to <code>false</code>:
   * If the checkProp-parameter is <code>null</code>, the method will return <code>true</code>   
   * If the checkProp equals the property of this class or one of the values of the checkProp equals
   * one of the values of the property of this class, the method will return <code>false</code>. <br>
   * @param checkProp the property to check, might be <code>null</code>
   * @return flag, indicating if the given property will pass this selector
   * @throws ResourceException 
   */
  public boolean pass(IProperty checkProp) throws ResourceException {
    if (checkProp == null) { // property not found: check if it must be set
      return (!this.getInclude());
    }

    // check if property types are equal
    if (!checkProp.getType().equals(this.getProperty().getType())) {
      if (log.beWarning()) {
        log.warningT(
          "Mismatch in property types, will filter resource out. Type of resource property is <"
            + checkProp.getType().toString()
            + ">, Type of filter property is <"
            + this.getProperty().getType().toString()
            + ">");
        return false; //this must never match!
      }
    }

    //multi valued property?
    if (this.getProperty().isMultivalued()) {
      return this.passMultiValued(checkProp);
    }
    else {
      return (this.checkPropertyValue(this.getProperty().getValue(), checkProp));
    }
  }

  private boolean passMultiValued(IProperty checkProp) throws ResourceException {
    List sel_props = this.getProperty().getValues();
    int sel_prop_count = sel_props.size() - 1;

    /*
     * if a resource matches just one of the criteria of this multivalued property,
     * it will pass this filter. Therefore, as soon as we find a matching property,
     * we can abort the loop and just return true
     */
    if (this.getInclude()) {
      while (sel_prop_count >= 0) {
        if (this.checkPropertyValue(sel_props.get(sel_prop_count--), checkProp)) {
          return true;
        }
      }
      return false;
    }

    /*
     * we have to loop over all property values here. If the resource matches just one of these
     * property values, it will be filtered out.
     * Example: This class defines a filter for a mimetype, stating that all resources with either
     * mimetype "a" or "b" must be filtered out. As a resource can only have one mimetype, let's say
     * the resource has mimetype "b". The test for "a" will let the resource pass, so we keep on looping.
     * The test for mimetype "b" will filter out that resource.
     */
    while (sel_prop_count >= 0) {
      if (!this.checkPropertyValue(sel_props.get(sel_prop_count--), checkProp)) {
        //result = false;
        return false;
      }
    }
    return true;

  }

  private boolean checkPropertyValue(Object value, IProperty resProp) throws ResourceException {
    if (resProp.isMultivalued()) {
      if (this.getInclude() != resProp.getValues().contains(value)) {
        if (log.beDebug()) {
          log.debugT(
            "property does not match - filter out current resource: "
              + resProp.getPropertyName()
              + " current values: "
              + resProp.getValues()
              + ", expected value: "
              + value,
            ", inclusion mode: " + include);
        }
        return false;
      }
    }
    else {
      if (this.getInclude() != value.equals(resProp.getValue())) {
        if (log.beDebug()) {
          log.debugT(
            "property does not match - filter out current resource: "
              + resProp.getPropertyName()
              + " current value: "
              + resProp.getValue()
              + ", expected value: "
              + value,
            ", inclusion mode: " + include);
        }
        return false;
      }
    }
    return true;
  }

}
