/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: PropertyDef.java,v 1.3 2004/05/27 11:33:52 jre Exp $
 */

package com.sapportals.wcm.repository;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import com.sapportals.wcm.WcmObject;
import com.sapportals.wcm.repository.enum.PropertyType;

/**
 * Information about the type, name and other attributes of a property.
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @version $Revision: 1.3 $
 */
public class PropertyDef extends WcmObject implements IPropertyDef {

  private final static Map MAP_SV;
  private final static Map MAP_SV_RO;
  private final static Map MAP_SV_HI;
  private final static Map MAP_SV_RO_HI;
  private final static Map MAP_MV;

  private static PropertyDef PD_SYSTEM_CREATED_BY;
  private static PropertyDef PD_SYSTEM_LASTMODIFIED_BY;
  private static PropertyDef PD_SYSTEM_CREATIONDATE;
  private static PropertyDef PD_SYSTEM_HIDDEN;
  private static PropertyDef PD_SYSTEM_LASTMODIFIED;
  private static PropertyDef PD_SYSTEM_DESCRIPTION;
  private static PropertyDef PD_SYSTEM_DISPLAYNAME;
  private static PropertyDef PD_SYSTEM_READONLY;
  private static PropertyDef PD_SYSTEM_CONTENT_LENGTH;
  private static PropertyDef PD_SYSTEM_CONTENT_TYPE;
  private static PropertyDef PD_SYSTEM_ETAG;
  private static PropertyDef PD_SYSTEM_CONTENT_LANGUAGE;
  private static PropertyDef PD_SYSTEM_RESOURCETYPE;

  static {
    Map map = new HashMap(17);
    map.put(PropertyType.BOOLEAN, new PropertyDef(PropertyType.BOOLEAN, true, false, false));
    map.put(PropertyType.INTEGER, new PropertyDef(PropertyType.INTEGER, true, false, false));
    map.put(PropertyType.LONG, new PropertyDef(PropertyType.LONG, true, false, false));
    map.put(PropertyType.STRING, new PropertyDef(PropertyType.STRING, true, false, false));
    map.put(PropertyType.XML, new PropertyDef(PropertyType.XML, true, false, false));
    map.put(PropertyType.DATE, new PropertyDef(PropertyType.DATE, true, false, false));
    MAP_MV = map;

    map = new HashMap(17);
    map.put(PropertyType.BOOLEAN, new PropertyDef(PropertyType.BOOLEAN, false, false, false));
    map.put(PropertyType.INTEGER, new PropertyDef(PropertyType.INTEGER, false, false, false));
    map.put(PropertyType.LONG, new PropertyDef(PropertyType.LONG, false, false, false));
    map.put(PropertyType.STRING, new PropertyDef(PropertyType.STRING, false, false, false));
    map.put(PropertyType.XML, new PropertyDef(PropertyType.XML, false, false, false));
    map.put(PropertyType.DATE, new PropertyDef(PropertyType.DATE, false, false, false));
    MAP_SV = map;

    map = new HashMap(17);
    map.put(PropertyType.BOOLEAN, new PropertyDef(PropertyType.BOOLEAN, false, true, false));
    map.put(PropertyType.INTEGER, new PropertyDef(PropertyType.INTEGER, false, true, false));
    map.put(PropertyType.LONG, new PropertyDef(PropertyType.LONG, false, true, false));
    map.put(PropertyType.STRING, new PropertyDef(PropertyType.STRING, false, true, false));
    map.put(PropertyType.XML, new PropertyDef(PropertyType.XML, false, true, false));
    map.put(PropertyType.DATE, new PropertyDef(PropertyType.DATE, false, true, false));
    MAP_SV_RO = map;

    map = new HashMap(17);
    map.put(PropertyType.BOOLEAN, new PropertyDef(PropertyType.BOOLEAN, false, true, true));
    map.put(PropertyType.INTEGER, new PropertyDef(PropertyType.INTEGER, false, true, true));
    map.put(PropertyType.LONG, new PropertyDef(PropertyType.LONG, false, true, true));
    map.put(PropertyType.STRING, new PropertyDef(PropertyType.STRING, false, true, true));
    map.put(PropertyType.XML, new PropertyDef(PropertyType.XML, false, true, true));
    map.put(PropertyType.DATE, new PropertyDef(PropertyType.DATE, false, true, true));
    MAP_SV_RO_HI = map;

    map = new HashMap(17);
    map.put(PropertyType.BOOLEAN, new PropertyDef(PropertyType.BOOLEAN, false, false, true));
    map.put(PropertyType.INTEGER, new PropertyDef(PropertyType.INTEGER, false, false, true));
    map.put(PropertyType.LONG, new PropertyDef(PropertyType.LONG, false, false, true));
    map.put(PropertyType.STRING, new PropertyDef(PropertyType.STRING, false, false, true));
    map.put(PropertyType.XML, new PropertyDef(PropertyType.XML, false, false, true));
    map.put(PropertyType.DATE, new PropertyDef(PropertyType.DATE, false, false, true));
    MAP_SV_HI = map;

    try {
      PropertyDef.PD_SYSTEM_CREATED_BY = new PropertyDef(PropertyType.STRING, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_LASTMODIFIED_BY = new PropertyDef(PropertyType.STRING, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_CREATIONDATE = new PropertyDef(PropertyType.DATE, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_HIDDEN = new PropertyDef(PropertyType.BOOLEAN, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_LASTMODIFIED = new PropertyDef(PropertyType.DATE, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_DESCRIPTION = new PropertyDef(PropertyType.STRING, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_DISPLAYNAME = new PropertyDef(PropertyType.STRING, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_READONLY = new PropertyDef(PropertyType.BOOLEAN, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_CONTENT_LENGTH = new PropertyDef(PropertyType.LONG, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_CONTENT_TYPE = new PropertyDef(PropertyType.STRING, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_ETAG = new PropertyDef(PropertyType.STRING, false, false, true, false, false);
      PropertyDef.PD_SYSTEM_CONTENT_LANGUAGE = new PropertyDef(PropertyType.STRING, false, false, false, false, false);
      PropertyDef.PD_SYSTEM_RESOURCETYPE = new PropertyDef(PropertyType.STRING, false, false, true, false, false);
    }
    catch (ResourceException ex) {
      //$JL-EXC$ 
      throw new RuntimeException("failed to initialize system property definitions: " + ex.getMessage());
    }
  }

  public final static PropertyDef valueOf(PropertyType type, boolean isMultivalued) {
    PropertyDef pd = (PropertyDef)(isMultivalued ? MAP_MV.get(type) : MAP_SV.get(type));
    if (pd != null) {
      return pd;
    }
    else {
      return new PropertyDef(type, isMultivalued);
    }
  }

  public final static PropertyDef valueOf(PropertyType type, boolean isMultivalued,
    boolean isReadOnly, boolean isHidden) {
    if (!isReadOnly && !isHidden) {
      return valueOf(type, isMultivalued);
    }
    else {
      Map map = null;
      if (!isMultivalued) {
        // Not Multivalued
        //
        if (isReadOnly) {
          if (isHidden) {
            // readonly + hidden
            map = MAP_SV_RO_HI;
          }
          else {
            // readonly, not hidden
            map = MAP_SV_RO;
          }
        }
        else {
          // not readonly, hidden
          map = MAP_SV_HI;
        }
      }

      PropertyDef pd = null;
      if (map != null) {
        pd = (PropertyDef)map.get(type);
      }

      if (pd == null) {
        pd = new PropertyDef(type, isMultivalued, isReadOnly, isHidden);
      }

      return pd;
    }
  }

  public static IPropertyDef createCreatedBy() {
    return PropertyDef.PD_SYSTEM_CREATED_BY;
  }

  public static IPropertyDef createLastModifiedBy() {
    return PropertyDef.PD_SYSTEM_LASTMODIFIED_BY;
  }

  public static IPropertyDef createCreationDate() {
    return PropertyDef.PD_SYSTEM_CREATIONDATE;
  }

  public static IPropertyDef createHidden() {
    return PropertyDef.PD_SYSTEM_HIDDEN;
  }

  public static IPropertyDef createLastModified() {
    return PropertyDef.PD_SYSTEM_LASTMODIFIED;
  }

  public static IPropertyDef createDescription() {
    return PropertyDef.PD_SYSTEM_DESCRIPTION;
  }

  public static IPropertyDef createDisplayname() {
    return PropertyDef.PD_SYSTEM_DISPLAYNAME;
  }

  public static IPropertyDef createReadOnly() {
    return PropertyDef.PD_SYSTEM_READONLY;
  }

  public static IPropertyDef createContentLength() {
    return PropertyDef.PD_SYSTEM_CONTENT_LENGTH;
  }

  public static IPropertyDef createContentType() {
    return PropertyDef.PD_SYSTEM_CONTENT_TYPE;
  }

  public static IPropertyDef createETag() {
    return PropertyDef.PD_SYSTEM_ETAG;
  }

  public static IPropertyDef createContentLanguage() {
    return PropertyDef.PD_SYSTEM_CONTENT_LANGUAGE;
  }

  public static IPropertyDef createResourceType() {
    return PropertyDef.PD_SYSTEM_RESOURCETYPE;
  }


  private final PropertyType type;
  private final boolean isMultivalued;
  private final boolean isRequired;
  private final boolean isReadonly;
  private final boolean isIndexed;
  private final boolean isHidden;
  private final Properties attributes;

  private int hash = 0;

  /**
   * Constructs a new PropertyDef
   *
   * @param type The property Type
   * @param isMultivalued If the property has multible values
   * @param isRequired If the property can not be deleted
   * @param isReadonly If the property value can not be changed
   * @param isIndexed If the property value is indexed
   * @param isHidden TBD: Description of the incoming method parameter
   * @exception ResourceException If the property name contains illegal
   *      characters (e.g. whitespace)
   */
  public PropertyDef(PropertyType type,
    boolean isMultivalued,
    boolean isRequired,
    boolean isReadonly,
    boolean isIndexed,
    boolean isHidden)
    throws ResourceException {
    if (type == null) {
      throw new IllegalArgumentException("type is null");
    }
    this.type = type;
    this.isMultivalued = isMultivalued;
    this.isRequired = isRequired;
    this.isReadonly = isReadonly;
    this.isIndexed = isIndexed;
    this.isHidden = isHidden;
    this.attributes = null;
  }


  /**
   * Constructs a new PropertyDef
   *
   * @param type The property Type
   * @param isMultivalued If the property has multible values
   * @param isRequired If the property can not be deleted
   * @param isReadonly If the property value can not be changed
   * @param isIndexed If the property value is indexed
   * @param isHidden TBD: Description of the incoming method parameter
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException If the property name contains illegal
   *      characters (e.g. whitespace)
   */
  public PropertyDef(PropertyType type,
    boolean isMultivalued,
    boolean isRequired,
    boolean isReadonly,
    boolean isIndexed,
    boolean isHidden,
    Properties attributes)
    throws ResourceException {
    if (type == null) {
      throw new ResourceException("type is null");
    }
    this.type = type;
    this.isMultivalued = isMultivalued;
    this.isRequired = isRequired;
    this.isReadonly = isReadonly;
    this.isIndexed = isIndexed;
    this.isHidden = isHidden;
    this.attributes = attributes;
  }

  /**
   * Construct a new PropertyDef
   *
   * @param def TBD: Description of the incoming method parameter
   */
  public PropertyDef(PropertyDef def) {
    if (def == null) {
      throw new java.lang.NullPointerException();
    }
    this.type = def.type;
    this.isMultivalued = def.isMultivalued;
    this.isRequired = def.isRequired;
    this.isReadonly = def.isReadonly;
    this.isIndexed = def.isIndexed;
    this.isHidden = def.isHidden;
    this.attributes = def.attributes;
  }

  public PropertyDef(PropertyType type, boolean isMultivalued) {
    if (type == null) {
      throw new NullPointerException("type is null");
    }
    this.type = type;
    this.isMultivalued = isMultivalued;
    this.isRequired = false;
    this.isReadonly = false;
    this.isIndexed = false;
    this.isHidden = false;
    this.attributes = null;
  }

  private PropertyDef(PropertyType type, boolean isMultivalued, boolean isReadonly, boolean isHidden) {
    if (type == null) {
      throw new NullPointerException("type is null");
    }
    this.type = type;
    this.isMultivalued = isMultivalued;
    this.isRequired = false;
    this.isReadonly = isReadonly;
    this.isIndexed = false;
    this.isHidden = isHidden;
    this.attributes = null;
  }

  /**
   * Construct a new PropertyDef
   *
   * @param def TBD: Description of the incoming method parameter
   */
  protected PropertyDef(IPropertyDef def) {
    if (def == null) {
      throw new java.lang.NullPointerException();
    }
    this.type = def.getType();
    this.isMultivalued = def.isMultivalued();
    this.isRequired = def.isRequired();
    this.isReadonly = def.isReadonly();
    this.isIndexed = def.isIndexed();
    this.isHidden = def.isHidden();
    this.attributes = def.getAttributes();
  }

  //////////////////////////////////////////////////////////////////////////////
  // public
  //////////////////////////////////////////////////////////////////////////////

  /**
   * Hashcode
   *
   * @return TBD: Description of the outgoing return value
   */
  public int hashCode() {
    if (this.hash == 0) {
      this.hash = (hashCode(this.type) << 16)
         | valueOf(this.isMultivalued, 8)
         | valueOf(this.isHidden, 7)
         | valueOf(this.isIndexed, 6)
         | valueOf(this.isReadonly, 5)
         | valueOf(this.isRequired, 4);
    }
    return this.hash;
  }

  /**
   * Compare two PropertyDef objects for equality
   *
   * @param propDef TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public boolean equals(Object propDef) {
    if (!(propDef instanceof IPropertyDef)) {
      return false;
    }
    IPropertyDef other = (IPropertyDef)propDef;
    return this.type.equals(other.getType())
       && this.isMultivalued == other.isMultivalued()
       && this.isHidden == other.isHidden()
       && this.isIndexed == other.isIndexed()
       && this.isReadonly == other.isReadonly()
       && this.isRequired == other.isRequired();
  }


  /**
   * String representation of a property def
   *
   * @return TBD: Description of the outgoing return value
   */
  public String toString() {
    return "(PT "
       + toString(this.type)
       + " multi:" + valueOf(this.isMultivalued, 0)
       + " hidden: " + valueOf(this.isHidden, 0)
       + " indexed: " + valueOf(this.isIndexed, 0)
       + " readonly: " + valueOf(this.isReadonly, 0)
       + " required: " + valueOf(this.isRequired, 0);
  }


  //////////////////////////////////////////////////////////////////////////////
  // IPropertyDef interface
  //////////////////////////////////////////////////////////////////////////////


  /**
   * Returns true if the property is multi-valued.
   *
   * @return A boolean value
   */
  public boolean isMultivalued() {
    return this.isMultivalued;
  }


  /**
   * Returns the value of the Required attribute
   *
   * @return A boolean value
   */
  public boolean isRequired() {
    return this.isRequired;
  }

  /**
   * Returns the type of the property.
   *
   * @return type
   */
  public PropertyType getType() {
    return this.type;
  }


  /**
   * Returns the value of the Readonly attribute
   *
   * @return readonly
   */
  public boolean isReadonly() {
    return this.isReadonly;
  }

  /**
   * Returns the value of the Indexed attribute
   *
   * @return indexed
   */
  public boolean isIndexed() {
    return this.isIndexed;
  }

  /**
   * Returns the value of the Hidden attribute
   *
   * @return hidden
   */
  public boolean isHidden() {
    return this.isHidden;
  }


  /**
   * Get the value of an attribute.
   *
   * @param name Attribute name
   * @return The value
   */
  public String getAttribute(String name) {
    if (this.attributes == null) {
      return null;
    }
    else {
      return this.attributes.getProperty(name);
    }
  }


  /**
   * Returns a collection of attributes
   *
   * @return All attributes
   */
  public Properties getAttributes() {
    if (this.attributes == null) {
      return null;
    }
    else {
      return (Properties)this.attributes.clone();
    }
  }


  //-------------------------------- private -----------------------------------

  private final static int hashCode(PropertyType type) {
    if (PropertyType.STRING.equals(type)) {
      return 1;
    }
    else if (PropertyType.DATE.equals(type)) {
      return 2;
    }
    else if (PropertyType.INTEGER.equals(type)) {
      return 3;
    }
    else if (PropertyType.XML.equals(type)) {
      return 4;
    }
    if (PropertyType.BOOLEAN.equals(type)) {
      return 5;
    }
    else if (PropertyType.LONG.equals(type)) {
      return 6;
    }
    else {
      return type.hashCode();
    }
  }

  private final static int valueOf(boolean value, int shift) {
    return value ? (1 << shift) : 0;
  }

  private final static String toString(PropertyType type) {
    if (PropertyType.STRING.equals(type)) {
      return "String";
    }
    else if (PropertyType.DATE.equals(type)) {
      return "Date";
    }
    else if (PropertyType.INTEGER.equals(type)) {
      return "Integer";
    }
    else if (PropertyType.XML.equals(type)) {
      return "Xml";
    }
    if (PropertyType.BOOLEAN.equals(type)) {
      return "Boolean";
    }
    else if (PropertyType.LONG.equals(type)) {
      return "Long";
    }
    else {
      return "Unknown";
    }
  }

}

