/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: Property.java,v 1.5 2004/05/27 11:33:52 jre Exp $
 */

package com.sapportals.wcm.repository;

import java.text.DateFormat;
import java.util.*;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.WcmObject;
import com.sapportals.wcm.repository.enum.PropertyType;
import com.sapportals.wcm.util.resource.ResourceBundles;

/**
 * A read-only property of a resource object.
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @version $Revision: 1.5 $
 */
public class Property extends WcmObject implements IProperty, java.io.Serializable {

  private final static int FIXED_COUNT = 17;
  private static IPropertyName[] g_systemNames = null;
  private static ResourceBundles res = ResourceBundles.getBundle("com.sapportals.wcm.repository.PropertyRes");

  /**
   * The property name
   */
  private final IPropertyName m_name;

  /**
   * Multi-valued flag
   */
  private final boolean m_isMultivalued;

  /**
   * The property data type
   */
  private final PropertyType m_type;

  /**
   * Reference to the property definition
   */
  private final IPropertyDef m_def;

  /**
   * The value
   */
  protected String m_value = null;
  protected Object m_value_object = null;

  /**
   * The value if the type is Date
   */
  protected Date m_dateValue = null;

  /**
   * Values of multivalued property
   */
  protected List m_values = null;

  /**
   * The collection of property attributes
   */
  protected Properties m_attributes = null;

  protected ResourceBundles m_res = Property.res;

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param value The property string value
   */
  public Property(IPropertyName name, String value)
    throws ResourceException {
    this(name, PropertyType.STRING, false);

    m_value = value;
    m_value_object = value;    
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param value The property date value
   */
  public Property(IPropertyName name, Date value)
    throws ResourceException {
    this(name, PropertyType.DATE, false);

    m_dateValue = value;
    m_value_object = value;
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param value The property boolean value
   */
  public Property(IPropertyName name, Boolean value)
    throws ResourceException {
    this(name, PropertyType.BOOLEAN, false);

    m_value = value.toString();
    m_value_object = value;    
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param value The property integer value
   */
  public Property(IPropertyName name, Integer value)
    throws ResourceException {
    this(name, PropertyType.INTEGER, false);

    m_value = value.toString();
    m_value_object = value;
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param value The property long value
   */
  public Property(IPropertyName name, Long value)
    throws ResourceException {
    this(name, PropertyType.LONG, false);

    m_value = value.toString();
    m_value_object = value;    
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param value The property xml value
   */
  public Property(IPropertyName name, XMLMarkup value)
    throws ResourceException {
    this(name, PropertyType.XML, false);

    m_value = value.toString();
    m_value_object = value;    
  }

  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param type The property data type
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public Property(IPropertyName name, PropertyType type, List values)
    throws ResourceException {
    this(name, type, true);

    if (values != null) {
      int count = values.size() - 1;
      while (count >= 0) {
        this.validateType(values.get(count));
        count--;
      }
      m_values = Collections.unmodifiableList(new ArrayList(values));
    }
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup.
   *
   * @param name The property name
   * @param value The value object
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed
   */
  public Property(IPropertyName name, Object value)
    throws ResourceException {
    if (name == null) {
      throw new NullPointerException();
    }
    m_name = name;
    m_isMultivalued = false;

    if (value instanceof String) {
      m_type = PropertyType.STRING;
    }
    else if (value instanceof Integer) {
      m_type = PropertyType.INTEGER;
    }
    else if (value instanceof Long) {
      m_type = PropertyType.LONG;
    }
    else if (value instanceof Boolean) {
      m_type = PropertyType.BOOLEAN;
    }
    else if (value instanceof Date) {
      m_type = PropertyType.DATE;
      m_dateValue = (Date)value;
    }
    else if (value instanceof XMLMarkup) {
      m_type = PropertyType.XML;
    }
    else {
      throw new ResourceException("Illegal property value instance: " + value.getClass().getName());
    }

    m_def = PropertyDef.valueOf(m_type, m_isMultivalued);
    m_value = value.toString();
    m_value_object = value;    
  }

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, String value)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.STRING)) {
      throw new ResourceException("PropertyType in PropertyDef is not STRING");
    }

    m_value = value;
    m_value_object = value;    
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Date value)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.DATE)) {
      throw new ResourceException("PropertyType in PropertyDef is not DATE");
    }
    m_dateValue = value;
    m_value_object = value;
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Boolean value)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.BOOLEAN)) {
      throw new ResourceException("PropertyType in PropertyDef is not BOOLEAN");
    }
    m_value = value.toString();
    m_value_object = value;
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Integer value)
    throws ResourceException {
    this(name, def);

    if (value == null) {
      throw new java.lang.NullPointerException();
    }
    if (!def.getType().equals(PropertyType.INTEGER)) {
      throw new ResourceException("PropertyType in PropertyDef is not INTEGER");
    }
    m_value = value.toString();
    m_value_object = value;
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Long value)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.LONG)) {
      throw new ResourceException("PropertyType in PropertyDef is not LONG");
    }
    m_value = value.toString();
    m_value_object = value;
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, XMLMarkup value)
    throws ResourceException {
    this(name, def);

    if (value == null) {
      throw new java.lang.NullPointerException();
    }
    if (!def.getType().equals(PropertyType.XML)) {
      throw new ResourceException("PropertyType in PropertyDef is not XML");
    }
    m_value = value.toString();
    m_value_object = value;
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup and must match the
   * PropertyDef's type.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The value object
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed or does not matach the type
   */
  public Property(IPropertyName name, IPropertyDef def, Object value)
    throws ResourceException {
    this(name, def);

    if (def.getType().equals(PropertyType.STRING) && !(value instanceof String)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.INTEGER) && !(value instanceof Integer)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.LONG) && !(value instanceof Long)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.BOOLEAN) && !(value instanceof Boolean)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.DATE) && !(value instanceof Date)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.XML) && !(value instanceof XMLMarkup)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    m_value = value.toString();
    m_value_object = value;
    if (def.getType().equals(PropertyType.DATE)) {
      m_dateValue = (Date)value;
    }
  }

  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param def The property definition
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public Property(IPropertyName name, IPropertyDef def, List values)
    throws ResourceException {
    this(name, def);

    if (!def.isMultivalued()) {
      throw new ResourceException("PropertyDef is not multi-valued");
    }

    if (values != null) {
      int count = values.size() - 1;
      while (count >= 0) {
         this.validateType(values.get(count));
         count--;
      }
      m_values = Collections.unmodifiableList(new ArrayList(values));
    }
  }


  // ---- constructors with attributes ----

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param value The property string value
   * @param attributes
   */
  public Property(IPropertyName name, String value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.STRING, false);
    if (value == null) {
      throw new java.lang.NullPointerException();
    }

    m_value = value;
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param value The property date value
   * @param attributes
   */
  public Property(IPropertyName name, Date value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.DATE, false);
    if (value == null) {
      throw new java.lang.NullPointerException();
    }

    m_dateValue = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param value The property booelan value
   * @param attributes
   */
  public Property(IPropertyName name, Boolean value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.BOOLEAN, false);
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param value The property integer value
   */
  public Property(IPropertyName name, Integer value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.INTEGER, false);
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param value The property long value
   */
  public Property(IPropertyName name, Long value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.LONG, false);
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param value The property xml value
   */
  public Property(IPropertyName name, XMLMarkup value, Properties attributes)
    throws ResourceException {
    this(name, PropertyType.XML, false);
    if (value == null) {
      throw new java.lang.NullPointerException();
    }
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup.
   *
   * @param name The property name
   * @param value The value object
   * @param attributes
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed
   * @todo: Description of the incoming method parameter
   */
  public Property(IPropertyName name, Object value, Properties attributes)
    throws ResourceException {
    if (name == null) {
      throw new NullPointerException();
    }
    m_name = name;

    if (value instanceof String) {
      m_type = PropertyType.STRING;
    }
    else if (value instanceof Integer) {
      m_type = PropertyType.INTEGER;
    }
    else if (value instanceof Long) {
      m_type = PropertyType.LONG;
    }
    else if (value instanceof Boolean) {
      m_type = PropertyType.BOOLEAN;
    }
    else if (value instanceof Date) {
      m_type = PropertyType.DATE;
      m_dateValue = (Date)value;
    }
    else if (value instanceof XMLMarkup) {
      m_type = PropertyType.XML;
    }
    else {
      throw new ResourceException("Illegal property value instance: " + value.getClass().getName());
    }
    m_value = value.toString();
    m_value_object = value;
    m_isMultivalued = false;
    m_def = PropertyDef.valueOf(m_type, m_isMultivalued);
    m_attributes = attributes;
  }


  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param type The property data type
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @param attributes
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   * @todo: Description of the incoming method parameter
   */
  public Property(IPropertyName name, PropertyType type, List values, Properties attributes)
    throws ResourceException {
    this(name, type, true);

    m_attributes = attributes;

    if (values != null) {
      int count = values.size() - 1;
      while (count >= 0) {
        this.validateType(values.get(count));
        count--;
      }
      m_values = Collections.unmodifiableList(new ArrayList(values));
    }
  }

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, String value, Properties attributes)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.STRING)) {
      throw new ResourceException("PropertyType in PropertyDef is not STRING");
    }
    m_value = value;
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Date value, Properties attributes)
    throws ResourceException {
    this(name, def);
    if (value == null) {
      throw new java.lang.NullPointerException();
    }
    if (!def.getType().equals(PropertyType.DATE)) {
      throw new ResourceException("PropertyType in PropertyDef is not DATE");
    }
    m_dateValue = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Boolean value, Properties attributes)
    throws ResourceException {
    this(name, def);
    if (!def.getType().equals(PropertyType.BOOLEAN)) {
      throw new ResourceException("PropertyType in PropertyDef is not BOOLEAN");
    }
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Integer value, Properties attributes)
    throws ResourceException {
    this(name, def);
    if (value == null) {
      throw new java.lang.NullPointerException();
    }
    if (!def.getType().equals(PropertyType.INTEGER)) {
      throw new ResourceException("PropertyType in PropertyDef is not INTEGER");
    }
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, Long value, Properties attributes)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.LONG)) {
      throw new ResourceException("PropertyType in PropertyDef is not LONG");
    }
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   */
  public Property(IPropertyName name, IPropertyDef def, XMLMarkup value, Properties attributes)
    throws ResourceException {
    this(name, def);

    if (!def.getType().equals(PropertyType.XML)) {
      throw new ResourceException("PropertyType in PropertyDef is not XML");
    }
    m_value = value.toString();
    m_value_object = value;
    m_attributes = attributes;
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup and must match the
   * PropertyDef's type.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The value object
   * @param attributes
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed or does not matach the type
   * @todo: Description of the incoming method parameter
   */
  public Property(IPropertyName name, IPropertyDef def, Object value, Properties attributes)
    throws ResourceException {
    this(name, def);

    if (def.getType().equals(PropertyType.STRING) && !(value instanceof String)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.INTEGER) && !(value instanceof Integer)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.LONG) && !(value instanceof Long)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.BOOLEAN) && !(value instanceof Boolean)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.DATE) && !(value instanceof Date)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    else if (def.getType().equals(PropertyType.XML) && !(value instanceof XMLMarkup)) {
      throw new ResourceException("Property value instance does not match type: " + value.getClass().getName());
    }
    m_value = value.toString();
    m_value_object = value;
    if (def.getType().equals(PropertyType.DATE)) {
      m_dateValue = (Date)value;
    }
    m_attributes = attributes;
  }


  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param def The property definition
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @param attributes
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   * @todo: Description of the incoming method parameter
   */
  public Property(IPropertyName name, IPropertyDef def, List values, Properties attributes)
    throws ResourceException {
    this(name, def);

    if (!def.isMultivalued()) {
      throw new ResourceException("PropertyDef is not multi-valued");
    }

    m_attributes = attributes;

    if (values != null) {
      int count = values.size() - 1;       
      while (count >= 0) {
        this.validateType(values.get(count));
        count--;
      }
      m_values = Collections.unmodifiableList(new ArrayList(values));
    }
  }

  /**
   * Construct a property as a copy
   *
   * @param p
   * @todo: Description of the incoming method parameter
   */
  protected Property(Property p) {
    m_name = p.m_name;
    m_def = p.m_def;
    m_type = p.m_type;
    m_isMultivalued = p.m_isMultivalued;
    m_value = p.m_value;
    m_value_object = p.m_value_object;
    m_values = p.m_values;
    m_dateValue = p.m_dateValue;
    m_attributes = p.m_attributes;
  }

  protected Property(IPropertyName name, IPropertyDef def)
    throws ResourceException {
    if (name == null) {
      throw new NullPointerException();
    }
    m_name = name;

    if (def == null) {
      throw new java.lang.NullPointerException();
    }
    m_def = def;
    m_type = m_def.getType();
    m_isMultivalued = def.isMultivalued();
  }

  protected Property(IPropertyName name, PropertyType type, boolean isMultivalued)
    throws ResourceException {
    if (name == null) {
      throw new java.lang.NullPointerException();
    }

    m_name = name;
    m_type = type;
    m_isMultivalued = isMultivalued;
    m_def = PropertyDef.valueOf(m_type, m_isMultivalued);
  }

  // ---------------------------------------------------------------------------
  // IProperty interface
  // ---------------------------------------------------------------------------


  public IPropertyName getPropertyName() {
    return m_name;
  }

  public PropertyType getType() {
    if (m_type == null) {
      return m_def.getType();
    }
    else {
      return m_type;
    }
  }

  public boolean isMultivalued() {
    return m_isMultivalued;
  }


  public IPropertyDef getPropertyDef() {
    return m_def;
  }


  public String getValueAsString() {
    if (m_dateValue != null) {
      return DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG).format(m_dateValue);
    }
    if (m_value != null) {
      return m_value;
    }
    else {
      return null;
    }
  }


  public boolean getBooleanValue() {
    return (new Boolean(m_value)).booleanValue();
  }


  public int getIntValue() {
    return (new Integer(m_value)).intValue();
  }


  public long getLongIntValue() {
    return (new Long(m_value)).longValue();
  }


  public Date getDateValue() {
    if (m_dateValue != null) {
      return (Date)(m_dateValue.clone());
    }
    try {
      return DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG).parse(m_value);
    }
    catch (Exception e) {
      return null;
    }
  }


  public String getStringValue() {
    return getValueAsString();
  }


  public List getValues() throws ResourceException {
    if (!m_isMultivalued) {
      return null;
    }
    if (m_values == null) {
      return new ArrayList();
    }
    else {
      return new ArrayList(m_values);
    }
  }

  public Object getValue() throws ResourceException {
    if (m_isMultivalued) {
      return null;
    }
    else {
      return m_value_object;
    }
  }

  public IMutableProperty getMutable() {
    return new MutableProperty(this);
  }


  public String getAttribute(String name)
    throws ResourceException {
    if (m_attributes == null) {
      return null;
    }
    return m_attributes.getProperty(name);
  }

  public Properties getAttributes()
    throws ResourceException {
    if (m_attributes == null) {
      return null;
    }
    return (Properties)m_attributes.clone();
  }

  public String getDescription()
    throws WcmException {
    String description = null;

    try {
      if (res != null && this.m_res.exists(this.m_name.getName())) {
        description = m_res.getString(this.m_name.getName());
      }
      else {
        description = this.m_name.getName();
      }
    }
    catch (MissingResourceException ex) {
      throw new WcmException(ex);
    }

    return description;
  }


  public String getDescription(IResourceContext context)
    throws WcmException {
    String description = null;

    try {
      if (res != null && this.m_res.exists(this.m_name.getName())) {
        description = m_res.getString(this.m_name.getName(), context.getLocale());
      }
      else {
        description = this.m_name.getName();
      }
    }
    catch (MissingResourceException ex) {
      throw new WcmException(ex);
    }

    return description;
  }


  public String toString() {
    if (m_values != null) {
      return m_values.toString();
    }
    else {
      return getValueAsString();
    }
  }


  public boolean equals(Object prop) {
    if (prop == null) {
      return false;
    }
    if (!(prop instanceof Property)) {
      return false;
    }
    return m_name.equals(((Property)prop).getPropertyName());
  }


  // ---------------------------------------------------------------------------
  // private / protected
  // ---------------------------------------------------------------------------

  /**
   * Validates the instance of the property value
   *
   * @param v The value to set
   */
  protected void validateType(Object v)
    throws ResourceException {
    PropertyType type = getType();
    if (type.equals(PropertyType.BOOLEAN)) {
      if (!(v instanceof Boolean)) {
        throw new ResourceException("Wrong instance, expected Boolean");
      }
    }
    else if (type.equals(PropertyType.DATE)) {
      if (!(v instanceof Date)) {
        throw new ResourceException("Wrong instance, expected Date");
      }
    }
    else if (type.equals(PropertyType.INTEGER)) {
      if (!(v instanceof Integer)) {
        throw new ResourceException("Wrong instance, expected Integer");
      }
    }
    else if (type.equals(PropertyType.LONG)) {
      if (!(v instanceof Long)) {
        throw new ResourceException("Wrong instance, expected Long");
      }
    }
    else if (type.equals(PropertyType.STRING)) {
      if (!(v instanceof String)) {
        throw new ResourceException("Wrong instance, expected String");
      }
    }
    else if (type.equals(PropertyType.XML)) {
      if (!(v instanceof XMLMarkup)) {
        throw new ResourceException("Wrong instance, expected String");
      }
    }
  }


  /**
   * Returns true if it is the name of a system property
   *
   * @param propName
   * @return Boolean value
   */
  public static synchronized boolean isSystemProperty(IPropertyName propName)
    throws ResourceException {
    if (g_systemNames == null) {
      g_systemNames = new IPropertyName[FIXED_COUNT];
      g_systemNames[0] = PropertyName.createHidden();
      g_systemNames[1] = PropertyName.createReadOnly();
      g_systemNames[2] = PropertyName.createCreationDate();
      g_systemNames[3] = PropertyName.createLastModified();
      g_systemNames[4] = PropertyName.createDescription();
      g_systemNames[5] = PropertyName.createDisplayname();
      g_systemNames[6] = PropertyName.createCreatedBy();
      g_systemNames[7] = PropertyName.createLastModifiedBy();
      g_systemNames[8] = PropertyName.createContentType();
      g_systemNames[9] = PropertyName.createContentLength();
      g_systemNames[10] = PropertyName.createETag();
      g_systemNames[11] = PropertyName.createVersioned();
      g_systemNames[12] = PropertyName.createEnhancedCollection();
      g_systemNames[13] = PropertyName.createContentLanguage();
      g_systemNames[14] = PropertyName.createResourceType();
      g_systemNames[15] = PropertyName.createLinkType();
      g_systemNames[16] = PropertyName.createCollection();      
    }

    boolean isSystem = false;
    for (int l = 0; l < FIXED_COUNT; l++) {
      if (g_systemNames[l].equals(propName)) {
        isSystem = true;
        break;
      }
    }
    return isSystem;
  }


  public static IProperty createCreatedByProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createCreatedBy(), value);
  }

  public static IProperty createLastModifiedByProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createLastModifiedBy(), value);
  }

  public static IProperty createCreationDateProp(Date date)
    throws ResourceException {
    return new Property(PropertyName.createCreationDate(), date);
  }

  public static IProperty createHiddenProp(boolean value)
    throws ResourceException {
    return new Property(PropertyName.createHidden(), new Boolean(value));
  }

  public static IProperty createLastModifiedProp(Date date)
    throws ResourceException {
    return new Property(PropertyName.createLastModified(), date);
  }

  public static IProperty createDescriptionProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createDescription(), value);
  }

  public static IProperty createDisplaynameProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createDisplayname(), value);
  }

  public static IProperty createReadOnlyProp(boolean value)
    throws ResourceException {
    return new Property(PropertyName.createReadOnly(), new Boolean(value));
  }

  public static IProperty createContentLengthProp(long length)
    throws ResourceException {
    return new Property(PropertyName.createContentLength(), new Long(length));
  }

  public static IProperty createContentTypeProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createContentType(), value);
  }

  public static IProperty createETagProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createETag(), value);
  }

  public static IProperty createContentLanguageProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createContentLanguage(), value);
  }

  public static IProperty createResourceTypeProp(String value)
    throws ResourceException {
    return new Property(PropertyName.createResourceType(), value);
  }
}
