/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.io.*;
import java.util.*;

/**
 * This object contains information about the restrictions for resource and
 * collection names of a specific repository. This information should be
 * provided by any repository via the <code>IRepositoryManager.getNameInfo()
 * </code> method. <p>
 *
 * The implementation of this object ensures that the WCM framework's
 * restrictions are added. The WCM framework enforced the following restrictions
 * on the namespace of all repositories for new resources and collections.
 * Reserved/forbidden characters in resource and collection names: '/', '\', '?'
 * Maximum length of name and path: 2048 characters <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author Markus Breitenfelder
 * @version $Id: NameInfo.java,v 1.9 2002/09/04 12:10:53 sei Exp $
 */
public class NameInfo {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.NameInfo.class);

  final static String FORBIDDEN_CHARS_FILE = "forbidden_chars.txt";

  static List forbiddenChars = null;

  private final char[] resCharsArray;
  private final char[] collCharsArray;

  private final int maxNameLength;
  private final int maxPathLength;

  /**
   * Construct a new name info
   *
   * @param resNameChars An array of reserved or forbidden characters in
   *      resource names
   * @param collNameChars An array of reserved or forbidden characters in
   *      collection names
   * @param maxNameLength The maximum length for resource names.
   * @param maxPathLength The maximum path length. The max. path length will not be checked by the framework.
   */
  public NameInfo(char[] resNameChars, char[] collNameChars, int maxNameLength, int maxPathLength) {
    this.initStatic();

    this.maxPathLength = maxPathLength;
    this.maxNameLength = maxNameLength;

    Set resChars = new HashSet();
    resChars.addAll(NameInfo.forbiddenChars);
    for (int i = 0; i < resNameChars.length; i++) {
      resChars.add(new Character(resNameChars[i]));
    }

    Set collChars = new HashSet();
    collChars.addAll(NameInfo.forbiddenChars);
    for (int i = 0; i < resNameChars.length; i++) {
      collChars.add(new Character(collNameChars[i]));
    }

    this.resCharsArray = new char[resChars.size()];
    int pos = 0;
    for (Iterator i = resChars.iterator(); i.hasNext(); ) {
      this.resCharsArray[pos++] = ((Character)i.next()).charValue();
    }

    this.collCharsArray = new char[collChars.size()];
    pos = 0;
    for (Iterator i = collChars.iterator(); i.hasNext(); ) {
      this.collCharsArray[pos++] = ((Character)i.next()).charValue();
    }
  }

  // -- BEGIN HACK ------------------------------------------------------------------------

  // Hack for SP4 Patch 1: Read forbidden chars from file - encoding must be UTF-8

  // 7/2004: Seems that this freaky pice of code will stay forever !

  private void initStatic() {
    if (NameInfo.forbiddenChars != null) {
      return;
    }

    NameInfo.forbiddenChars = new ArrayList();
    NameInfo.forbiddenChars.add(new Character('/'));
    NameInfo.forbiddenChars.add(new Character('\\'));
    NameInfo.forbiddenChars.add(new Character('?'));

    String path = new File(this.getClassPath()).getParent();
    NameInfo.log.debugT("initStatic(123)", "path=" + path);
    File file = new File(path, FORBIDDEN_CHARS_FILE);
    if (file == null || !file.exists()) {
      NameInfo.log.debugT("initStatic(126)", "Unable to locate the file: " + file.getPath());
    }
    else {
      try {
        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-16"));
        String crlf = System.getProperty("line.separator");
        String line = null;
        while ((line = reader.readLine()) != null) {
          line = line.trim();
          if (line.length() > 0 && crlf.indexOf(line) == -1) {
            NameInfo.forbiddenChars.add(new Character(line.charAt(0)));
            NameInfo.log.debugT("initStatic(137)", "Added forbidden char: " + line.charAt(0));
          }
        }
        reader.close();
      }
      catch (IOException ex) {
        //$JL-EXC$         
        NameInfo.log.errorT("initStatic(143)", "Error parsing file " + FORBIDDEN_CHARS_FILE + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
  }

  // Taken from ConfigurationManager
  private String getClassPath() {
    final String JAR_FILE_PREFIX = "jar:file:";
    final String FILE_PREFIX = "file:";

    try {
      String className = this.getClass().getName().replace('.', '/') + ".class";
      String url = CrtClassLoaderRegistry.getClassLoader().getResource(className).toString();

      if (url.startsWith(JAR_FILE_PREFIX)) {
        return (new File(url.substring(JAR_FILE_PREFIX.length(),
          url.length() - className.length() - 2))).getParent().toString();
      }
      // ???
      if (url.startsWith(FILE_PREFIX)) {
        return (new File(url.substring(FILE_PREFIX.length(),
          url.length() - className.length()))).toString();
      }

      return null;
    }
    catch (Exception ex) {
      //$JL-EXC$ 
      NameInfo.log.debugT("getClassPath(170)", LoggingFormatter.extractCallstack(ex));
      return null;
    }
  }

  // -- END HACK ------------------------------------------------------------------------


  public String toString() {
    StringBuffer sb = new StringBuffer(256);
    sb.append(this.maxNameLength).append(", ").append(this.maxPathLength).append(", [");
    for (int i = 0; i < this.resCharsArray.length; i++) {
      sb.append(this.resCharsArray[i]);
    }
    sb.append("], [");
    for (int i = 0; i < this.collCharsArray.length; i++) {
      sb.append(this.collCharsArray[i]);
    }
    sb.append("]");
    return sb.toString();
  }


  /**
   * Returns the array of reserved or forbidden characters in resource names
   *
   * @return reservedResourceNameChars
   */
  public char[] getReservedResourceNameChars() {
    return this.resCharsArray;
  }

  /**
   * Returns the array of reserved or forbidden characters in collection names
   *
   * @return reservedCollectionNameChars
   */
  public char[] getReservedCollectionNameChars() {
    return this.collCharsArray;
  }

  /**
   * Returns the maximum length for resource names
   *
   * @return maxNameLength
   */
  public int getMaxNameLength() {
    return this.maxNameLength;
  }

  /**
   * Returns the maximum path length
   *
   * @return maxPathLength
   */
  public int getMaxPathLength() {
    return this.maxPathLength;
  }

  public void checkName(String name)
    throws InvalidNameException {
    if (name.length() == 0) {
      throw new InvalidNameException("Invalid name (empty)", this);
    }
    // no leading or trailing spaces
    if (name.charAt(0) == ' ' || name.charAt(name.length() - 1) == ' ') {
      throw new InvalidNameException("Invalid name: Trailing or leading spaces are not allowed.", this);
    }
    // forbidden names: ".", ".."
    if (name.equals(".")) {
      throw new InvalidNameException("Invalid name: \".\"", this);
    }
    if (name.equals("..")) {
      throw new InvalidNameException("Invalid name: \"..\"", this);
    }

    if (this.maxNameLength > 0 && name.length() > this.maxNameLength) {
      throw new InvalidNameException("name too long", this);
    }

    for (int i = 0, n = this.resCharsArray.length; i < n; ++i) {
      if (name.indexOf(this.resCharsArray[i]) >= 0) {
        throw new InvalidNameException("Invalid character in resource name: " + this.resCharsArray[i], this);
      }
    }
  }
}
