/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.*;
import com.sapportals.wcm.repository.enum.*;
import java.text.DateFormat;
import java.util.*;

/**
 * A property of a resource object that can be modified <p>
 *
 *
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Revision: 1.19 $
 */
public class MutableProperty extends Property implements IMutableProperty, java.io.Serializable {

  /**
   * Construct a new mutable property from a read-only property. This is used in
   * Property.getMutable().
   *
   * @param p TBD: Description of the incoming method parameter
   */
  protected MutableProperty(Property p) {
    super(p);
  }

  // ----- new constructors ------


  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, String value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param value The property date value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Date value)
    throws ResourceException {
    super(name, value);
  }


  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param value The property booelan value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Boolean value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param value The property integer value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Integer value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param value The property long value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Long value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param value The property xml value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, XMLMarkup value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param type The property data type
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public MutableProperty(IPropertyName name, PropertyType type, List values)
    throws ResourceException {
    super(name, type, values);
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup.
   *
   * @param name The property name
   * @param value The value object
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed
   */
  public MutableProperty(IPropertyName name, Object value)
    throws ResourceException {
    super(name, value);
  }

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, String value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Date value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Boolean value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Integer value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Long value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, XMLMarkup value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup and must match the
   * PropertyDef's type.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The value object
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed or does not matach the type
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Object value)
    throws ResourceException {
    super(name, def, value);
  }

  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param def The property definition
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, List values)
    throws ResourceException {
    super(name, def, values);
  }


  // ---- constructors with attributes ----

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, String value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param value The property date value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Date value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param value The property booelan value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Boolean value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param value The property integer value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Integer value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param value The property long value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, Long value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param value The property xml value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, XMLMarkup value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup.
   *
   * @param name The property name
   * @param value The value object
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed
   */
  public MutableProperty(IPropertyName name, Object value, Properties attributes)
    throws ResourceException {
    super(name, value, attributes);
  }


  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param type The property data type
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public MutableProperty(IPropertyName name, PropertyType type, List values, Properties attributes)
    throws ResourceException {
    super(name, type, values, attributes);
  }

  /**
   * Construct a new property of type STRING.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, String value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property of type DATE.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Date value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property of type BOOLEAN.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Boolean value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property of type INTEGER.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Integer value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property of type LONG.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Long value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property of type XML.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The property string value
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, XMLMarkup value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }

  /**
   * Construct a new property. The value object's instance must be one of
   * String, Integer, Long, Boolean, Date or XMLMarkup and must match the
   * PropertyDef's type.
   *
   * @param name The property name
   * @param def The property definition
   * @param value The value object
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException If the instance of the value parameter is not
   *      allowed or does not matach the type
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, Object value, Properties attributes)
    throws ResourceException {
    super(name, def, value, attributes);
  }


  /**
   * Construct a new multi-valued property.
   *
   * @param name The property name
   * @param def The property definition
   * @param values A list of values. The list must contain suitable object
   *      instances (String, Date, Integer, Long, Boolean, XMLMarkup) for the
   *      specified property type.
   * @param attributes TBD: Description of the incoming method parameter
   * @exception ResourceException If the instance of the value does not match
   *      the property type
   */
  public MutableProperty(IPropertyName name, IPropertyDef def, List values, Properties attributes)
    throws ResourceException {
    super(name, def, values, attributes);
  }


  // ---------------------------------------------------------------------------
  // IMutableProperty interface
  // ---------------------------------------------------------------------------


  public void setIntValue(int value)
    throws ResourceException {
    if (!getType().equals(PropertyType.INTEGER)) {
      throw new ResourceException("Property type is not INTEGER");
    }
    m_value = String.valueOf(value);
  }

  public void setDateValue(Date value)
    throws ResourceException {
    if (!getType().equals(PropertyType.DATE)) {
      throw new ResourceException("Property type is not DATE");
    }
    m_dateValue = new Date(value.getTime());
  }

  public void setDateValueFromLong(long value)
    throws ResourceException {
    if (!getType().equals(PropertyType.DATE)) {
      throw new ResourceException("Property type is not DATE");
    }
    m_dateValue = new Date(value);
  }

  public void setStringValue(String value)
    throws ResourceException {
    if (!getType().equals(PropertyType.STRING)) {
      throw new ResourceException("Property type is not STRING");
    }
    m_value = value;
  }

  public void setLongIntValue(long value)
    throws ResourceException {
    if (!getType().equals(PropertyType.LONG)) {
      throw new ResourceException("Property type is not LONG");
    }
    m_value = String.valueOf(value);
  }

  public void setBooleanValue(boolean value)
    throws ResourceException {
    if (!getType().equals(PropertyType.BOOLEAN)) {
      throw new ResourceException("Property type is not BOOLEAN");
    }
    m_value = new Boolean(value).toString();
  }

  public void setValues(List values)
    throws ResourceException {
    if (!isMultivalued()) {
      throw new ResourceException("Property type is not multi-valued");
    }
    Iterator it = values.iterator();
    while (it.hasNext()) {
      validateType(it.next());
    }
    m_values = new ArrayList(values);
  }

  public void addValue(Object value)
    throws ResourceException {
    if (!isMultivalued()) {
      throw new ResourceException("Property type is not multi-valued");
    }
    validateType(value);
    if (m_values == null) {
      m_values = new ArrayList();
    }
    m_values.add(value);
  }

  public void setValue(int index, Object value)
    throws ResourceException {
    if (!isMultivalued()) {
      throw new ResourceException("Property type is not multi-valued");
    }
    validateType(value);
    try {
      m_values.set(index, value);
    }
    catch (IndexOutOfBoundsException ex) {
      //$JL-EXC$ 
      throw new ResourceException("Index out of bounds");
    }
    catch (Exception ex) {
      throw new ResourceException(ex);
    }
  }

  public void removeValue(int index)
    throws ResourceException {
    if (!isMultivalued()) {
      throw new ResourceException("Property type is not multi-valued");
    }
    try {
      m_values.remove(index);
    }
    catch (IndexOutOfBoundsException ex) {
      //$JL-EXC$ 
      throw new ResourceException("Index out of bounds");
    }
    catch (Exception ex) {
      //$JL-EXC$ 
      throw new ResourceException(ex);
    }
  }


  public IMutableProperty getMutable() {
    return this;
  }

  public void setAttribute(String name, String value)
    throws ResourceException {
    if (m_attributes == null) {
      m_attributes = new Properties();
    }
    m_attributes.setProperty(name, value);
  }

  public void setAttributes(Properties attributes)
    throws ResourceException {
    m_attributes = attributes;
  }

  public void removeAttribute(String name)
    throws ResourceException {
    if (m_attributes == null) {
      return;
    }
    m_attributes.remove(name);
  }
}
