/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.enum.*;

import java.util.Date;

/**
 * This object represents an active lock on a resource. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/repository/framework/api/com/sapportals/wcm/repository/LockInfo.java#8
 *      $
 */
public class LockInfo implements ILockInfo {

  public final static int LOCK_TIMEOUT_INFINITE = -1;

  protected LockType type = LockType.WRITE;

  protected LockScope scope = LockScope.EXCLUSIVE;

  protected LockDepth depth = LockDepth.SHALLOW;

  protected String owner = null;

  protected XMLMarkup clientSuppliedOwner = null;

  protected String lockToken = null;

  /**
   * Timeout in seconds
   */
  protected int timeout = LOCK_TIMEOUT_INFINITE;

  protected Date expiration = null;

  protected Date created = null;

  /**
   * @param type lock type
   * @param scope lock scope
   * @param depth lock depth
   * @param owner lock owner
   * @param timeout timeout in seconds
   * @param lockToken The lock token must be either an absolute URI (as defined
   *      in <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> or a
   *      relative URI reference (in which case the <a
   *      href="http://greenbytes.de/tech/webdav/rfc2518.html#rfc.section.6.4">
   *      opaquelocktoken</a> URI scheme is assumed).
   * @param created creation time
   * @param ownerInfoFromClient A client can submit his own owner information
   *      that SHOULD be returned upon lock discovery. If present, the format
   *      MUST be the XML serialization of a <a
   *      href="http://www.greenbytes.de/tech/webdav/rfc2518.html#ELEMENT_owner"
   *      >DAV:owner</a> element.
   */
  public LockInfo(LockType type, LockScope scope, LockDepth depth, String owner, int timeout,
    String lockToken, Date created, XMLMarkup ownerInfoFromClient) {
    constructLockInfo(type, scope, depth, owner, timeout, lockToken, created,
      ownerInfoFromClient);
  }

  public LockInfo(LockType type, LockScope scope, LockDepth depth, String owner, int timeout,
    String lockToken, Date created) {
    constructLockInfo(type, scope, depth, owner, timeout, lockToken, created, null);
  }

  public LockInfo(LockType type, LockScope scope, LockDepth depth, String owner, int timeout,
    String lockToken) {
    constructLockInfo(type, scope, depth, owner, timeout, lockToken, new Date(), null);
  }


  //----------------------------------------------------------------------------
  // Interface ILockInfo
  //----------------------------------------------------------------------------


  public LockType getType() {
    return this.type;
  }


  public LockScope getScope() {
    return this.scope;
  }


  public LockDepth getDepth() {
    return this.depth;
  }


  public String getOwner() {
    return this.owner;
  }


  public int getTimeout() {
    return this.timeout;
  }

  public XMLMarkup getClientSuppliedOwner() {
    return this.clientSuppliedOwner;
  }

  /**
   * The date and time when the lock will expire. Returns null if the lock never
   * times out (infinite).
   *
   * @return expiration
   */
  public Date getExpiration() {
    return this.expiration;
  }

  /**
   * The time remaining until the lock expires. Returns -1 if the lock will
   * never expire.
   *
   * @return TBD: Description of the outgoing return value
   */
  public long timeRemaining() {
    if (this.expiration != null) {
      long now = new Date().getTime();
      long t = this.expiration.getTime() - now;
      if (t < 0) {
        t = 0;
      }
      return t;
    }
    else {
      return -1;// infinite
    }
  }

  public String getLockToken() {
    return this.lockToken;
  }

  //----------------------------------------------------------------------------
  // public
  //----------------------------------------------------------------------------

  public Date getCreated() {
    return this.created;
  }

  /**
   * Compare this LockInfo with another one or with a String (token)
   *
   * @param o TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (o instanceof String) {
      return ((String)o).equals(this.lockToken);
    }
    else if (o instanceof LockInfo) {
      return ((LockInfo)o).getLockToken().equals(this.lockToken);
    }
    else {
      return false;
    }
  }

  public int hashCode() {
    return this.lockToken.hashCode();
  }

  //----------------------------------------------------------------------------
  // private
  //----------------------------------------------------------------------------

  private void constructLockInfo(LockType type, LockScope scope, LockDepth depth, String owner,
    int timeout, String lockToken, Date created, XMLMarkup clientSuppliedOwner) {

    this.created = created;
    this.type = type;
    this.scope = scope;
    this.depth = depth;
    if (owner != null) {
      this.owner = owner;
    }
    else {
      this.owner = "";
    }
    if (timeout <= 0 && timeout != LOCK_TIMEOUT_INFINITE) {
      timeout = LOCK_TIMEOUT_INFINITE;
    }
    this.timeout = timeout;
    if (timeout == LOCK_TIMEOUT_INFINITE) {
      this.expiration = null;
    }
    else {
      this.expiration = new Date(this.created.getTime() + timeout * 1000);
    }
    if (lockToken != null) {
      this.lockToken = lockToken;
    }
    else {
      this.lockToken = "";
    }

    this.clientSuppliedOwner = clientSuppliedOwner;
  }
}

