/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.Set;

import com.sapportals.wcm.util.name.IName;
import com.sapportals.wcm.util.name.Name;
import com.sapportals.wcm.util.uri.IRidSet;
import com.sapportals.wcm.util.uri.RID;

/**
 * Interface for version (revision) resources (see <a
 * href="http://greenbytes.de/tech/webdav/rfc3253.html">RFC3253</a> ). <p>
 *
 * Copyright (c) SAP AG 2002-2004
 * @author julian.reschke@greenbytes.de
 * @version $Id: IVersionResource.java,v 1.3 2004/03/23 13:20:46 jre Exp $
 */
public interface IVersionResource {

  /**
   * Forks are discouraged, and a checkin/checkout request <em>must</em>
   * specifically allow them.
   */
  public final static IName FORKBEHAVIOUR_DISCOURAGED = new Name("DAV:", "discouraged");

  /**
   * Forks are forbidden, and a checkin/checkout requests creating a fork will
   * be rejected.
   */
  public final static IName FORKBEHAVIOUR_FORBIDDEN = new Name("DAV:", "forbidden");

  /**
   * Checkout the version.
   *
   * @param forkOk set to <em>true</em> if it's ok to cause a fork in the
   *      version history (may be rejected nevertheless if the checkout fork
   *      behaviour dictates this).
   * @return information about chechkout result
   * @exception ResourceException Exception raised in failure situation
   */
  public ICheckOutInfo2 checkOut(boolean forkOk)
    throws ResourceException;

  /**
   * @return set of current checkouts of this version
   * @throws ResourceException
   */
  public IResourceList getCheckedOutResources()
    throws ResourceException;

  /**
   * @return a set of INames describing the checkout-in behaviour for this
   *      resource.
   * @exception ResourceException Exception raised in failure situation
   * @see #FORKBEHAVIOUR_DISCOURAGED
   * @see #FORKBEHAVIOUR_FORBIDDEN
   */
  public Set getCheckInForkBehaviour()
    throws ResourceException;

  /**
   * @return a set of INames describing the checkout-fork behaviour for this
   *      resource.
   * @exception ResourceException Exception raised in failure situation
   * @see #FORKBEHAVIOUR_DISCOURAGED
   * @see #FORKBEHAVIOUR_FORBIDDEN
   */
  public Set getCheckOutForkBehaviour()
    throws ResourceException;

  /**
   * @return set of current checkouts of this version
   * @throws ResourceException
   */
  public IRidSet getCheckedOutRIDs()
    throws ResourceException;

  /**
   * @return set of immediate predecessors if this version
   * @throws ResourceException
   */
  public IResourceList getPredecessorResources()
    throws ResourceException;

  /**
   * @return set of immediate predecessors if this version
   * @throws ResourceException
   */
  public IRidSet getPredecessorRIDs()
    throws ResourceException;

  /**
   * @return IResource for version history resource (or <code>null</code> if
   *      version history resources are not supported).
   * @throws ResourceException
   */
  public IResource getVersionHistoryResource()
    throws ResourceException;

  /**
   * @return RID of version history resource (or <code>null</code> if version
   *      history resources are not supported).
   * @throws ResourceException
   */
  public RID getVersionHistoryRID()
    throws ResourceException;

  /**
   * @return set of immediate successors of this version
   * @throws ResourceException
   */
  public IResourceList getSuccessorResources()
    throws ResourceException;

  /**
   * @return set of immediate successors of this version
   * @throws ResourceException
   */
  public IRidSet getSuccessorRIDs()
    throws ResourceException;

  // LABEL support

  /**
   * Adds the given label if and only if it didn't exist before in the version's
   * version history.
   *
   * @param labelName label
   * @exception ResourceException Exception raised in failure situation
   */
  public void addLabel(String labelName)
    throws ResourceException;

  /**
   * Sets the given label on the version (removing it from other versions if
   * present)
   *
   * @param labelName label
   * @exception ResourceException Exception raised in failure situation
   */
  public void setLabel(String labelName)
    throws ResourceException;

  /**
   * Removes the given label from the resource if present (removing non-existing
   * labels should be signaled through an exception)
   *
   * @param labelName label
   * @exception ResourceException Exception raised in failure situation
   */
  public void removeLabel(String labelName)
    throws ResourceException;

  /**
   * @return set of labels present on this version (empty list when no labels
   *      present).
   * @exception ResourceException Exception raised in failure situation
   */
  public Set getLabelSet()
    throws ResourceException;
}

