/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.Set;

import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.name.IName;
import com.sapportals.wcm.util.name.Name;
import com.sapportals.wcm.util.uri.IRidSet;
import com.sapportals.wcm.util.uri.RID;

/**
 * Interface for version controlled resources (see <a
 * href="http://greenbytes.de/tech/webdav/rfc3253.html">RFC3253</a>).
 * <p>
 * Coding example for obtaining the interface from an {@link IResource}:
 * <pre>
 * IResource resource = ...;
 * ICheckOutInfo2 coi = null;
 * IVersionControlledResource vcr = (IVersionControlledResource)resource.as(IVersionControlledResource.class);
 * if (vcr != null) {
 *   coi = vcr.checkOut(true, true);
 * }
 * </pre>
 * <p>
 * Copyright (c) SAP AG 2002-2004
 * @author julian.reschke@greenbytes.de
 */
public interface IVersionControlledResource {

  /**
   * See <a href="http://greenbytes.de/tech/webdav/rfc3253.html#PROPERTY_auto-version">
   * RFC3253, section 3.2.2</a>
   */
  public final static IName AUTOVERSIONING_MODE_CHECKOUT_CHECKIN = new Name("DAV:", "checkout-checkin");
  public final static IName AUTOVERSIONING_MODE_CHECKOUT_UNLOCKED_CHECKIN = new Name("DAV:", "checkout-unlocked-checkin");
  public final static IName AUTOVERSIONING_MODE_CHECKOUT = new Name("DAV:", "checkout");
  public final static IName AUTOVERSIONING_MODE_LOCKED_CHECKOUT = new Name("DAV:", "locked-checkout");
  public final static IName AUTOVERSIONING_MODE_NONE = null;

  /**
   * Check in a new version of the resource with new content and/or properties.
   *
   * @param newContent The new content, can be <code>null</code>
   * @param properties The new properties, can be <code>null</code>
   * @param ignorePropertyFailures Ignore exceptions if some or all properties
   *      could not be set/removed
   * @param expectedCheckInRID Create the new version with this RID. The
   *      check-in <em>may</em> fail if the new version RID would be different.
   *      In this case an ExpectedCheckInRID exception will be thrown which will
   *      contain a new "expected check-in RID". To make sure that the
   *      expectedCheckInRID actually <em>was</em> used, it needs to be compared
   *      to the RID returned by {@link ICheckInInfo#getRevisionRID()} (this is
   *      because a repository may check-in the resource although it couldn't
   *      use the expectedCheckInRID).
   * @param forkOk set to <em>true</em> if it's ok to cause a fork in the
   *      version history (may be rejected nevertheless if the checkin fork
   *      behaviour dictates this).
   * @param keepCheckedOut keep the resource checked out
   * @return checkin status information
   * @exception ResourceException Exception raised in failure situation
   * @see #checkOut(boolean, boolean)
   */
  public ICheckInInfo2 checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, RID expectedCheckInRID,
    boolean forkOk, boolean keepCheckedOut)
    throws ResourceException;

  /**
   * Checkout the version controlled resource or it's checked-in version.
   *
   * @param applyToVersion checkout the checked-in version of the version
   *      controlled resource (resource must be checked-in!).
   * @param forkOk set to <em>true</em> if it's ok to cause a fork in the
   *      version history (may be rejected nevertheless if the checkout fork
   *      behaviour dictates this).
   * @return information about checkout result
   * @exception ResourceException Exception raised in failure situation
   */
  public ICheckOutInfo2 checkOut(boolean applyToVersion, boolean forkOk)
    throws ResourceException;

  /**
   * @return IName of current autoversioning mode (or <code>null</code> when no
   *      autoversioning present)
   * @throws ResourceException
   */
  public IName getAutoVersioningMode()
    throws ResourceException;

  /**
   * @return IResource for currently checked-in version (or <code>null</code> )
   * @throws ResourceException
   */
  public IResource getCheckedInResource()
    throws ResourceException;

  /**
   * @return RID of currently checked-in version (or <code>null</code> )
   * @throws ResourceException
   */
  public RID getCheckedInRID()
    throws ResourceException;

  /**
   * @return IResource for currently checked-out version (or <code>null</code> )
   * @throws ResourceException
   */
  public IResource getCheckedOutResource()
    throws ResourceException;

  /**
   * @return RID of currently checked-out version (or <code>null</code> )
   * @throws ResourceException
   */
  public RID getCheckedOutRID()
    throws ResourceException;

  /**
   * @return a set of INames describing the checkout-in behaviour for this
   *      resource (when checked-out).
   * @exception ResourceException Exception raised in failure situation
   * @see IVersionResource#getCheckInForkBehaviour()
   */
  public Set getCheckInForkBehaviour()
    throws ResourceException;

  /**
   * @return a set of INames describing the checkout-fork behaviour for this
   *      resource (when checked-out).
   * @exception ResourceException Exception raised in failure situation
   * @see IVersionResource#getCheckOutForkBehaviour()
   */
  public Set getCheckOutForkBehaviour()
    throws ResourceException;

  /**
   * (version controlled collection only)
   *
   * @return list of the internal member names of all version-controlled
   *      resources that are currently eclipsed by a non-version-controlled
   *      internal member of the collection (see <a
   *      href="http://greenbytes.de/tech/webdav/rfc3253.html#PROPERTY_eclipsed-set">
   *      RFC3253, section 14.1.1</a> ).
   * @exception ResourceException Exception raised in failure situation
   */
  public Set getEclipsedSet()
    throws ResourceException;

  /**
   * @return for checked-out resources, the set of immediate predecessors of the
   *      version that would be created upon checkin (empty otherwise)
   * @throws ResourceException
   */
  public IResourceList getPredecessorResources()
    throws ResourceException;

  /**
   * @return for checked-out resources, the set of immediate predecessors of the
   *      version that would be created upon checkin (empty otherwise)
   * @throws ResourceException
   */
  public IRidSet getPredecessorRIDs()
    throws ResourceException;

  /**
   * @return IResource for version history resource (or <code>null</code> if
   *      version history resources are not supported).
   * @throws ResourceException
   */
  public IResource getVersionHistoryResource()
    throws ResourceException;

  /**
   * @return RID of version history resource (or <code>null</code> if version
   *      history resources are not supported).
   * @throws ResourceException
   */
  public RID getVersionHistoryRID()
    throws ResourceException;

  /**
   * update content and dead properties from a specific version.
   *
   * @param version from where to copy contents and properties
   * @return RIDs of all modified resources
   * @throws ResourceException
   */
  public IRidSet updateFromVersion(IResource version)
    throws ResourceException;

  /**
   * update content and dead properties from a specific version.
   *
   * @param version from where to copy contents and properties
   * @return RIDs of all modified resources
   * @throws ResourceException
   */
  public IRidSet updateFromVersion(RID version)
    throws ResourceException;
}

