/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.Comparator;

/**
 * A typed list of <code>IResource</code> instances. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Revision:$
 */
public interface IResourceList {

  /**
   * Returns the size of the list.
   *
   * @return the size of the list.
   */
  public int size();

  /**
   * Returns true if the specified resource is a member of the list.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return true if the specified resource is a member of the list.
   */
  public boolean containsResource(IResource resource);

  /**
   * Add the resource to the list.
   *
   * @param resource The resource to add to the list.
   * @return TBD: Description of the outgoing return value
   */
  public boolean add(IResource resource);

  /**
   * Add all the elements in the specified list to this list.
   *
   * @param list The list whose elements are to be added to this list.
   * @return TBD: Description of the outgoing return value
   */
  public boolean addAll(IResourceList list);

  /**
   * Removes the first occurence of the specified resource from the list.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return Returns true is this list contained the specified resource.
   */
  public boolean remove(IResource resource);

  /**
   * Returns the resource at the specified position.
   *
   * @param index of the resource to return.
   * @return the resource at the specified position.
   */
  public IResource get(int index);

  /**
   * Replaces the element at the specified position in this list with the
   * specified resource.
   *
   * @param index index of resource to replace.
   * @param resource resource to be stored at the specified position.
   * @return the element previously at the specified position.
   */
  public IResource set(int index, IResource resource);

  /**
   * Inserts the specified resource at the specified position in this list.
   * Shifts the resources currently at that position (if any) and any subsequent
   * resources to the right (adds one to their indices).
   *
   * @param index at which the specified resource is to be inserted.
   * @param resource element to be inserted.
   */
  public void add(int index, IResource resource);

  /**
   * Removes the resource at the specified position in this list. Shifts any
   * subsequent resources to the left (subtracts one from their indices).
   *
   * @param index the index of the resource to removed.
   * @return the element previously at the specified position.
   */
  public IResource remove(int index);

  /**
   * Returns the index in this list of the first occurrence of the specified
   * resource, or -1 if the List does not contain this resource.
   *
   * @param resource resource to search for.
   * @return the index in this list of the first occurrence of the specified
   *      resource, or -1 if the List does not contain this resource.
   */
  public int indexOf(IResource resource);

  /**
   * Returns the index in this list of the last occurrence of the specified
   * resource, or -1 if the List does not contain this resource.
   *
   * @param resource resource to search for.
   * @return the index in this list of the last occurrence of the specified
   *      resource, or -1 if the List does not contain this resource.
   */
  public int lastIndexOf(IResource resource);

  /**
   * Returns a resource-list-iterator of the elements in this list.
   *
   * @return a resource-list-iterator of the elements in this list.
   */
  public IResourceListIterator listIterator();

  /**
   * Returns a list-iterator of the elements in this list, starting at the
   * specified position in the list.
   *
   * @param index index of first element to be returned from the list-iterator
   *      (by a call to next).
   * @return a resource-list-iterator of the elements in this list.
   */
  public IResourceListIterator listIterator(int index);

  /**
   * @param index1 TBD: Description of the incoming method parameter
   * @param index2 TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @see java.util.AbstractList#subList(int, int)
   */
  public IResourceList subList(int index1, int index2);

  /**
   * Sort the elements in the list using the specified comparator
   *
   * @param comp The comparator used for sorting
   * @see java.util.Comparator
   */
  public void sort(Comparator comp);


  /**
   * Get the union of this list and the given list.<br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>a</i> , <i>b</i>
   * , <i>c</i> and <i>d</i> .
   *
   * @param list the <code>IResourceList</code> to union with this list.
   * @return the <code>IResourceList</code> with the union list.
   */
  public IResourceList union(IResourceList list);


  /**
   * Get the intersection of this list and the given list.<br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain <i>b</i> and <i>c
   * </i>.
   *
   * @param list the <code>IResourceList</code> to intersect with this list.
   * @return the <code>IResourceList</code> with the intersection list.
   */
  public IResourceList intersection(IResourceList list);


  /**
   * Get the difference of this list to the given list.<br>
   * Given <i>a</i> , <i>b</i> and <i>c</i> in this list; <i>b</i> , <i>c</i>
   * and <i>d</i> in the given list, the result will contain only <i>a</i> .
   *
   * @param list the <code>IResourceList</code> to subtract from this list.
   * @return the <code>IResourceList</code> with the difference list.
   */
  public IResourceList difference(IResourceList list);
}

