/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.Map;

import com.sapportals.wcm.repository.filter.IFilterFactory;
import com.sapportals.wcm.repository.manager.IManagerFactory;
import com.sapportals.wcm.repository.manager.IPermission;
import com.sapportals.wcm.repository.service.IRepositoryServiceFactory;
import com.sapportals.wcm.repository.batch.*;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.IUriList;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.uri.URI;

/**
 * The resource factory has methods to create resource objects ({@link
 * IResource}) and to access the manager and service factories.
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @see IManagerFactory
 */
public interface IResourceFactory {

  /**
   * @param uri TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @return resource
   * @exception ResourceException Exception raised in failure situation
   * @exception AuthorizationRequiredException Exception raised in failure
   *      situation
   * @deprecated As of EP 5.0 SP3, replaced by getResource() method with RID
   *      parameter.
   */
  public IResource getResource(URI uri, IResourceContext context)
    throws ResourceException, AuthorizationRequiredException;

  /**
   * @param uri TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @param createCollections TBD: Description of the incoming method parameter
   * @return resource
   * @exception ResourceException Exception raised in failure situation
   * @exception AuthorizationRequiredException Exception raised in failure
   *      situation
   * @deprecated As of EP 5.0 SP3, replaced by getResource() method with RID
   *      parameter.
   */
  public IResource getResource(URI uri, IResourceContext context, boolean createCollections)
    throws ResourceException, AuthorizationRequiredException;

  /**
   * Returns a resource object instance for the given resource identifier and
   * context. This method is a shortcut for <code>getResource(rid, true,
   * context)</code> . Returns <code>null</code> if the resource does not exist.
   *
   * @param rid The resource identifier of the resource.
   * @param context The resource context.
   * @return resource The resource instance. <code>null</code> is returned only
   *      if the resource definitively does not exist. If it can not be created
   *      by the repositoy for some other reason (e.g. internal error) an
   *      appropriate exception such as {@link ServiceUnavailableException} will
   *      be thrown
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getResource(RID rid, IResourceContext context)
    throws ResourceException;

  /**
   * Returns a resource object converted into a semantic object for the given
   * resource identifier and context.
   *
   * @param rid TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @param asClass TBD: Description of the incoming method parameter
   * @return object
   * @exception ResourceException Exception raised in failure situation
   */
  public Object getObject(RID rid, IResourceContext context, Class asClass)
    throws ResourceException;

  /**
   * Returns a resource object instance for the given RID and context. Internal
   * links to collections are resolved.
   *
   * @param rid The RID of the resource
   * @param createCollections If the resource does not exists create a new
   *      collection with that name. All collections in the path will be
   *      created.
   * @param context The resource context object
   * @return The resource object instance
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getResource(RID rid, IResourceContext context, boolean createCollections)
    throws ResourceException;

  /**
   * Returns a resource object instance for the given RID and context.
   *
   * @param rid The RID of the resource
   * @param resolveInternalLinks <code>true</code> : If the RID contains
   *      internal links to collections the framework will find and return the
   *      target resource. In this case the RID of the returned resource will
   *      not be equal to the specified RID parameter. <code>false</code> : If
   *      the RID contains internal links to collections a <code>
   *      InternalLinkException</code> exception is thrown.
   * @param context The resource context object
   * @return The resource object instance
   * @exception ResourceException Exception raised in failure situation
   * @see com.sapportals.wcm.repository.IResource#getRID()
   * @see com.sapportals.wcm.repository.IResource#getAccessRID()
   */
  public IResource getResource(RID rid, boolean resolveInternalLinks, IResourceContext context)
    throws ResourceException;


  /**
   * @param uriList TBD: Description of the incoming method parameter
   * @param errors TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @return resources
   * @exception ResourceException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP3, replaced by getResources() method with
   *      IRidList parameter.
   */
  public IResourceList getResources(IUriList uriList,
    ResourceErrors errors,
    IResourceContext context)
    throws ResourceException;

  /**
   * Returns a list of resource instances for an array of resource identifiers.
   * The resulting list will not have the same size as list of RIDs if some
   * resources do not exist or ResourceExceptions occured for some resources.
   * Exceptions are collected in an <code>ResourceErrors</code> instance.
   *
   * @param ridList A list of resource identifiers. The list can contain RIDs
   *      from different managers.
   * @param errors A reference to a <code>ResourceErrors</code> instance. If
   *      this is <code>null</code> all exceptions will be ignored.
   * @param context The resource context
   * @return A list of resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getResources(IRidList ridList,
    ResourceErrors errors,
    IResourceContext context)
    throws ResourceException;

  /**
   * Returns a list of resource instances for an array of resource identifiers
   * and removes resources without the specified permission(s) from the result
   * list.
   *
   * @param ridList A list of resource identifiers. The list can contain RIDs
   *      from different managers.
   * @param errors A reference to a <code>ResourceErrors</code> instance. If
   *      this is <code>null</code> all exceptions will be ignored.
   * @param permissionNames An array of permission names. Permission name
   *      constants are defined in {@link IPermission}.
   * @param context The resource context
   * @return A list of resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getResources(IRidList ridList,
    ResourceErrors errors,
    String[] permissionNames,
    IResourceContext context)
    throws ResourceException;

  /**
   * Returns a list of resource instances for a list of resource identifiers and
   * possible {@IPropertyMap}s for the resources. The property maps need not be
   * present for each and every resource identifier. If a property map is given
   * for a resource, the caller indicates that he is content to see those
   * properties on the returned resource object. If these properties are visible
   * on the returned resource or if the repository will return the actual,
   * current properties is up to the repository.<p>
   *
   * Repositories are however encouraged to sacrifice exactness for speed, e.g.
   * prefer the client supplied properties compared to (maybe costly) resource
   * lookups and validations.
   *
   * @param ridList A list of resource identifiers. The list can contain RIDs
   *      from different managers.
   * @param propertyMaps a map holding an {@link IPropertyMap} with the resource
   *      identifier {@RID} as key.
   * @param errors A reference to a <code>ResourceErrors</code> instance. If
   *      this is <code>null</code> all exceptions will be ignored.
   * @param context The resource context
   * @return A list of resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getResources(IRidList ridList,
    Map propertyMaps,
    ResourceErrors errors,
    IResourceContext context)
    throws ResourceException;

  /**
   * Returns a list of resource instances for a list of resource identifiers and
   * possible {@IPropertyMap}s for the resources. The property maps need not be
   * present for each and every resource identifier. If a property map is given
   * for a resource, the caller indicates that he is content to see those
   * properties on the returned resource object. If these properties are visible
   * on the returned resource or if the repository will return the actual,
   * current properties is up to the repository.<p>
   *
   * Repositories are however encouraged to sacrifice exactness for speed, e.g.
   * prefer the client supplied properties compared to (maybe costly) resource
   * lookups and validations.
   *
   * @param ridList A list of resource identifiers. The list can contain RIDs
   *      from different managers.
   * @param propertyMaps a map holding an {@link IPropertyMap} with the resource
   *      identifier {@RID} as key.
   * @param errors A reference to a <code>ResourceErrors</code> instance. If
   *      this is <code>null</code> all exceptions will be ignored.
   * @param permissionNames An array of permission names. Permission name
   *      constants are defined in {@link IPermission}.
   * @param context The resource context
   * @return A list of resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getResources(IRidList ridList,
    Map propertyMaps,
    ResourceErrors errors,
    String[] permissionNames,
    IResourceContext context)
    throws ResourceException;

  /**
   * @deprecated As of EP 5.0 SP3, replaced by method with RID parameter.
   */
  public boolean checkExistence(URI uri, IResourceContext context)
    throws ResourceException, AuthorizationRequiredException;


  /**
   * Returns <code>true</code> if a resource exists for the given resource
   * identifier.
   * <p>
   * Note that this method should only be used if the caller really only requires this piece of information. Using the
   * return value to decide whether it makes sense to call {@link #getResource(RID, IResourceContext)}, or to detect whether a particular
   * RID isn't in use yet DOES NOT WORK due to possible race conditions. Instead, just do the respective method calls
   * and properly check results/exceptions.
   * 
   * @param rid The RID to check
   * @param context The resource context
   * @return True if the resource exists
   * @exception ResourceException Exception raised in failure situation
   * @exception AuthorizationRequiredException Exception raised in failure
   *      situation
   */
  public boolean checkExistence(RID rid, IResourceContext context)
    throws ResourceException, AuthorizationRequiredException;

  /**
   * Returns a instance of the ServiceFactory
   *
   * @return The service factory
   * @exception ResourceException Exception raised in failure situation
   */
  public IRepositoryServiceFactory getServiceFactory()
    throws ResourceException;

  /**
   * Returns a instance of the ManagerFactory
   *
   * @return The service factory
   * @exception ResourceException Exception raised in failure situation
   */
  public IManagerFactory getManagerFactory()
    throws ResourceException;

  /**
   * Returns a instance of the FilterFactory
   *
   * @return The service factory
   * @exception ResourceException Exception raised in failure situation
   */
  public IFilterFactory getFilterFactory()
    throws ResourceException;

  /**
   * Returns a new resource context for services with an anonymous user
   *
   * @return a new resource context for services with an anonymous user
   * @exception ResourceException on error
   */
  public IResourceContext getServiceContext()
    throws ResourceException;

  /**
   * Returns a new resource context for services with an anonymous user
   *
   * @param userID the ID of the service user account
   * @return a new resource context for services with an anonymous user
   * @exception ResourceException on error
   */
  public IResourceContext getServiceContext(String userID)
    throws ResourceException;
  
  /**
   * Creates a resource batch instance for the given resource list.
   * @param resourceList a resource list instance that MUST be the result of
   * a previous call of {@link ICollection#getChildren()} or {@link IResourceFactory#getResources(IRidList, ResourceErrors, IResourceContext)}.
   */
  //public IResourceBatch getResourceBatch(IResourceList resourceList);
}
