/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.so.*;
import com.sapportals.wcm.util.content.*;
import com.sapportals.wcm.util.events.IEventList;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.uri.URI;
import com.sapportals.wcm.util.uri.URL;

import java.util.Date;
import java.util.List;

/**
 * A resource represents an object in a repository and is uniquely identified by
 * a resource identifier ({@link RID}). Clients use this interface and the
 * derived {@link ICollection} interface to manipulate the resources in a
 * repository. Every repository is controlled by a repository manager ({@link
 * #getRepositoryManager()}), which is resposible for createing resource instances
 * and carrying out all basic operations (create, update content or properties,
 * deleting, etc.). <p>
 *
 * The Repository Framework knows these kinds of resources: <p>
 *
 * <b>Normal resources:</b> A resource has an identifier (RID) which is unique
 * within a CM system ({@link #getRID()}) and usually has content ({@link
 * #getContent()}) and properties ({@link #getProperties()}). <p>
 *
 * <b>Collections:</b> {@link #isCollection()} returns <code>true</code> . The
 * items in a collection are called "children" and are accessed with {@link
 * ICollection#getChildren()}. <p>
 *
 * <b>Links:</b> {@link #getLinkType()} returns INTERNAL or EXTERNAL. Internal
 * links point to other resources (in the same or a different repository),
 * external links point to resources outside the CM system (URL). The link
 * target for internal links is access with {@link #getTargetResource()}. The
 * target can again be a link. Links to collections are themselfs collections. A
 * link resource usually has no content but might have properties. <p>
 *
 * <b>Version Controlled Resources:</b> {@link #isVersioned()} returns <code>true
 * </code>. A version controlled resource must be checked out to be modified.
 * All revisions of the content and/or properties will be recorded in a version
 * history ({@link #getVersionHistory()}). The resources in the version history
 * are called "revisions". <p>
 *
 * <b>Revisions:</b> {@link #isRevision()} returns <code>true</code> . A revision
 * provides read-only access to the properties and the content of a version of
 * the version controlled resource. <p>
 *
 * <b>Deleted resources:</b> As soon as the {@link #delete()} method has been
 * called the resource instance is in a special state. All methods except
 * getRID() and getName() are prohibited and will throw an exception. <p>
 * <p>
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @version $Revision: 1.5 $
 */
public interface IResource extends ITypeCast {

  /**
   * Returns a collection of supported resource events and semantic events for
   * this resource.
   *
   * @return The set of supported events
   * @exception ResourceException Exception raised in failure situation
   */
  public IEventList getSupportedEvents()
    throws ResourceException;

  /**
   * Returns a collection of supported options for the resource.
   *
   * @return The set of supported operations
   */
  public ISupportedOptionSet getSupportedOptions();

  /**
   * Returns the resource context
   *
   * @return The context
   */
  public IResourceContext getContext();

  /**
   * Get the repository manager for this resource
   *
   * @return Reference to a repository manager instance
   */
  public IRepositoryManager getRepositoryManager();

  /**
   * Get name of the resource. The resource name ist the last part if its RID.
   * path.
   *
   * @return The name of the resource
   * @exception ResourceException
   */
  public String getName()
    throws ResourceException;

  /**
   * Get value of the system property CreationDate
   *
   * @return Property value, can be null if property is not supported by the
   *      repository.
   */
  public Date getCreationDate();

  /**
   * Get value of the system property LastModified
   *
   * @return Property value, can be null if property is not supported by the
   *      repository.
   */
  public Date getLastModified();

  /**
   * Get value of the system property Description
   *
   * @return Property value, can be null if property is not supported by the
   *      repository.
   */
  public String getDescription();

  /**
   * Returns the value of the Displayname system property or <code>null</code>
   * if the property is not set.
   *
   * @return the value of the Displayname system property or <code>null</code>
   *      if the property is not set.
   */
  public String getDisplayName();

  /**
   * Returns the value of the Displayname system property or <code>null</code>
   * if the property is not set. If the boolean parameter is <code>true</code>
   * the name (last part of the resource identifier) is used as a fallback if
   * the property is not set.
   *
   * @param orNameIfNull If true, the name (last part of the resource
   *      identifier) is used as a fallback if the DisplayName property is not
   *      set.
   * @return The display name.
   */
  public String getDisplayName(boolean orNameIfNull);

  /**
   * Returns the value of the ResourceType system property or an empty string if
   * the property is not set.
   *
   * @return the value of the ResourceType system property or an empty string if
   *      the property is not set
   */
  public String getResourceType();

  /**
   * Get value of the system property CreatedBy. This will be some kind of user
   * name or ID.
   *
   * @return Property value, can be null if property is not supported by the
   *      repository.
   */
  public String getCreatedBy();

  /**
   * Get value of the system property LastModifiedBy. This will be some kind of
   * user name or ID.
   *
   * @return Property value, can be null if property is not supported by the
   *      repository.
   */
  public String getLastModifiedBy();

  /**
   * Get value of the system property ETag, reflecting the <a
   * href="http://greenbytes.de/tech/webdav/rfc2616.html#rfc.section.3.11">HTTP
   * entity tag</a> for the unfiltered content of this resource. If two strong
   * entity tags for a given RID are identical, the contents is identical
   * octet-by-octet. If two weak entitiy tags are identical, the contents is
   * semantically equivalent, but may differ in it's octet representation. Tags
   * are weak if and only if they start with a leading "W/". <p>
   *
   * Note: The ETag of a resource is not a unique identifier.
   *
   * @return Property value, can be null if etags are not supported by the
   *      resource.
   */
  public String getETag();

  /**
   * Returns the language
   *
   * @return The language ID
   */
  public String getLanguage();

  /**
   * @return uRI
   * @exception ResourceException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP3, replaced by {@link #getRID()}
   */
  public URI getURI()
    throws ResourceException;

  /**
   * Get the ID (path) of this resource. This RID might differ from the <i>
   * access RID</i> that was used to create this resource object with a call to
   * ResourceFactory.getResource(), e.g. if this RID contained one ore more
   * links to collections in the same or a different repository.<br>
   * Example:<br>
   * Access RID: <code>getResource("/prefix-repository-a/xxx/link-to-collection-in-repository-b/xxx/test.html",
   * ...)</code> <code>getRID()</code> returns:
   * /prefix-repository-b/dir1/dir2/link-target/xxx/test.html <p>
   *
   * A repository might provide special RIDs for revision resources to provide
   * access to all the resources in the version history via some kind of
   * "virtual" collection. The syntax of the revision RID is specific to each
   * repository.
   *
   * @return The RID
   * @exception ResourceException
   * @see #isRevision()
   * @see #getAccessRID()
   */
  public RID getRID()
    throws ResourceException;

  /**
   * @return accessURI
   * @exception ResourceException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP3, replaced by {@link #getAccessRID()}
   */
  public URI getAccessURI()
    throws ResourceException;

  /**
   * Returns the RID that was used to create this resource object in <code>
   * IResourceFactory.getResource</code>
   *
   * @return The access RID
   * @exception ResourceException
   * @see #getRID()
   */
  public RID getAccessRID()
    throws ResourceException;


  /**
   * Get value of the system property Readonly
   *
   * @return Property value
   */
  public boolean isReadOnly();

  /**
   * Get value of the system property Hidden
   *
   * @return Property value
   */
  public boolean isHidden();

  /**
   * Returns <code>true</code> if this resource is a collection
   *
   * @return <code>true</code> if this resource is a collection
   */
  public boolean isCollection();

  /**
   * Changes the name of this resource.
   *
   * @param newName The new name
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void rename(String newName)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns the parent collection of this resource. The return value will be
   * <code>null</code> if this resource is the root collection. If this resource
   * is a revision (@link #isRevision) the returned resource is the parent
   * collection of the versioned controlled resource of the revision. The return
   * value might be <code>null</code> because repository manager implementation
   * might not support this call for revision at all or it is not possible for
   * some revisions, for example if the version controlled resource does not
   * exist anymore.
   *
   * @return the parent collection of this resource
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public ICollection getParentCollection()
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the content of this resource.
   *
   * @return the content of this resource.
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public IContent getContent()
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the content of this resource.
   *
   * @param handleExternalLink If <code>true</code> , try to read the content
   *      from an external source, if the resource is a external link.
   * @return the content of this resource.
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   * @deprecated As of EP 5.0 SP 5 <p>
   *
   *      This method is deprecated because it opens a potential security hole:
   *      The URL of an external link can have different schemas (e.g. "file:")
   *      which are all accessed with the security context of the servlet
   *      engine. This could be (ab)used to access files/resources on the local
   *      server and other systems without the security context of the logged in
   *      user. <p>
   *
   *      <b>No longer supported as of SP6: Throws a NotSupportedException if
   *      parameter is <code>true</code> </b>
   */
  public IContent getContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the content of this resource without applying content filters.
   *
   * @return the unfiltered content of this resource.
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public IContent getUnfilteredContent()
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the content of this resource without applying content filters.
   *
   * @param handleExternalLink If <code>true</code> , try to read the content
   *      from an external source, if the resource is a external link.
   * @return the unfiltered content of this resource.
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public IContent getUnfilteredContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException;

  /**
   * Deletes this resource from the repository. If the resource is a collection
   * all its children will be deleted.
   *
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void delete()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Updates the content of this resource.
   *
   * @param newContent The new content
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @see Content
   */
  public void updateContent(IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Updates the content and properties of this resource in an atomic operation.
   * If some or all properties cannot be saved the content will not change - and
   * vice versa.
   *
   * @param newContent The new content
   * @param properties A map with properties
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @exception SetPropertiesException If some properties could not be set or
   *      removed
   * @see Content
   */
  public void update(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException;


  /**
   * @param destinationURI TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by copy() with RID parameter.
   */
  public IResource copy(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * @param destinationURI TBD: Description of the incoming method parameter
   * @param position TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by copy() with RID parameter.
   */
  public IResource copy(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * @param destinationURI TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by move() with RID parameter.
   */
  public IResource move(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * @param destinationURI TBD: Description of the incoming method parameter
   * @param position TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by move() with RID parameter.
   */
  public IResource move(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Copy the resource.
   *
   * @param destinationRID the RID of the destination (includes the name). All
   *      collections in the destination RID must already exists.
   * @param param The copy parameter object, can be null
   * @return The new resource object
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IResource copy(RID destinationRID, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;


  /**
   * Copy the resource to a destination collection that supports ordered
   * children.
   *
   * @param destinationRID the RID of the destination (includes the name). All
   *      collections in the destination RID must already exists.
   * @param position The positioning information
   * @param param The copy parameter object, can be null
   * @return The new resource object
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IResource copy(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Move the resource's content and properties
   *
   * @param destinationRID The RID of the destination (includes the name). All
   *      collections in the destination RID must already exists.
   * @param param The copy parameter object, can be null
   * @return The new resource object
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IResource move(RID destinationRID, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Move the resource to a destination collection that supports ordered
   * children. If the destinationRID is the same as the RID of this resource the
   * position parameter can be used to reposition the resource in the sorted
   * collection. In this case the return value will be <code>null</code> .
   *
   * @param destinationRID The RID of the destination
   * @param position The positioning information
   * @param param The copy parameter object, can be null
   * @return The new resource object or null
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IResource move(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns a map with <code>IProperty</code> references for the resource.
   *
   * @return The property map
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IPropertyMap getProperties()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns a map with <code>IProperty</code> references for the resource.
   *
   * @param propNameList The list with the requested property names.
   * @return The property map
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IPropertyMap getProperties(IPropertyNameList propNameList)
    throws ResourceException, NotSupportedException, AccessDeniedException;


  /**
   * Set all properties of the resource. The call is not atomic. That means
   * failuers to set or remove some properties will not cause the operation to
   * fail completely. All errors for properties in the map and also attemps to
   * remove required properties are collected and reported via a
   * SetPropertiesException.
   *
   * @param props The map with the properties.
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @exception SetPropertiesException If the call was succesfull but some of
   *      the properties could not be set or removed
   */
  public void setProperties(IPropertyMap props)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException;

  /**
   * Change the properties of the resource. The call is not atomic. That means
   * failures to set or remove some properties will not cause the operation to
   * fail completely. All errors for properties in the map and also attempts to
   * remove required properties are collected and reported via a
   * SetPropertiesException.
   *
   * @param propChangeList change list (contains {@link IProperty} entries for properties to be set and {@link IPropertyName} entries for properties to be removed)
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @exception SetPropertiesException If the call was succesfull but some of
   *      the properties could not be set or removed
   */
  public void setProperties(List propChangeList)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException;


  /**
   * Get a property.
   *
   * @param propName The property name of the property to read
   * @return A reference to the property or null if the property was not found
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IProperty getProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Get a property of the resource or inherit it. If the resource does not have
   * the property attached to it it will inherit it from a collection in its
   * path. The list of all parent(ancestor) collections up to the root
   * collection is searched for the property.
   *
   * @param propName The property name of the property to find
   * @return A reference to the property or null if no property was found
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public IProperty getInheritedProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Sets a property.
   *
   * @param prop The property to set
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void setProperty(IProperty prop)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Removes the property.
   *
   * @param propName The property name of the property to delete
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void deleteProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Lock the resource using the current user in the context. Scope is
   * "exclusive", type is "write" and time-out is "infinite".
   *
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ILockInfo lock()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Lock the resource using the current user in the context.
   *
   * @param lockProperties The lock properties defining the scope, type and
   *      time-out of the lock.
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ILockInfo lock(ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Unlock the resource using the current user in the context.
   *
   * @param lockInfo The LockInfo
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void unlock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Refresh the lock.
   *
   * @param lockInfo The LockInfo
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void refreshLock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Get a LockInfo for the token.
   *
   * @param lockToken TBD: Description of the incoming method parameter
   * @return lockByToken
   * @exception NotSupportedException
   * @exception ResourceException Exception raised in failure situation
   */
  public ILockInfo getLockByToken(String lockToken)
    throws ResourceException, NotSupportedException;

  /**
   * Returns a collection of all existing locks.
   *
   * @return A collection of ILockInfo instances
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ILockInfoCollection getLocks()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns <code>true</code> if this resource is locked.
   *
   * @return <code>true</code> if this resource is locked.
   * @exception ResourceException
   */
  public boolean isLocked()
    throws ResourceException;

  /**
   * Returns <code>true</code> if this resource is locked by the current user in
   * the context.
   *
   * @return <code>true</code> if this resource is locked by the current user in
   *      the context.
   * @exception ResourceException
   */
  public boolean isLockedByMe()
    throws ResourceException;

  /**
   * Returns the target URL of a link or <code>null</code> if the resource is
   * not a link.
   *
   * @return the target URL of a link or <code>null</code> if the resource is
   *      not a link.
   * @exception ResourceException
   */
  public URL getTargetURL()
    throws ResourceException;

  /**
   * Changes the target URL of a link.
   *
   * @param url targetURL to be set
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public void setTargetURL(URL url)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns the target resource of an internal link. If the resource is not a
   * link or an external link the method returns <code>null</code> . The method
   * will resolve relativ internal links and links to target resources which are
   * links.
   *
   * @return The target resource
   * @exception ResourceException
   */
  public IResource getTargetResource()
    throws ResourceException;

  /**
   * Returns the link type (NONE if the resource is not a link).
   *
   * @return the link type (NONE if the resource is not a link).
   * @exception ResourceException
   */
  public LinkType getLinkType()
    throws ResourceException;

  /**
   * Changes the link type.
   *
   * @param linkType linkType to be set
   * @exception ResourceException
   * @exception AccessDeniedException If authorization failed
   */
  public void setLinkType(LinkType linkType)
    throws ResourceException, AccessDeniedException;


  // --------------- Versioning ------------------

  /**
   * Enables or disables versioning. The operation will not be performed if the
   * parameter is <code>true</code> /<code>false</code> but versioning is
   * already enabled/disabled.
   *
   * @param enable TBD: Description of the incoming method parameter
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void enableVersioning(boolean enable)
    throws ResourceException, NotSupportedException, AccessDeniedException;


  /**
   * Returns <code>true</code> if versioning is enabled for this resource.
   *
   * @return <code>true</code> if versioning is enabled for this resource.
   * @exception ResourceException
   */
  public boolean isVersioned()
    throws ResourceException;

  /**
   * Returns <code>true</code> if this resource is a revision.
   *
   * @return <code>true</code> if this resource is a revision.
   */
  public boolean isRevision();

  /**
   * Returns the identifier for this revision. Usually this will be some kind of
   * a version number (depends on the repository manager implementation). If
   * this resource is a version controlled resource ({@link #isVersioned()}) the
   * return value will be <code>null</code> .
   *
   * @return the identifier for this revision.
   */
  public String getRevisionID();

  /**
   * Returns <code>true</code> if this resource is currently checked out.
   *
   * @return <code>true</code> if this resource is currently checked out.
   * @exception ResourceException
   */
  public boolean isCheckedOut()
    throws ResourceException;

  /**
   * Check out the resource for editing. This method optionally returns an
   * CheckOutInfo object. This holds information about the expected revision ID
   * and RID of the new revision that will be created when the resource is
   * checked in. The application can provide this RID when checking in. This
   * check-in will fail if the new revision RID would be different for some
   * reason.
   *
   * @return A ICheckOutInfo instance or <code>null</code> if the repository
   *      does not support the "expected check-in RID" option.
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ICheckOutInfo checkOut()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Undo a previous check out operation.
   *
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public void undoCheckOut()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * @param newContent TBD: Description of the incoming method parameter
   * @param properties TBD: Description of the incoming method parameter
   * @param ignorePropertyFailures TBD: Description of the incoming method
   *      parameter
   * @param expectedCheckInURI TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @exception ExpectedCheckInURIException Exception raised in failure
   *      situation
   * @deprecated As of EP 5.0 SP4
   */
  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, URI expectedCheckInURI)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInURIException;

  /**
   * Check in a new version of the resource with new content and/or properties.
   *
   * @param newContent The new content. Can be <code>null</code> : the existing
   *      content will be used for the new version.
   * @param properties The new properties, can be <code>null</code>
   * @return checkin status information
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Check in a new version of the resource with new content and/or properties.
   *
   * @param newContent The new content. Can be <code>null</code> : the existing
   *      content will be used for the new version.
   * @param properties The new properties, can be <code>null</code>
   * @param ignorePropertyFailures Ignore exceptions if some or all properties
   *      could not be set/removed
   * @return checkin status information
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   */
  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Check in a new version of the resource with new content and/or properties.
   *
   * @param newContent The new content, can be <code>null</code>
   * @param properties The new properties, can be <code>null</code>
   * @param ignorePropertyFailures Ignore exceptions if some or all properties
   *      could not be set/removed
   * @param expectedCheckInRID Create the new revision with this revision RID.
   *      The check-in <em>may</em> fail if the new revision RID would be
   *      different. In this case an ExpectedCheckInRID exception will be thrown
   *      which will contain a new "expected check-in RID". To make sure that
   *      the expectedCheckInRID actually <em>was</em> used, it needs to be
   *      compared to the RID returned by {@link ICheckInInfo#getRevisionRID()}
   *      (this is because a repository may check-in the resource although it
   *      couldn't use the expectedCheckInRID).
   * @return checkin status information
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @exception ExpectedCheckInRIDException Exception raised in failure
   *      situation
   * @see #checkOut()
   */
  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, RID expectedCheckInRID)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInRIDException;

  /**
   * Returns the version history of this resource. Note that {@link
   * IVersionHistory} is restricted to linear (non-branching) version histories,
   * therefore this method will fail if the version history is not linear.
   *
   * @return versionHistory
   * @exception ResourceException
   * @exception NotSupportedException
   * @returns the version history of this resource.
   */
  public IVersionHistory getVersionHistory()
    throws ResourceException, NotSupportedException;

  /**
   * Make this revision of the resource the current version. This is equivalent
   * to copying the revision to the checked-out resource (updating content and
   * dead properties) followed by a checkin. <p>
   *
   * Note that {@IVersionControlledResource} supports a more powerful update
   * mechanism that will actually create proper branches in the version history.
   * <p>
   *
   *
   *
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If authorization failed
   * @deprecated As of EP 5.0 SP5 This method will fail if it's not possible to
   *      unambigously identify the version-controlled resource to be updated
   *      (this may happen when working resources or workspaces are used).
   *      Instead check out the version-controlled resource, copy the version
   *      onto it, then check it back again.
   */
  public void setAsCurrentVersion()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns a list of currently checked out resources. This method is usually
   * only supported for revision resources.
   *
   * @return Resource list, or <code>null</code> if no resources are checked out
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException
   */
  public IResourceList getCheckedOutResources()
    throws ResourceException, NotSupportedException, AccessDeniedException;

  // Classification of resource type

  /**
   * Returns whether this resource has a representation as the class. This
   * method must be implemented in a resource class that extends <code>
   * ResourceImpl</code> class. This base class will always return <code>false
   * </code>.
   *
   * @param classtest class to test for.
   * @return a
   */
  public boolean isA(Class classtest);

  /**
   * Get a representation of this resource as the specified class. This method
   * must be implemented in a resource class that extends <code>ResourceImpl
   * </code> class. This base class will always return <code>null</code> .
   *
   * @param classref TBD: Description of the incoming method parameter
   * @return reference to object of this class or <code>null</code> .
   * @exception ResourceException Exception raised in failure situation
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   */
  public Object as(Class classref)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Queries the resource and optionally the collection hierarchy for resources.
   * The search criteria are properties and property values. Note that this
   * method does <em>not</em> automatically follow links. Also note that the
   * query expression <em>must</em> have been built using the correct property
   * search manager (the one configured for the repository manager the resource
   * is in).
   *
   * @param query The query specification
   * @param depth The search depth (0, 1, or Integer.MAX_VALUE)
   * @param maxResults The maximum number of resources that will be returned
   * @param includeRevisions when set to <code>true</code> , also descend into
   *      version history (independantly of depth)
   * @return The result list
   * @exception ResourceException
   * @exception NotSupportedException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   */
  public IResourceList search(IQueryExpression query, int depth, int maxResults, boolean includeRevisions)
    throws ResourceException, NotSupportedException, AccessDeniedException;

}
