/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

/**
 * Constructs {@link IQueryExpression}s for later use with the corresponding
 * property search manager. <p>
 *
 * An {@link IQueryBuilder} constructs and combines {@link IQueryExpression}s, creating new
 * {@link IQueryExpression}s, which finally are executed against the property search
 * manager. Construction is done in the following way <pre>
 * builder.equ(x, y).and(builder.eq(z, "name")).or(builder.like(y, "%jpg"))
 *
 * builder.not(builder.greaterThan(x, new Integer(7)))
 * </pre>
 * <p>
  * Note that the result of comparing with both undefined properties and multi-valued properties is "undefined" (as per
 * standard SQL three-valued logic). When combined with "true" and "false" in expressions,
 * "undefined" behaves as below:
 * <pre>
 * undefined AND undefined -> undefined
 * undefined OR undefined -> undefined
 * undefined AND true -> undefined
 * undefined AND false -> false
 * undefined OR true -> true
 * undefined OR false -> undefined
 * NOT(undefined) -> undefined
 * </pre>
 * (the only exception being the special query expression isDefined() which always evaluates to "true" or "false").
 * <p>
 *
 * Copyright (c) SAP AG 2001-2004
 *
 * @author stefan.eissing@greenbytes.de
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.15 $
 */
public interface IQueryBuilder {

  /**
   * Returns a new expression which tests property == value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property == value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eq(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property == value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property == value
   */
  public IQueryExpression eqIgnoreCase(IPropertyName name, String value);


  /**
   * Returns a new expression which tests property != value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property != value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEq(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property != value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property != value
   */
  public IQueryExpression notEqIgnoreCase(IPropertyName name, String value);


  /**
   * Returns a new expression which tests property < value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression less(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property < value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression less(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property < value, ignoring case.
   *
   * @param name the name3 of the property
   * @param value to compare against
   * @return expression representing property < value
   */
  public IQueryExpression lessIgnoreCase(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property <= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property <= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEq(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property <= value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property <= value
   */
  public IQueryExpression lessEqIgnoreCase(IPropertyName name, String value);


  /**
   * Returns a new expression which tests property > value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greater(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property > value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greater(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property > value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property > value
   */
  public IQueryExpression greaterIgnoreCase(IPropertyName name, String value);


  /**
   * Returns a new expression which tests property >= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests property >= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEq(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property >= value, ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing property >= value
   */
  public IQueryExpression greaterEqIgnoreCase(IPropertyName name, String value);

  /**
   * Returns a new expression which is the complement of the given expression.
   *
   * @param expression to return complement of
   * @return expression representing !expression
   */
  public IQueryExpression not(IQueryExpression expression);

  /**
   * Returns a new expression which tests property is like value, where value is
   * a wildcard expression. Wildcard characters are "_" and "%", where "_"
   * matches one character and "%" matches 0 or more characters. "\" can be used
   * as escape character and is only allowed to escape itself, "_" and "%".
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing property like value
   */
  public IQueryExpression like(IPropertyName name, String value);

  /**
   * Returns a new expression which tests property is like value, where value is
   * a wildcard expression. Wildcard characters are "_" and "%", where "_"
   * matches one character and "%" matches 0 or more characters. "\" can be used
   * as escape character and is only allowed to escape itself, "_" and "%".
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing property like value
   */
  public IQueryExpression likeIgnoreCase(IPropertyName name, String value);

  /**
   * Returns a new expression which tests for the properties existence.
   *
   * @param name the name of the property
   * @return expression representing "if exists property"
   */
  public IQueryExpression isDefined(IPropertyName name);

  /**
   * Returns a new expression which tests if the resource is a collection
   *
   * @return expression representing test if resource is collection
   */
  public IQueryExpression isCollection();

  /**
   * @return constant false
   */
  public IQueryExpression constFalse();

  /**
   * @return constant true
   */
  public IQueryExpression constTrue();
}

