/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.Iterator;
import java.util.Map;

/**
 * A read-only map of properties. <p>
 *
 *
 *
 * @author m.breitenfelder@sapportals.com
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.17 $
 */
public interface IPropertyMap {

  /**
   * Returns the number of properties in the map.
   *
   * @return the number of properties.
   */
  public int size();

  /**
   * Returns true if the map is empty.
   *
   * @return true if the map is empty.
   */
  public boolean isEmpty();

  /**
   * Returns true if the map contains a property with the specified name.
   *
   * @param propertyName the property name to search for.
   * @return true if the map contains a property with the specified name.
   */
  public boolean containsProperty(IPropertyName propertyName);

  /**
   * Returns true if the map contains the specified property.
   *
   * @param property the property to search for.
   * @return true if the map contains the specified property.
   */
  public boolean containsProperty(IProperty property);

  /**
   * Returns the property with the specified name or <code>null</code> if there
   * is no such property.
   *
   * @param propertyName the property name to search for.
   * @return the property with the specified name or <code>null</code> if there
   *      is no such property.
   */
  public IProperty get(IPropertyName propertyName);

  /**
   * Returns an iterator for the property map.
   *
   * @return an iterator for the property map.
   */
  public IPropertyIterator iterator();

  /**
   * Returns a copy of the map that can be modified.
   *
   * @return a copy of the map that can be modified.
   */
  public IMutablePropertyMap getMutable();


  /**
   * Returns a map from IPropertyNames to Exceptions for every property for
   * which an internal error occured during retrieval (not being defined is <em>
   * not</em> an error condition).
   *
   * @return the map (will be empty if no exception have been recorded)
   */
  public Map getExceptions();

  /**
   * Given a property that does not appear in the property map, query for a
   * possible exception that occured internally when computing it.
   *
   * @param propertyName name of property
   * @return the exception or <code>null</code>
   */
  public Exception getException(IPropertyName propertyName);
}

