/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.WcmException;

import com.sapportals.wcm.repository.enum.PropertyType;

import java.util.Date;
import java.util.List;
import java.util.Properties;

/**
 * This interface provides read-only access to a property of a resource. A
 * property has a name ({@link IPropertyName}), data type ({@link
 * PropertyType}), value(s), atttributes, a definition ({@link IPropertyDef})
 * and a description.
 *
 * @author SAP AG
 * @see IPropertyName
 * @see PropertyType
 * @see IPropertyDef
 * @see IMutableProperty
 * @version $Revision: #31 $
 */
public interface IProperty {

  /**
   * Returns the property name
   *
   * @return A property name object
   */
  IPropertyName getPropertyName();

  /**
   * Return the property's data type
   *
   * @return A property type constant
   */
  PropertyType getType();

  /**
   * Returns true if the property is multi-valued
   *
   * @return multivalued
   */
  boolean isMultivalued();

  /**
   * Get a reference to the proeprty definition.
   *
   * @return The property definition
   */
  IPropertyDef getPropertyDef();

  /**
   * Get the property value converted to a string.
   *
   * @return A String
   */
  String getValueAsString();

  /**
   * Get the property value if the type is PropertyType.BOOLEAN
   *
   * @return The value
   */
  boolean getBooleanValue();

  /**
   * Get the property value if the type is PropertyType.INTEGER
   *
   * @return The value
   */
  int getIntValue();

  /**
   * Get the property value if the type is PropertyType.LONG
   *
   * @return The value
   */
  long getLongIntValue();

  /**
   * Get the property value if the type is PropertyType.DATE
   *
   * @return The value
   */
  Date getDateValue();

  /**
   * Get the property value if the type is String
   *
   * @return The value
   */
  String getStringValue();

  /**
   * Returns the property values if this is a multi-valued property.
   * @return An unmodifiable List of value objects (String, Date, Integer, Long, Boolean, XMLMarkup)
   */
  public List getValues() throws ResourceException;
   
  /**
   * Returns the property value as a Object if this is not a multi-valued property.
   * @return The value object (String, Date, Integer, Long, Boolean, XMLMarkup)
   */
  public Object getValue() throws ResourceException;

  /**
   * Returns a copy of this propertry that can be modified
   *
   * @return A mutable property instance
   */
  public IMutableProperty getMutable();

  /**
   * Get the value of an attribute
   *
   * @param name Attribute name
   * @return The value, ot null if the name does not exist
   * @exception ResourceException
   */
  String getAttribute(String name)
    throws ResourceException;

  /**
   * Returns a collection of attributes
   *
   * @return The attributes
   * @exception ResourceException
   */
  Properties getAttributes()
    throws ResourceException;


  /**
   * Returns the description of the property
   *
   * @return description
   * @exception WcmException Exception raised in failure situation
   */
  public String getDescription()
    throws WcmException;

  /*
   * Returns the description of the property,
   * using the locale from the resource context
   */
  public String getDescription(IResourceContext context)
    throws WcmException;

  /**
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IPropertyDef lnkIPropertyDef;
   */
  /**
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IPropertyName lnkIPropertyName;
   */
}
