/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;
import java.util.Date;

import java.util.List;
import java.util.Properties;

/**
 * A property of a resource object with a mutable interface.
 *
 * @author SAP AG
 * @version $Revision: #31 $
 */
public interface IMutableProperty extends IProperty {

  /**
   * Sets the property value if the type ist PropertyType.INTEGER
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setIntValue(int value)
    throws ResourceException;

  /**
   * Sets the property value if the type ist PropertyType.DATE
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setDateValue(Date value)
    throws ResourceException;

  /**
   * Sets the property value if the type ist PropertyType.DATE
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setDateValueFromLong(long value)
    throws ResourceException;

  /**
   * Sets the property value if the type ist PropertyType.STRING
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setStringValue(String value)
    throws ResourceException;

  /**
   * Sets the property value if the type ist PropertyType.LONG
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setLongIntValue(long value)
    throws ResourceException;

  /**
   * Sets the property value if the type ist PropertyType.BOOLEAN
   *
   * @param value The new value
   * @exception ResourceException
   */
  public void setBooleanValue(boolean value)
    throws ResourceException;

  /**
   * Sets all values of a multi-value property
   *
   * @param values The new values. The following table shows the instance that
   *      has to be used for each type. </code> PropertyType | instanceof
   *      ------------------------- STRING | String BOOLEAN | Boolean INTEGER |
   *      Integer LONG | Long DATE | Date XML | String </code>
   * @exception ResourceException If the property is not defined as multi-valued
   *      If a instance of a value does not match the property type
   */
  public void setValues(List values)
    throws ResourceException;

  /**
   * Adds a value to the list for a multi-value property
   *
   * @param value The value to add. The following table shows the instance that
   *      has to be used for each type. </code> PropertyType | instanceof
   *      ------------------------- STRING | String BOOLEAN | Boolean INTEGER |
   *      Integer LONG | Long DATE | Date XML | String </code>
   * @exception ResourceException If the property is not defined as multi-valued
   *      If the instance of the value does not match the property type
   */
  public void addValue(Object value)
    throws ResourceException;

  /**
   * Set a value of a multi-value property
   *
   * @param index The zero-based index
   * @param value The new value
   * @exception ResourceException If the property is not defined as multi-valued
   *      If the instance of the value does not match the property type If the
   *      index it out of bounds
   */
  public void setValue(int index, Object value)
    throws ResourceException;

  /**
   * Remove a value of a multi-value property
   *
   * @param index The zero-based index
   * @exception ResourceException If the property is not defined as multi-valued
   *      If the index it out of bounds
   */
  public void removeValue(int index)
    throws ResourceException;

  /**
   * Set an attribute
   *
   * @param name Attribute name
   * @param value Attribute value
   * @exception ResourceException
   */
  public void setAttribute(String name, String value)
    throws ResourceException;

  /**
   * Set the complete collection of attributes
   *
   * @param attributes All attributes
   * @exception ResourceException
   */
  public void setAttributes(Properties attributes)
    throws ResourceException;

  /**
   * Removes a attribute
   *
   * @param name Attribute name
   * @exception ResourceException
   */
  public void removeAttribute(String name)
    throws ResourceException;
}
