/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;


/**
 * Builder for queries that operate on multi-valued properties. The expressions
 * are similar to the single-valued variations, evaluating to true if the
 * condition is met for an individual value of the property value list. <p>
 *
 * Note that the result of comparing with both undefined properties and single-valued properties is "undefined" (as per
 * standard SQL three-valued logic). When combined with "true" and "false" in expressions,
 * "undefined" behaves as below:
 * <pre>
 * undefined AND undefined -> undefined
 * undefined OR undefined -> undefined
 * undefined AND true -> undefined
 * undefined AND false -> false
 * undefined OR true -> true
 * undefined OR false -> undefined
 * NOT(undefined) -> undefined
 * </pre>
 * <p>
 *
 * Copyright (c) SAP AG 2002-2004
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.3 $
 */
public interface IMultiValueQueryBuilder {

  /**
   * Returns a new expression which tests that someValueOf(property) == value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) == value
   */
  public IQueryExpression someEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) == value,
   * ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) == value
   */
  public IQueryExpression someEqIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) != value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) != value
   */
  public IQueryExpression someNotEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) != value,
   * ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) != value
   */
  public IQueryExpression someNotEqIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) < value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) < value
   */
  public IQueryExpression someLess(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) < value,
   * ignoring case.
   *
   * @param name the name3 of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) < value
   */
  public IQueryExpression someLessIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) <= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) <= value
   */
  public IQueryExpression someLessEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) <= value,
   * ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) <= value
   */
  public IQueryExpression someLessEqIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) > value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) > value
   */
  public IQueryExpression someGreater(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) > value,
   * ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) > value
   */
  public IQueryExpression someGreaterIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) >= value.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) >= value
   */
  public IQueryExpression someGreaterEq(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) >= value,
   * ignoring case.
   *
   * @param name the name of the property
   * @param value to compare against
   * @return expression representing someValueOf(property) >= value
   */
  public IQueryExpression someGreaterEqIgnoreCase(IPropertyName name, Object value);

  /**
   * Returns a new expression which tests someValueOf(property) is like value,
   * where value is a wildcard expression. Wildcard characters are "_" and "%",
   * where "_" matches one character and "%" matches 0 or more characters. "\"
   * can be used as escape character.
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing someValueOf(property) like value
   */
  public IQueryExpression someLike(IPropertyName name, String value);

  /**
   * Returns a new expression which tests someValueOf(property) is like value,
   * where value is a wildcard expression. Wildcard characters are "_" and "%",
   * where "_" matches one character and "%" matches 0 or more characters. "\"
   * can be used as escape character.
   *
   * @param name the name of the property
   * @param value the wildcard expression
   * @return expression representing someValueOf(property) like value
   */
  public IQueryExpression someLikeIgnoreCase(IPropertyName name, String value);

}

