/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.util.uri.*;

/**
 * Extended interface for collections, providing <br>
 * - extended link creation methods. <br>
 * - workspace support methods. <p>
 * *
 * Copyright (c) SAP AG 2002-2004
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: IExtendedCollection.java,v 1.3 2004/03/23 13:20:46 jre Exp $
 */
public interface IExtendedCollection {

  /**
   * Creates an external link, compatible to {@link LinkType#EXTERNAL}
   *
   * @param name member name in collection
   * @param isPermanent whether the redirection is temporary (continue using the
   *      RID of the link) or permanent (use of the RID of the link discouraged)
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @param absoluteTargetUri TBD: Description of the incoming method parameter
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createExternalLink(String name, IUri absoluteTargetUri, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException;

  /**
   * Creates an internal link that automatically follows the target resource and
   * disappears when the link target disappears. Note that this type of link can
   * only be created when the link target already exists. <p>
   *
   * <em> Note that a relative RID will be resolved upon creation of the link
   * resource. </em>
   *
   * @param name member name in collection
   * @param targetRid link target
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createInternalDynamicLink(String name, RID targetRid, IPosition position,
    IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException;

  /**
   * Creates an internal link, compatible to {@link LinkType#INTERNAL}
   *
   * @param name member name in collection
   * @param isPermanent whether the redirection is temporary (continue using the
   *      RID of the link) or permanent (use of the RID of the link discouraged)
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @param targetRid TBD: Description of the incoming method parameter
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createInternalStaticLink(String name, RID targetRid, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException;

  /**
   * Creates a new workspace resource.
   *
   * @param name member name in collection
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @return workspace resource
   * @exception ResourceException Exception raised in failure situation
   * @throws NotSupportedException if called on a resource that is not a
   *      workspace collection
   */
  public ICollection createWorkspace(String name, IPosition position)
    throws ResourceException, NotSupportedException;

  /**
   * If called on a workspace controlled collection, a new version controlled
   * resource based on the given revision is created
   *
   * @param baseRevisionRid the rid of the new vcr's base revision
   * @param name member name in collection
   * @param position position where to create the vcr (or <code>null</code> when
   *      unspecified)
   * @return version controlled resource
   * @exception ResourceException Exception raised in failure situation
   * @throws NotSupportedException if called on a resource that is not a sibling
   *      of a workspace resource <br>
   *      - if the resource identified by baseRevisionRid doesn't exist or is no
   *      version resource
   */
  public IResource createVersionControlledResource(
    RID baseRevisionRid, String name, IPosition position)
    throws ResourceException, NotSupportedException;

  /**
   * Returns a filtered list of the resources in this collection.
   *
   * @param visible true: only visible resource (system property "hidden" is
   *      false), false: only hidden resources, null: not relevant
   * @param collections true: only collections, false: only resources which are
   *      not collections, null: note relevant
   * @param links true: only links, false: only non-link resources, null: not
   *      relevant
   * @return A list of resources
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResourceList getChildren(Boolean visible, Boolean collections, Boolean links)
    throws ResourceException, AccessDeniedException;
}
